import { Button, Form, Input, Modal, Select } from 'antd';
import { useContext, useEffect, useState } from 'react';
import { CommonContext } from '~/lib/commonProvider';
import { useGeolocation } from '~/lib/hooks';
import api, { PublishParams, TypeResp } from './api';
import styles from './index.module.scss';
import { phoneNumber } from '~/lib/validateUtils';

type Props = {
  open?: boolean;
  onOk?: () => void;
  onCancel?: () => void;
};
export default function PublishModal(props: Props) {
  const [types, setTypes] = useState<Array<TypeResp>>([]); //需求类型
  const [params, setParams] = useState<PublishParams>({
    publishName: '',
    publishPhone: -1,
    requireDescription: '',
    requirementTypeId: -1,
  });
  const [form] = Form.useForm();
  const position = useGeolocation();
  const { reloadRequirements, setReloadRequirements } = useContext(CommonContext);

  useEffect(() => {
    api.listType().then((res) => {
      setTypes(res.result || []);
    });
  }, []);

  const onFinish = (values: any) => {
    console.log('Success:', values);
    values.publishPhone = Number(values.publishPhone);
    api
      .publish({
        ...params,
        ...values,
        provinceCode: position?.address?.addressComponent?.adcode,
      })
      .then((res) => {
        if (res.code === '200') {
          props.onCancel && props.onCancel();
          window.messageApi.success('发布成功');
          setReloadRequirements(!reloadRequirements);
          setTimeout(() => {
            form.resetFields();
          }, 500);
        }
      });
  };

  const onFinishFailed = (errorInfo: any) => {
    console.log('Failed:', errorInfo);
  };

  return (
    <Modal
      open={props.open}
      onOk={props.onOk}
      onCancel={props.onCancel}
      title='需求发布'
      className={styles.modal}
      width={460}
      footer={null}
    >
      <Form
        labelCol={{ span: 5 }}
        labelAlign='left'
        onFinish={onFinish}
        onFinishFailed={onFinishFailed}
        form={form}
      >
        <Form.Item
          label='姓名'
          name='publishName'
          rules={[{ required: true, message: '请输入姓名！' }]}
        >
          <Input placeholder='输入姓名'></Input>
        </Form.Item>
        <Form.Item
          label='手机号'
          name='publishPhone'
          rules={[
            { required: true, message: '请输入手机号！' },
            {
              pattern: /^1\d{10}$/,
              message: '很输入11位手机号',
            },
          ]}
        >
          <Input onInput={phoneNumber} maxLength={11} allowClear placeholder='输入手机号'></Input>
        </Form.Item>
        <Form.Item
          label='需求类型'
          name='requirementTypeId'
          rules={[{ required: true, message: '请选择需求类型！' }]}
        >
          <Select
            placeholder='选择需求类型'
            options={types}
            fieldNames={{ label: 'typeName', value: 'id' }}
          ></Select>
        </Form.Item>
        <Form.Item
          name='requireDescription'
          rules={[{ required: true, message: '请输入需求描述！' }]}
        >
          <Input.TextArea placeholder='项目需求描述' style={{ height: 162 }}></Input.TextArea>
        </Form.Item>
        <Form.Item>
          <Button
            type='primary'
            htmlType='submit'
            style={{ width: '100%', height: 40, borderRadius: 0 }}
          >
            立即发布
          </Button>
        </Form.Item>
      </Form>
    </Modal>
  );
}
