import { useContext, useEffect, useState } from 'react';

import { Button, Col, message, Modal, Row } from 'antd';

import { UserContext } from '~/lib/userProvider';

import api, { OrderDetailResp, UserWalletResp } from './api';
import styles from './index.module.scss';

function formatNumber(num: number) {
  return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ',');
}

type Props = {
  open?: boolean;
  onOk?: () => void;
  onCancel?: () => void;
  orderNo?: string;
};

export default function PayModal(props: Props) {
  const [data, setData] = useState<OrderDetailResp | null>(null);
  const { userInfo } = useContext(UserContext);
  const [wallet, setWallet] = useState<UserWalletResp | null>(null); // 钱包
  const [loading, setLoading] = useState(false); // 付款按钮loading

  useEffect(() => {
    if (props.open) {
      api.userWallet().then((res) => {
        setWallet(res.result || null);
      });
    }
  }, [props.open]);

  useEffect(() => {
    if (props.orderNo) {
      api.orderDetail({ orderNo: props.orderNo }).then((res) => {
        setData(res.result || null);
      });
    } else {
      setData(null);
    }
  }, [props.orderNo]);

  function onPay() {
    if (props.orderNo) {
      setLoading(true);
      api
        .orderPayment({
          orderNo: props.orderNo,
        })
        .then((res) => {
          if (res.code == '200') {
            message.success('付款成功');
            setTimeout(() => {
              props.onCancel && props.onCancel();
              setLoading(false);
            }, 1000);
          } else {
            res.message && message.error(res.message);
            setLoading(false);
          }
        })
        .catch((err) => {
          setLoading(false);
        });
    }
  }

  return (
    <>
      <Modal
        width={420}
        open={props.open}
        onOk={props.onOk}
        onCancel={props.onCancel}
        title={
          <>
            <div style={{ textAlign: 'center' }} className={styles.font5}>
              租赁付款
            </div>
          </>
        }
        footer={
          <Button
            type='primary'
            className={styles.btn}
            style={{ marginTop: 43 }}
            onClick={onPay}
            loading={loading}
          >
            立即付款
          </Button>
        }
      >
        <div
          style={{ marginTop: 16, marginBottom: 34, textAlign: 'center' }}
          className={styles.font1}
        >
          {formatNumber(data?.shouldPay || 0)}{' '}
          <span
            className={styles.font2}
            style={{ transform: 'translateY(-3px)', display: 'inline-block' }}
          >
            元
          </span>
        </div>
        <Row gutter={[0, 16]}>
          <Col span={6} className={styles.font3}>
            云仓账号
          </Col>
          <Col span={18} className={styles.font4}>
            UID{userInfo?.uid}
          </Col>
          <Col span={6} className={styles.font3}>
            付款方式
          </Col>
          <Col span={18} className={styles.font4}>
            充值余额（可用:¥{wallet?.cashAmt}）
          </Col>
        </Row>
      </Modal>
    </>
  );
}
