package com.mmc.pms.service.impl;
import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.GoodsInfoDao;
import com.mmc.pms.dao.IndustrySpecDao;
import com.mmc.pms.dao.ProductDao;
import com.mmc.pms.entity.*;
import com.mmc.pms.model.order.dto.OrderGoodsIndstDTO;
import com.mmc.pms.model.order.dto.OrderGoodsProdDTO;
import com.mmc.pms.model.order.dto.OrderGoodsProdDetailDTO;
import com.mmc.pms.model.qo.MallOrderGoodsInfoQO;
import com.mmc.pms.model.qo.ProductSpecPriceQO;
import com.mmc.pms.model.sale.dto.*;
import com.mmc.pms.model.sale.qo.MallGoodsQO;
import com.mmc.pms.model.sale.vo.*;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.GoodsInfoService;
import com.mmc.pms.util.CodeUtil;
import com.mmc.pms.util.TDateUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author 23214
 * @description 针对表【goods_info(商品基本信息)】的数据库操作Service实现de
 * @createDate 2023-05-27 14:08:45
 */
@Service
public class GoodsInfoServiceImpl implements GoodsInfoService {

  @Resource private GoodsInfoDao goodsInfoDao;

  @Resource private ProductDao productDao;

  @Resource private ProductServiceImpl productSkuService;

  @Resource private IndustrySpecDao industrySpecDao;

  @Override
  @Transactional(rollbackFor = Exception.class)
  public ResultBody addGoods(GoodsAddVO goodsAddVO, Integer userAccountId) {
    // 判断商品名称是否存在
    if (goodsInfoDao.countGoodsInfoByName(goodsAddVO) > 0) {
      return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
    }
    // 判断商品详情中描述是否为空
    if (goodsAddVO.getGoodsDetailVO().getGoodsDesc() == null) {
      return ResultBody.error(ResultEnum.GOODS_DESC_IS_NOT_NULL);
    }
    String YYYY_MM_DD_HH_MM_SS = "yyyyMMddHHmmss";
    // 初始化商品对象,并构建对象
    GoodsInfo goodsInfo = new GoodsInfo(goodsAddVO);
    goodsInfo.setGoodsNo(
        "IUAV" + TDateUtil.getDateStr(new Date(), YYYY_MM_DD_HH_MM_SS) + CodeUtil.getRandomNum(4));
    // 暂未接入用户默认传1
    goodsInfo.setAddGoodsUserId(1);
    goodsInfo.setSort(goodsInfoDao.countGoodsInfo() + 1);
    // 插入商品基本信息
    goodsInfoDao.insertGoodsInfo(goodsInfo);
    // 调用图片视频插入信息方法
    addGoodsImageInfo(goodsInfo.getId(), goodsAddVO);
    // 调用商品详情信息方法
    addGoodsDetail(goodsInfo.getId(), goodsAddVO.getGoodsDetailVO());
    // 判断其他服务是否为空,不为空则插入
    if (!CollectionUtils.isEmpty(goodsAddVO.getOtherService())) {
      addOtherService(goodsAddVO.getOtherService(), goodsInfo.getId());
    }
    // 调用产品规格的新增方法
    if (!goodsAddVO.getDirectoryId().equals(2)) {
      // 添加产品规格信息
      productSkuSpecOperation(goodsInfo, goodsAddVO.getProductSpec());
    } else {
      industrySkuSpecOperation(goodsInfo, goodsAddVO.getProductSpec());
    }
    return ResultBody.success();
  }

  /** 插入行业sku及规格信息 */
  @Transactional(rollbackFor = Exception.class)
  public void industrySkuSpecOperation(GoodsInfo goodsInfo, List<GoodsProdSpecVO> productSpec) {
    for (GoodsProdSpecVO goodsSpecVO : productSpec) {
      MallIndustrySkuInfoDO mallIndustrySkuInfoDO =
          new MallIndustrySkuInfoDO(goodsSpecVO).setGoodsInfoId(goodsInfo.getId());
      // 插入数据库商品对应的sku信息
      goodsInfoDao.insertMallIndustrySkuInfo(mallIndustrySkuInfoDO);
      for (MallProductSpecVO mallProductSpecVO : goodsSpecVO.getSpecIds()) {
        // 构建商品对应的规格信息
        MallIndustrySkuInfoSpecDO mallIndustrySkuInfoSpecDO =
            new MallIndustrySkuInfoSpecDO()
                .setIndustrySpecId(mallProductSpecVO.getMallSpecId())
                .setMallIndustrySkuInfoId(mallIndustrySkuInfoDO.getId())
                .setGoodsInfoId(goodsInfo.getId());
        // 插入数据库商品对应的规格信息
        goodsInfoDao.insertMallIndustrySkuInfoSpec(mallIndustrySkuInfoSpecDO);
      }
    }
  }

  @Transactional(rollbackFor = Exception.class)
  public void productSkuSpecOperation(GoodsInfo goodsInfo, List<GoodsProdSpecVO> productSpec) {
    // 遍历规格信息，获取其中自定义的规格信息
    List<GoodsProdSpecVO> customGoodsSpecList =
        productSpec.stream().filter(spec -> spec.getFlag().equals(1)).collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(customGoodsSpecList)) {
      addCustomization(goodsInfo, customGoodsSpecList);
    }
    // 遍历规格信息，获取其中非自定义的规格信息
    List<GoodsProdSpecVO> goodsSpecList =
        productSpec.stream().filter(spec -> spec.getFlag().equals(0)).collect(Collectors.toList());
    for (GoodsProdSpecVO goodsSpecVO : goodsSpecList) {
      MallProdInfoDO mallProdSkuInfoDO =
          new MallProdInfoDO(goodsSpecVO)
              .setProductId(goodsSpecVO.getSkuId())
              .setGoodsInfoId(goodsInfo.getId());
      // 插入数据库商品对应的sku信息
      productDao.insertMallProdSkuInfo(mallProdSkuInfoDO);
      for (MallProductSpecVO mallProductSpecVO : goodsSpecVO.getSpecIds()) {
        // 构建商品对应的规格信息
        MallProdSkuInfoSpecDO mallProdSkuInfoSpecDO =
            new MallProdSkuInfoSpecDO()
                .setProductSpecId(mallProductSpecVO.getMallSpecId())
                .setMallProdSkuInfoId(mallProdSkuInfoDO.getId())
                .setGoodsInfoId(goodsInfo.getId());
        // 插入数据库商品对应的规格信息
        goodsInfoDao.insertMallProdSkuInfoSpec(mallProdSkuInfoSpecDO);
      }
    }
  }

  @Transactional(rollbackFor = Exception.class)
  public MallProdInfoDO buildMallProdSkuInfo(
      GoodsInfo goodsInfo, GoodsProdSpecVO goodsSpec, Integer id) {
    MallProdInfoDO mallProdInfoDO = new MallProdInfoDO();
    mallProdInfoDO.setGoodsInfoId(goodsInfo.getId());
    mallProdInfoDO.setProductId(id);
    mallProdInfoDO.setProdSkuSpecName(goodsSpec.getGoodsSpecName());
    mallProdInfoDO.setCategoriesId(goodsSpec.getCategoryId());
    mallProdInfoDO.setChooseType(goodsSpec.getChooseType());
    mallProdInfoDO.setMust(goodsSpec.getMust());
    mallProdInfoDO.setFlag(goodsSpec.getFlag());
    mallProdInfoDO.setSkuUnitId(goodsSpec.getSkuUnitId());
    return mallProdInfoDO;
  }

  @Transactional(rollbackFor = Exception.class)
  public void addCustomization(GoodsInfo goodsInfo, List<GoodsProdSpecVO> customGoodsSpecList) {
    // 遍历自定义规格信息
    for (GoodsProdSpecVO productSpec : customGoodsSpecList) {
      // 构建商品对应的sku信息
      ProductDO productSkuDO =
          new ProductDO().setProductName(productSpec.getProductName()).setCustomize(1);
      // 插入产品sku信息
      productDao.insertProductSku(productSkuDO);
      MallProdInfoDO mallProdInfoDO =
          buildMallProdSkuInfo(goodsInfo, productSpec, productSkuDO.getId());
      // 插入数据库商品对应的sku信息
      productDao.insertMallProdSkuInfo(mallProdInfoDO);
      // 先将自定的信息存储到数据库中
      productSpec.getCustomizeInfo().stream()
          .peek(
              param -> {
                ProductSpecDO productSpecDO =
                    new ProductSpecDO()
                        .setProductId(productSkuDO.getId())
                        .setVersionDesc("自定义")
                        .setSpecImage(param.getSpecImage())
                        .setSpecName(param.getSpecName())
                        .setPartNo(param.getPartNo());
                // 新增产品sku
                productDao.insertProductSpec(productSpecDO);
                // 批量配置价格信息
                ProductSpecCPQVO productSpecCPQVO = param.getProductSpecCPQVO();
                if (productSpecCPQVO != null) {
                  productSpecCPQVO.setProductSpecId(productSpecDO.getId());
                  productSkuService.insertSpecPrice(productSpecCPQVO);
                }
                // 构建商品对应的规格信息
                MallProdSkuInfoSpecDO mallProdSkuInfoSpecDO =
                    new MallProdSkuInfoSpecDO()
                        .setProductSpecId(productSpecDO.getId())
                        .setMallProdSkuInfoId(mallProdInfoDO.getId())
                        .setGoodsInfoId(goodsInfo.getId());
                // 插入数据库商品对应的规格信息
                goodsInfoDao.insertMallProdSkuInfoSpec(mallProdSkuInfoSpecDO);
              })
          .collect(Collectors.toList());
    }
  }

  @Transactional(rollbackFor = Exception.class)
  public void addOtherService(List<Integer> otherService, Integer goodsId) {
    // 插入其他服务
    List<GoodsServiceDO> otherList =
        otherService.stream()
            .map(
                d -> {
                  GoodsServiceDO goodsServiceDO = new GoodsServiceDO();
                  goodsServiceDO.setGoodsInfoId(goodsId);
                  goodsServiceDO.setSaleServiceId(d);
                  return goodsServiceDO;
                })
            .collect(Collectors.toList());
    goodsInfoDao.insertGoodsService(otherList);
  }

  @Transactional(rollbackFor = Exception.class)
  public void addGoodsDetail(Integer goodsId, GoodsDetailVO goodsDetailVO) {
    GoodsDetailDO goodsDetailDO = new GoodsDetailDO(goodsDetailVO);
    goodsDetailDO.setGoodsInfoId(goodsId);
    goodsInfoDao.insertGoodsDetail(goodsDetailDO);
  }

  @Transactional(rollbackFor = Exception.class)
  public void addGoodsImageInfo(Integer goodsId, GoodsAddVO goodsAddVO) {
    List<GoodsImgDO> list =
        goodsAddVO.getImages().stream()
            .map(
                d -> {
                  GoodsImgDO goodsImgDO = new GoodsImgDO(d);
                  goodsImgDO.setGoodsInfoId(goodsId);
                  return goodsImgDO;
                })
            .collect(Collectors.toList());
    // 插入图片信息
    goodsInfoDao.insertGoodsImgInfo(list);
    // 插入商品视频信息
    if (goodsAddVO.getGoodsVideo() != null) {
      GoodsVideoDO goodsVideoDO =
          new GoodsVideoDO().setGoodsInfoId(goodsId).setVideoUrl(goodsAddVO.getGoodsVideo());
      goodsInfoDao.insertVideoInfo(goodsVideoDO);
    }
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public ResultBody editGoodsInfo(GoodsAddVO goodsAddVO) {
    // 判断商品是否还存在
    int count = goodsInfoDao.countGoodsInfoById(goodsAddVO.getId());
    if (count <= 0) {
      return ResultBody.error(ResultEnum.GOODS_NOT_EXIST_OR_ALREADY_DOWN_SHELF);
    }
    // 判断商品名称是否存在
    if (goodsInfoDao.countGoodsInfoByName(goodsAddVO) > 0) {
      return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
    }
    // 判断商品详情中描述是否为空
    if (goodsAddVO.getGoodsDetailVO().getGoodsDesc() == null) {
      return ResultBody.error(ResultEnum.GOODS_DESC_IS_NOT_NULL);
    }
    // 初始化商品对象,并构建对象
    GoodsInfo goodsInfo = new GoodsInfo(goodsAddVO);
    // 插入商品基本信息
    goodsInfoDao.updateGoodsInfo(goodsInfo);
    // 修改商品详情信息
    updateGoodsDetail(goodsAddVO);
    // 修改商品图片及视频信息
    updateImageInfo(goodsAddVO);
    // 修改其他服务信息
    updateOtherService(goodsAddVO);
    // 修改规格信息
    if (!goodsAddVO.getDirectoryId().equals(2)) {
      updateMallProductSkuSpec(goodsAddVO);
    } else {
      updateMallIndustrySkuSpec(goodsAddVO);
    }
    return ResultBody.success();
  }

  @Transactional(rollbackFor = Exception.class)
  public void updateMallIndustrySkuSpec(GoodsAddVO goodsAddVO) {
    // 先获取用户输入的商品绑定sku的id集合
    Set<Integer> inputMallSkuIds =
        goodsAddVO.getProductSpec().stream()
            .map(GoodsProdSpecVO::getId)
            .filter(Objects::nonNull)
            .collect(Collectors.toSet());
    // 从数据库获取该商品对应绑定的skuId的集合
    List<MallIndustrySkuInfoDO> mallIndusSkuInfoList =
        goodsInfoDao.getMallIndustrySkuInfo(goodsAddVO.getId());
    List<Integer> dbMallSkuIds =
        mallIndusSkuInfoList.stream()
            .map(MallIndustrySkuInfoDO::getId)
            .collect(Collectors.toList());
    // 对比数据库id如果不在就删除
    List<Integer> delIds =
        dbMallSkuIds.stream()
            .filter(id -> !inputMallSkuIds.contains(id))
            .collect(Collectors.toList());
    if (delIds.size() != 0) {
      // 先删除商品绑定sku下的spec信息
      industrySpecDao.batchUpdateMallIndustrySpec(delIds, goodsAddVO.getId());
      industrySpecDao.batchUpdateMallIndustrySku(delIds);
    }
    // 修改商品绑定的sku信息
    List<GoodsProdSpecVO> updateGoodsSpec =
        goodsAddVO.getProductSpec().stream().filter(Objects::nonNull).collect(Collectors.toList());
    List<MallIndustrySkuInfoDO> mallIndustrySkuInfoList =
        updateGoodsSpec.stream()
            .map(
                d -> {
                  MallIndustrySkuInfoDO mallIndustrySkuInfoDO = new MallIndustrySkuInfoDO();
                  mallIndustrySkuInfoDO
                      .setId(d.getId())
                      .setIndustrySkuId(d.getSkuId())
                      .setCategoriesId(d.getCategoryId())
                      .setIndustrySkuSpecName(d.getGoodsSpecName())
                      .setChooseType(d.getChooseType())
                      .setSkuUnitId(d.getSkuUnitId())
                      .setMust(d.getMust());
                  return mallIndustrySkuInfoDO;
                })
            .collect(Collectors.toList());
    // 批量修改sku的信息
    industrySpecDao.batchUpdateMallIndustrySkuInfo(mallIndustrySkuInfoList);
    // 获取新的sku信息
    List<GoodsProdSpecVO> list =
        goodsAddVO.getProductSpec().stream()
            .filter(d -> d.getId() == null)
            .collect(Collectors.toList());
    industrySkuSpecOperation(new GoodsInfo(goodsAddVO.getId()), list);
    // 获取商品对应的skuId
    List<Integer> mallSkuIds =
        goodsAddVO.getProductSpec().stream()
            .map(GoodsProdSpecVO::getId)
            .filter(Objects::nonNull)
            .collect(Collectors.toList());
    // 根据这些id查出下面对应的规格信息
    List<MallIndustrySkuInfoSpecDO> mallIndustrySkuInfoSpecList =
        industrySpecDao.listMallIndustrySpecInfo(mallSkuIds);
    // 分组
    Map<Integer, List<MallIndustrySkuInfoSpecDO>> dbSpecMap =
        mallIndustrySkuInfoSpecList.stream()
            .collect(Collectors.groupingBy(MallIndustrySkuInfoSpecDO::getMallIndustrySkuInfoId));
    List<GoodsProdSpecVO> goodsSpecList =
        goodsAddVO.getProductSpec().stream()
            .filter(d -> d.getId() != null)
            .collect(Collectors.toList());
    for (GoodsProdSpecVO goodsSpecVO : goodsSpecList) {
      // 分组后的每一条商品绑定的规格信息
      List<MallIndustrySkuInfoSpecDO> dbMallIndustrySkuSpecList =
          dbSpecMap.get(goodsSpecVO.getId());
      Set<Integer> inputMallSpecIds =
          goodsSpecVO.getSpecIds().stream()
              .map(MallProductSpecVO::getId)
              .filter(Objects::nonNull)
              .collect(Collectors.toSet());
      List<Integer> delSpecId =
          dbMallIndustrySkuSpecList.stream()
              .map(MallIndustrySkuInfoSpecDO::getId)
              .filter(id -> !inputMallSpecIds.contains(id))
              .collect(Collectors.toList());
      if (delSpecId.size() != 0) {
        // 删除多余的spec信息
        industrySpecDao.batchUpdateMallIndustSpec(delSpecId, goodsAddVO.getId());
      }
      // 新增spec的信息
      for (MallProductSpecVO mallProductSpecVO : goodsSpecVO.getSpecIds()) {
        if (mallProductSpecVO.getId() == null) {
          MallIndustrySkuInfoSpecDO mallIndustrySkuInfoSpecDO =
              new MallIndustrySkuInfoSpecDO()
                  .setIndustrySpecId(mallProductSpecVO.getMallSpecId())
                  .setMallIndustrySkuInfoId(goodsSpecVO.getId())
                  .setGoodsInfoId(goodsAddVO.getId());
          // 插入数据库商品对应的规格信息
          goodsInfoDao.insertMallIndustrySkuInfoSpec(mallIndustrySkuInfoSpecDO);
        }
      }
    }
  }

  @Transactional(rollbackFor = Exception.class)
  public void updateMallProductSkuSpec(GoodsAddVO goodsAddVO) {
    GoodsInfo goodsInfo = new GoodsInfo(goodsAddVO);
    // 1.1先获取用户输入的商品绑定sku的id集合
    List<GoodsProdSpecVO> productSpec = goodsAddVO.getProductSpec();
    // 获取不为空的产品规格
    List<GoodsProdSpecVO> prod = productSpec.stream().filter(d -> d.getId() != null).collect(Collectors.toList());
    List<Integer> inputMallSkuIds =
            prod.stream().map(GoodsProdSpecVO::getId).collect(Collectors.toList());
    // 1.2从数据库获取该商品对应绑定的skuId的集合
    List<MallProdInfoDO> mallProSkuInfoList = goodsInfoDao.getMallProSkuInfo(goodsAddVO.getId());
    List<Integer> dbMallSkuIds =
            mallProSkuInfoList.stream().map(MallProdInfoDO::getId).collect(Collectors.toList());
    // 对比数据库id如果不在就删除
    List<Integer> delIds =
            dbMallSkuIds.stream()
                    .filter(id -> !inputMallSkuIds.contains(id))
                    .collect(Collectors.toList());
    if (delIds.size() != 0) {
      // 先删除商品绑定sku下的spec信息
      goodsInfoDao.batchUpdateMallProSpec(delIds, goodsAddVO.getId());
      goodsInfoDao.batchUpdateMallProductSku(delIds);
    }
    // 1、普通的sku修改逻辑
    List<GoodsProdSpecVO> specInfo =
            goodsAddVO.getProductSpec().stream()
                    .filter(spec -> !spec.getFlag().equals(1))
                    .collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(specInfo)) {
      // 1.1先获取用户输入的商品绑定sku的id集合
      List<GoodsProdSpecVO> updateGoodsSpec =
              specInfo.stream().filter(d -> d.getId() != null).collect(Collectors.toList());
      // 修改商品绑定的sku信息
      List<MallProdInfoDO> mallProdSkuInfoList =
          updateGoodsSpec.stream()
              .map(
                  d -> {
                    MallProdInfoDO mallProdInfoDO = new MallProdInfoDO();
                    mallProdInfoDO
                        .setId(d.getId())
                        .setProductId(d.getSkuId())
                        .setCategoriesId(d.getCategoryId())
                        .setProdSkuSpecName(d.getGoodsSpecName())
                        .setChooseType(d.getChooseType())
                        .setSkuUnitId(d.getSkuUnitId())
                        .setMust(d.getMust());
                    return mallProdInfoDO;
                  })
              .collect(Collectors.toList());
      // 1.3批量修改sku的信息
      for (MallProdInfoDO mallProdInfoDO : mallProdSkuInfoList) {
        goodsInfoDao.updateMallProdSkuInfo(mallProdInfoDO);
      }
      //      goodsInfoDao.batchUpdateMallProdSkuInfo(mallProdSkuInfoList);
      // 1.4对新的sku进行操作
      List<GoodsProdSpecVO> list =
          specInfo.stream().filter(d -> d.getId() == null).collect(Collectors.toList());
      productSkuSpecOperation(goodsInfo, list);
      // 获取商品对应的skuId
      List<Integer> mallSkuIds =
          goodsAddVO.getProductSpec().stream()
              .filter(d -> d.getId() != null)
              .map(GoodsProdSpecVO::getId)
              .collect(Collectors.toList());
      if (mallSkuIds.size() > 0) {
        // 根据这些id查出下面对应的规格信息
        List<MallProdSkuInfoSpecDO> mallProdSkuInfoSpecList =
            goodsInfoDao.listMallProdSpecInfo(mallSkuIds);
        // 分组
        Map<Integer, List<MallProdSkuInfoSpecDO>> dbSpecMap =
            mallProdSkuInfoSpecList.stream()
                .collect(Collectors.groupingBy(MallProdSkuInfoSpecDO::getMallProdSkuInfoId));
        for (GoodsProdSpecVO goodsSpecVO : updateGoodsSpec) {
          // 分组后的每一条商品绑定的规格信息
          List<MallProdSkuInfoSpecDO> dbMallProdSkuSpecList = dbSpecMap.get(goodsSpecVO.getId());
          Set<Integer> inputMallSpecIds =
              goodsSpecVO.getSpecIds().stream()
                  .map(MallProductSpecVO::getId)
                  .filter(Objects::nonNull)
                  .collect(Collectors.toSet());
          List<Integer> delSpecId =
              dbMallProdSkuSpecList.stream()
                  .map(MallProdSkuInfoSpecDO::getId)
                  .filter(id -> !inputMallSpecIds.contains(id))
                  .collect(Collectors.toList());
          if (delSpecId.size() != 0) {
            // 删除多余的spec信息
            goodsInfoDao.batchUpdateMallProdSpec(delSpecId);
          }
          // 新增spec的信息
          for (MallProductSpecVO mallProductSpecVO : goodsSpecVO.getSpecIds()) {
            if (mallProductSpecVO.getId() == null) {
              MallProdSkuInfoSpecDO mallProdSkuInfoSpecDO =
                  new MallProdSkuInfoSpecDO()
                      .setProductSpecId(mallProductSpecVO.getMallSpecId())
                      .setMallProdSkuInfoId(goodsSpecVO.getId())
                      .setGoodsInfoId(goodsAddVO.getId());
              // 插入数据库商品对应的规格信息
              goodsInfoDao.insertMallProdSkuInfoSpec(mallProdSkuInfoSpecDO);
            }
          }
        }
      }
    }
    // 2、对自定义的sku进行操作
    List<GoodsProdSpecVO> zdySpecInfo =
        goodsAddVO.getProductSpec().stream()
            .filter(spec -> spec.getFlag().equals(1))
            .collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(zdySpecInfo)) {
      updateCustomize(goodsInfo, zdySpecInfo, goodsAddVO.getId());
    }
  }

  @Transactional(rollbackFor = Exception.class)
  public void updateCustomize(GoodsInfo goodsInfo, List<GoodsProdSpecVO> zdySpecInfo, Integer id) {
    // 获取自定义产品及规格信息（获取带id的）
    List<GoodsProdSpecVO> zdySpec =
            zdySpecInfo.stream().filter(d -> d.getId() != null).collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(zdySpec)) {
      for (GoodsProdSpecVO d : zdySpec) {
        // 修改产品的规格基本信息
        MallProdInfoDO mallProdInfoDO = new MallProdInfoDO(d).setId(d.getId()).setProductId(d.getSkuId());
        goodsInfoDao.updateMallProdSkuInfo(mallProdInfoDO);
        // 构建商品对应的产品信息
        ProductDO productSkuDO =
                new ProductDO().setId(d.getSkuId()).setProductName(d.getProductName());
        productDao.updateProductSku(productSkuDO);
        List<ProductSpecVO> customizeInfoList =
                d.getCustomizeInfo().stream()
                        .filter(e -> e.getId() != null)
                        .collect(Collectors.toList());
        // 【2】删除多余的自定义规格
        if (!CollectionUtils.isEmpty(d.getDelProductSpecId())) {
          // 批量查出spec的数据
          List<MallProdSkuInfoSpecDO> prodSkuInfoSpecList =
                  productDao.getProductSpecByIds(d.getDelProductSpecId());
          for (MallProdSkuInfoSpecDO mallProdSkuInfoSpecDO : prodSkuInfoSpecList) {
            productDao.removeProductSpec(mallProdSkuInfoSpecDO.getProductSpecId());
          }
          goodsInfoDao.batchUpdateMallProdSpec(d.getDelProductSpecId());
        }
        for (ProductSpecVO e : customizeInfoList) {
          // ①修改价格配置
          ProductSpecCPQVO productSpecCPQVO = e.getProductSpecCPQVO();
          // 删除所有的随后在新增
          productDao.removeProductSpecCPQ(productSpecCPQVO);
          productSkuService.insertSpecPrice(productSpecCPQVO);
          ProductSpecDO productSpecDO = new ProductSpecDO(e);
          productSpecDO.setId(productSpecCPQVO.getProductSpecId());
          // ②在修改自定义的数据
          productDao.updateProductSpec(productSpecDO);
        }

        // 【3】新增最新的自定义规格
        List<ProductSpecVO> collect =
                d.getCustomizeInfo().stream()
                        .filter(e -> e.getId() == null)
                        .collect(Collectors.toList());
        for (ProductSpecVO param : collect) {
          ProductSpecDO productSpecDO = new ProductSpecDO(param);
          // 新增产品sku
          productDao.insertProductSpec(productSpecDO);
          // 批量配置价格信息
          ProductSpecCPQVO productSpecCPQVO = param.getProductSpecCPQVO();
          productSpecCPQVO.setProductSpecId(productSpecDO.getId());
          productSkuService.insertSpecPrice(productSpecCPQVO);
          // 构建商品对应的规格信息
          MallProdSkuInfoSpecDO mallProdSkuInfoSpecDO =
                  new MallProdSkuInfoSpecDO()
                          .setProductSpecId(productSpecDO.getId())
                          .setMallProdSkuInfoId(d.getId())
                          .setGoodsInfoId(id);
          // 插入数据库商品对应的规格信息
          goodsInfoDao.insertMallProdSkuInfoSpec(mallProdSkuInfoSpecDO);
        }
      }
    }
    // 2.2 新增新的自定义规格信息
    List<GoodsProdSpecVO> newZdySpec =
            zdySpecInfo.stream().filter(d -> d.getId() == null).collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(newZdySpec)) {
      productSkuSpecOperation(goodsInfo, newZdySpec);
    }
  }

  @Transactional(rollbackFor = Exception.class)
  public void updateOtherService(GoodsAddVO goodsAddVO) {
    // 删除其他服务
    goodsInfoDao.deleteGoodsServiceByGoodsId(goodsAddVO.getId());
    // 插入其他服务
    if (!CollectionUtils.isEmpty(goodsAddVO.getOtherService())) {
      List<Integer> other = goodsAddVO.getOtherService();
      List<GoodsServiceDO> otherList =
          other.stream()
              .map(
                  d -> {
                    GoodsServiceDO goodsServiceDO = new GoodsServiceDO();
                    goodsServiceDO.setGoodsInfoId(goodsAddVO.getId());
                    goodsServiceDO.setSaleServiceId(d);
                    return goodsServiceDO;
                  })
              .collect(Collectors.toList());
      goodsInfoDao.insertGoodsService(otherList);
    }
  }

  @Transactional(rollbackFor = Exception.class)
  public void updateImageInfo(GoodsAddVO goodsAddVO) {
    Set<Integer> imgIds =
        goodsAddVO.getImages().stream()
            .map(GoodsImgVO::getId)
            .filter(Objects::nonNull)
            .collect(Collectors.toSet());
    List<GoodsImgDO> dbImgList = goodsInfoDao.listGoodsInfoByGoodsId(goodsAddVO.getId());
    List<Integer> deleteIds =
        dbImgList.stream()
            .map(GoodsImgDO::getId)
            .filter(id -> !imgIds.contains(id))
            .collect(Collectors.toList());
    if (deleteIds.size() != 0) {
      // 删除多余的图片
      goodsInfoDao.deleteImgByIds(deleteIds);
    }
    // 新增图片
    List<GoodsImgDO> imgDOList =
        goodsAddVO.getImages().stream()
            .filter(d -> d.getId() == null)
            .map(
                d -> {
                  GoodsImgDO goodsImgDO = new GoodsImgDO(d);
                  goodsImgDO.setGoodsInfoId(goodsAddVO.getId());
                  return goodsImgDO;
                })
            .collect(Collectors.toList());
    if (imgDOList.size() != 0) {
      goodsInfoDao.insertGoodsImgInfo(imgDOList);
    }
    // 删除视频
    goodsInfoDao.deleteGoodsVideoById(goodsAddVO.getId());
    // 插入视频
    if (goodsAddVO.getGoodsVideo() != null) {
      GoodsVideoDO goodsVideoDO = new GoodsVideoDO();
      goodsVideoDO.setGoodsInfoId(goodsAddVO.getId());
      goodsVideoDO.setVideoUrl(goodsAddVO.getGoodsVideo());
      goodsInfoDao.insertVideoInfo(goodsVideoDO);
    }
  }

  @Transactional(rollbackFor = Exception.class)
  public void updateGoodsDetail(GoodsAddVO goodsAddVO) {
    GoodsDetailDO goodsDetailDO =
        new GoodsDetailDO(goodsAddVO.getGoodsDetailVO()).setGoodsInfoId(goodsAddVO.getId());
    // 商品详情修改
    goodsInfoDao.updateGoodsDetail(goodsDetailDO);
  }

  @Override
  public ResultBody getGoodsInfoDetail(Integer goodsInfoId, Integer type, Integer leaseTerm) {
    // 判断此商品是否还存在
    int count = goodsInfoDao.countGoodsInfoById(goodsInfoId);
    if (count <= 0) {
      return ResultBody.error(ResultEnum.GOODS_NOT_EXIST_OR_ALREADY_DOWN_SHELF);
    }
    // 初始化商品返回信息的对象
    MallGoodsDetailDTO mallGoodsDetail = new MallGoodsDetailDTO();
    // 获取商品基本信息
    GoodsInfo goodsInfo = goodsInfoDao.getGoodsSimpleInfo(goodsInfoId);
    mallGoodsDetail
        .setId(goodsInfo.getId())
        .setGoodsNo(goodsInfo.getGoodsNo())
        .setGoodsName(goodsInfo.getGoodsName())
        .setCategoryByOne(goodsInfo.getCategoryByOne())
        .setGoodsVideo(goodsInfo.getVideoUrl())
        .setDirectoryId(goodsInfo.getDirectoryId())
        .setCategoryByTwo(goodsInfo.getCategoryByTwo())
        .setTag(goodsInfo.getEcoLabel())
        .setShelfStatus(goodsInfo.getShelfStatus())
        .setGoodsVideoId(goodsInfo.getGoodsVideoId());
    // 获取商品图片信息
    mallGoodsDetail.setImages(getGoodsImageInfo(goodsInfoId));
    // 获取商品详细信息
    mallGoodsDetail.setGoodsDetail(getGoodsDetail(goodsInfoId));
    // 获取其他服务信息
    mallGoodsDetail.setOtherService(getOtherServiceInfo(goodsInfoId));
    // 获取规格信息
    if (!goodsInfo.getDirectoryId().equals(2)) {
      // 获取产品规格信息
      mallGoodsDetail.setGoodsSpec(getProductSpecInfo(goodsInfoId, type, leaseTerm));
    } else {
      // 获取行业规格信息
      mallGoodsDetail.setGoodsSpec(getIndustrySpecInfo(goodsInfoId));
    }
    return ResultBody.success(mallGoodsDetail);
  }

  private List<GoodsSpecDTO> getIndustrySpecInfo(Integer goodsInfoId) {
    // 获取商品对应绑定的行业sku信息
    List<MallIndustrySkuInfoDO> mallIndustrySkuInfoList =
        goodsInfoDao.getMallIndustrySkuInfo(goodsInfoId);
    List<GoodsSpecDTO> list =
        mallIndustrySkuInfoList.stream()
            .map(MallIndustrySkuInfoDO::buildGoodsSpecDTO)
            .collect(Collectors.toList());
    // 根据商品id查出该商品下绑定的规格信息
    List<MallIndustrySkuInfoSpecDO> mallIndustrySkuInfoSpec =
        industrySpecDao.getIndustrySkuInfoSpec(goodsInfoId);
    list =
        list.stream()
            .peek(
                d -> {
                  List<MallIndustrySpecDTO> industrySpec = new ArrayList<>();
                  for (MallIndustrySkuInfoSpecDO e : mallIndustrySkuInfoSpec) {
                    if (d.getId().equals(e.getMallIndustrySkuInfoId())) {
                      IndustrySpecDO industrySpecDO = e.getIndustrySpecDO();
                      MallIndustrySpecDTO industrySpecDTO =
                          industrySpecDO.buildMallIndustrySpecDTO();
                      industrySpecDTO.setId(e.getId());
                      industrySpecDTO.setIndustrySpecId(e.getIndustrySpecId());
                      industrySpecDTO.setIndustrySkuId(d.getSkuId());
                      industrySpec.add(industrySpecDTO);
                    }
                  }
                  d.setIndustrySpecList(industrySpec);
                })
            .collect(Collectors.toList());
    return list;
  }

  private List<GoodsSpecDTO> getProductSpecInfo(
      Integer goodsInfoId, Integer type, Integer leaseTerm) {
    // 获取商品对应绑定sku的信息
    List<MallProdInfoDO> mallProdSkuInfoList = goodsInfoDao.getAllMallProSkuInfo(goodsInfoId);
    List<GoodsSpecDTO> list =
        mallProdSkuInfoList.stream()
            .map(MallProdInfoDO::buildGoodsSpecDTO)
            .collect(Collectors.toList());
    // 获取规格来源详细信息
    List<MallProdSkuInfoSpecDO> mallProdSkuInfoSpec =
        goodsInfoDao.getMallProSkuInfoSpec(goodsInfoId);
    // 根据id查询出规格的具体信息
    list =
        list.stream()
            .peek(
                d -> {
                  List<MallProductSpecDTO> specList = new ArrayList<>();
                  for (MallProdSkuInfoSpecDO e : mallProdSkuInfoSpec) {
                    if (d.getId().equals(e.getMallProdSkuInfoId())) {
                      // 构建规格对象
                      ProductSpecDO productSpecDO = e.getProductSpecDO();
                      MallProductSpecDTO productSpecDTO = productSpecDO.buildMallProductSpecDTO();
                      productSpecDTO.setProductSkuId(d.getSkuId());
                      productSpecDTO.setId(e.getId());
                      productSpecDTO.setProductSpec(e.getProductSpecId());
                      // 获取自定义sku下规格的价格配置信息
                      ProductSpecCPQVO productSpecCPQVO = new ProductSpecCPQVO();
                      productSpecCPQVO.setProductSpecId(e.getProductSpecId());
                      productSpecCPQVO.setType(type);
                      if (type.equals(1)) {
                        productSpecCPQVO.setLeaseTerm(leaseTerm);
                      }
                      List<ProductSpecPriceDO> productSpecPrice =
                          productDao.getProductSpecPrice(productSpecCPQVO);
                      List<SpecPriceVO> collect =
                          productSpecPrice.stream()
                              .map(
                                  m -> {
                                    SpecPriceVO specPriceVO = new SpecPriceVO();
                                    specPriceVO.setId(m.getId());
                                    specPriceVO.setPrice(m.getPrice());
                                    specPriceVO.setCooperationTag(m.getCooperationTag());
                                    return specPriceVO;
                                  })
                              .collect(Collectors.toList());
                      productSpecCPQVO.setSpecPrice(collect);
                      productSpecDTO.setProductSpecCPQVO(productSpecCPQVO);

                      specList.add(productSpecDTO);
                    }
                  }
                  d.setProductSpecList(specList);
                })
            .collect(Collectors.toList());
    return list;
  }

  public List<GoodsOtherServiceDTO> getOtherServiceInfo(Integer goodsInfoId) {
    List<GoodsServiceDO> goodsServiceDO = goodsInfoDao.listGoodsServiceByGoodsId(goodsInfoId);
    return goodsServiceDO.stream()
        .map(GoodsServiceDO::buildGoodsOtherServiceDTO)
        .collect(Collectors.toList());
  }

  public GoodsDetailInfoDTO getGoodsDetail(Integer goodsInfoId) {
    GoodsDetailDO goodsDetail = goodsInfoDao.getGoodsDetailByGoodsId(goodsInfoId);
    return goodsDetail.buildGoodsDetailInfoDTO();
  }

  public List<GoodsImgDTO> getGoodsImageInfo(Integer goodsInfoId) {
    List<GoodsImgDO> goodsImgList = goodsInfoDao.listGoodsInfoByGoodsId(goodsInfoId);
    return goodsImgList.stream().map(GoodsImgDO::buildGoodsImgDTO).collect(Collectors.toList());
  }

  @Override
  public ResultBody getSaleServiceInfoToList() {
    List<SaleServiceDO> list = goodsInfoDao.listSaleServiceInfo();
    List<SaleServiceDTO> serviceDTOList =
        list.stream().map(d -> d.buildSaleServiceDTO()).collect(Collectors.toList());
    return ResultBody.success(serviceDTOList);
  }

  @Override
  public ResultBody getSkuUnit() {
    List<SkuUnitDO> skuUnitList = goodsInfoDao.getSkuUnit();
    List<SkuUnitDTO> list =
        skuUnitList.stream().map(SkuUnitDO::buildSkuUnitDTO).collect(Collectors.toList());
    return ResultBody.success(list);
  }

  /**
   * 根据商品id，商品规格id查询并填充相关信息
   *
   * @param param
   * @return
   */
  @Override
  public List<MallGoodsShopCarDTO> fillGoodsInfo(List<MallGoodsShopCarDTO> param) {
    Set<Integer> goodsIds = new HashSet<>();
    // 获取商品id
    for (MallGoodsShopCarDTO mallGoodsShopCarDTO : param) {
      goodsIds.add(mallGoodsShopCarDTO.getGoodsInfoId());
    }
    // 查询出商品信息后进行填充
    List<GoodsInfo> goodsInfoDOList = goodsInfoDao.listSimpleGoodsInfoByIds(goodsIds);
    for (MallGoodsShopCarDTO mallGoodsShopCarDTO : param) {
      for (GoodsInfo goodsInfo : goodsInfoDOList) {
        if (mallGoodsShopCarDTO.getGoodsInfoId().toString().equals(goodsInfo.getId().toString())) {
          mallGoodsShopCarDTO.setDirectoryId(goodsInfo.getDirectoryId());
          mallGoodsShopCarDTO.setShelfStatus(goodsInfo.getShelfStatus());
          mallGoodsShopCarDTO.setGoodsName(goodsInfo.getGoodsName());
          mallGoodsShopCarDTO.setMainImg(goodsInfo.getMainImg());
          break;
        }
      }
    }

    // 对产品和行业的规格id进行分类
    Set<Integer> prodIds = new HashSet<>();
    Set<Integer> indstIds = new HashSet<>();
    for (MallGoodsShopCarDTO mallGoodsShopCarDTO : param) {
      if (mallGoodsShopCarDTO.getDirectoryId().equals(1)) {
        for (MallSkuInfoSpecDTO mallSkuInfoSpecDTO : mallGoodsShopCarDTO.getSkuList()) {
          prodIds.add(mallSkuInfoSpecDTO.getMallSkuInfoSpecId());
        }
      } else if (mallGoodsShopCarDTO.getDirectoryId().equals(2)) {
        for (MallSkuInfoSpecDTO mallSkuInfoSpecDTO : mallGoodsShopCarDTO.getSkuList()) {
          indstIds.add(mallSkuInfoSpecDTO.getMallSkuInfoSpecId());
        }
      }
    }
    // 根据规格id查找规格信息
    List<MallGoodsSpecInfoDO> goodsSpecInfoDOList = new ArrayList<>();
    if (!CollectionUtils.isEmpty(prodIds)) {
      List<MallGoodsSpecInfoDO> goodsProdSpecInfoDOList = goodsInfoDao.listProdSpecInfo(prodIds);
      goodsSpecInfoDOList.addAll(goodsProdSpecInfoDOList);
    }
    if (!CollectionUtils.isEmpty(indstIds)) {
      List<MallGoodsSpecInfoDO> goodsIndstSpecInfoDOList = goodsInfoDao.listIndstSpecInfo(indstIds);
      goodsSpecInfoDOList.addAll(goodsIndstSpecInfoDOList);
    }
    // 根据查出来的sku，填充到MallGoodsShopCarDTO里面
    for (MallGoodsShopCarDTO mallGoodsShopCarDTO : param) {
      for (MallGoodsSpecInfoDO mallGoodsSpecInfoDO : goodsSpecInfoDOList) {
        // 找到对应商品
        if (mallGoodsShopCarDTO
                .getGoodsInfoId()
                .toString()
                .equals(mallGoodsSpecInfoDO.getId().toString())
            && mallGoodsShopCarDTO
                .getDirectoryId()
                .toString()
                .equals(mallGoodsSpecInfoDO.getDirectoryId().toString())) {
          for (SkuSpecDO skuSpecDO : mallGoodsSpecInfoDO.getSkuSpecDOList()) {
            for (MallSkuInfoSpecDTO mallSkuInfoSpecDTO : mallGoodsShopCarDTO.getSkuList()) {
              // 根据对应规格进行填充
              if (skuSpecDO
                  .getId()
                  .toString()
                  .equals(mallSkuInfoSpecDTO.getMallSkuInfoSpecId().toString())) {
                mallSkuInfoSpecDTO.setSpecName(skuSpecDO.getSpecName());
                mallSkuInfoSpecDTO.setDeleted(skuSpecDO.getSkuSpecDeleted());
                mallSkuInfoSpecDTO.setValid(
                    skuSpecDO.getSkuSpecDeleted() == 0 && skuSpecDO.getSpecDeleted() == 0);
                break;
              }
            }
          }
          break;
        }
      }
    }
    return param;
  }

  @Override
  public List<MallProductSpecPriceDTO> feignListProductSpecPrice(
      ProductSpecPriceQO productSpecPriceQO) {
    Set<Integer> ids = new HashSet<>(productSpecPriceQO.getProductSpecIds());
    List<MallProductSpecPriceDTO> mallProductSpecPriceDTOS =
        productSpecPriceQO.getProductSpecIds().stream()
            .map(
                d -> {
                  MallProductSpecPriceDTO mallProductSpecPriceDTO = new MallProductSpecPriceDTO();
                  mallProductSpecPriceDTO.setProductSpecId(d);
                  return mallProductSpecPriceDTO;
                })
            .collect(Collectors.toList());
    // 分别找出渠道价格和市场价格
    List<ProductSpecPriceDO> productSpecPriceDOS =
        productDao.listProductSpecPrice(productSpecPriceQO.getChannelClass(), ids);
    List<ProductSpecPriceDO> productSpecPriceDOS1 = productDao.listProductSpecPrice(0, ids);
    // 设置合作价格
    for (ProductSpecPriceDO productSpecPriceDO : productSpecPriceDOS) {
      for (MallProductSpecPriceDTO mallProductSpecPriceDTO : mallProductSpecPriceDTOS) {
        if (mallProductSpecPriceDTO
            .getProductSpecId()
            .equals(productSpecPriceDO.getProductSpecId())) {
          mallProductSpecPriceDTO.setOpPrice(productSpecPriceDO.getPrice());
          break;
        }
      }
    }
    // 设置市场价格
    for (ProductSpecPriceDO productSpecPriceDO : productSpecPriceDOS1) {
      for (MallProductSpecPriceDTO mallProductSpecPriceDTO : mallProductSpecPriceDTOS) {
        if (mallProductSpecPriceDTO
            .getProductSpecId()
            .equals(productSpecPriceDO.getProductSpecId())) {
          mallProductSpecPriceDTO.setMkPrice(productSpecPriceDO.getPrice());
          break;
        }
      }
    }
    return mallProductSpecPriceDTOS;
  }

  @Override
  public ProductSpecPriceDTO feignGetUnitPriceByTag(Integer id, Integer tagId) {
    ProductSpecPriceDTO price = productDao.feignGetUnitPrice(id, tagId);
    return price;
  }

  @Override
  public List<OrderGoodsProdDTO> feignListProdGoodsSkuInfo(
      MallOrderGoodsInfoQO mallOrderGoodsInfoQO) {
    // 查询渠道等级对应的商品价格
    Set<Integer> prodSkuSpecIds =
        productDao.listProductSpecIds(mallOrderGoodsInfoQO.getMallProdSkuSpecIds());
    // 查询商品信息
    List<OrderGoodsProdDTO> orderGoodsProdDTOList =
        productDao.listProdGoodsSkuInfo(mallOrderGoodsInfoQO);
    // 填充价格
    List<ProductSpecPriceDO> allProductSpecPriceDOS =
        productDao.listAllProductSpecPriceByType(prodSkuSpecIds);
    this.buildProductSpecPrice(
        orderGoodsProdDTOList, allProductSpecPriceDOS, mallOrderGoodsInfoQO.getCooperationTagIds());
    // 还需配置商品规格金额， 单个商品skuSpecAmount， 全部小sku金额
    List<Integer> goodsIds =
        orderGoodsProdDTOList.stream()
            .map(OrderGoodsProdDTO::getGoodsInfoId)
            .collect(Collectors.toList());
    // 拼装商品服务
    List<GoodsServiceDTO> services =
        goodsInfoDao.listGoodsService(goodsIds).stream()
            .map(d -> d.buildGoodsServiceDTO())
            .collect(Collectors.toList());
    Map<Integer, List<GoodsServiceDTO>> mapGoodsService =
        org.springframework.util.CollectionUtils.isEmpty(services)
            ? null
            : services.stream().collect(Collectors.groupingBy(GoodsServiceDTO::getGoodsInfoId));

    List<OrderGoodsProdDTO> collect =
        orderGoodsProdDTOList.stream()
            .map(
                d -> {
                  OrderGoodsProdDTO orderGoodsProdDTO = d;
                  if (mapGoodsService != null && mapGoodsService.containsKey(d.getGoodsInfoId())) {
                    orderGoodsProdDTO.setServices(mapGoodsService.get(d.getGoodsInfoId()));
                  }
                  return orderGoodsProdDTO;
                })
            .collect(Collectors.toList());
    return collect;
  }

  /**
   * 填充产品规格价格
   *
   * @param orderGoodsProdDTOList
   * @param allProductSpecPriceDOS
   * @param cooperationTagIds
   */
  private void buildProductSpecPrice(
      List<OrderGoodsProdDTO> orderGoodsProdDTOList,
      List<ProductSpecPriceDO> allProductSpecPriceDOS,
      List<Integer> cooperationTagIds) {
    Map<Integer, List<ProductSpecPriceDO>> productSpecPriceMap =
        allProductSpecPriceDOS.stream()
            .collect(Collectors.groupingBy(ProductSpecPriceDO::getProductSpecId));
    for (OrderGoodsProdDTO orderGoodsProdDTO : orderGoodsProdDTOList) {
      for (OrderGoodsProdDetailDTO prodDetailDTO :
          orderGoodsProdDTO.getOrderGoodsProdDetailDTOS()) {
        List<ProductSpecPriceDO> specPriceDOS =
            productSpecPriceMap.get(prodDetailDTO.getProductSpecId());
        if (CollectionUtils.isNotEmpty(specPriceDOS)) {
          List<ProductSpecPriceDO> tempSpecPrice =
              specPriceDOS.stream()
                  .filter(d -> cooperationTagIds.contains(d.getCooperationTag()))
                  .sorted(Comparator.comparing(ProductSpecPriceDO::getPrice))
                  .collect(Collectors.toList());
          prodDetailDTO.setUnitPrice(tempSpecPrice.get(0).getPrice());
        }
      }
    }
  }

  @Override
  public List<OrderGoodsIndstDTO> feignListIndstGoodsSkuInfo(
      MallOrderGoodsInfoQO mallOrderGoodsInfoQO) {
    // 查询商品信息
    List<MallGoodsInfoSimpleDO> mallGoodsInfoSimpleDOS =
        goodsInfoDao.listMallGoodsIndstSimpleInfo(mallOrderGoodsInfoQO.getMallIndstSkuSpecIds());
    List<Integer> industrySpecIds =
        goodsInfoDao.listIndustrySpecIds(mallOrderGoodsInfoQO.getMallIndstSkuSpecIds());
    // 查询商品价格
    List<IndustrySpecPriceDO> industrySpecPriceDOS =
        industrySpecDao.listIndustrySpecPrice(
            mallOrderGoodsInfoQO.getCooperationTagId(), industrySpecIds);
    // 查询商品行业规格对应的产品清单
    List<MallGoodsProductDO> mallGoodsProductDOS =
        goodsInfoDao.listIndustryProductList(industrySpecIds);
    // 配置商品行业规格价格
    this.fillOrderGoodsIndstPrice(mallGoodsInfoSimpleDOS, industrySpecPriceDOS, industrySpecIds);
    if (!CollectionUtils.isEmpty(industrySpecIds)) {
      // 查询商品价格，找出市场价
      List<IndustrySpecPriceDO> industryMarketPrice =
          industrySpecDao.listIndustrySpecPrice(0, industrySpecIds);
      this.fillOrderGoodsIndstPrice(mallGoodsInfoSimpleDOS, industryMarketPrice, industrySpecIds);
    }
    // 配置产品清单
    for (MallGoodsInfoSimpleDO mallGoodsInfoSimpleDO : mallGoodsInfoSimpleDOS) {
      for (MallGoodsSpecSimpleDO mallGoodsSpecSimpleDO :
          mallGoodsInfoSimpleDO.getMallGoodsSpecSimpleDOS()) {
        mallGoodsSpecSimpleDO.setMallGoodsProductDOS(new ArrayList<MallGoodsProductDO>());
        // 从产品清单中拿出对应的方案规格里面
        for (MallGoodsProductDO mallGoodsProductDO : mallGoodsProductDOS) {
          if (mallGoodsProductDO
              .getIndustrySpecId()
              .toString()
              .equals(mallGoodsSpecSimpleDO.getIndustrySpecId().toString())) {
            mallGoodsSpecSimpleDO.getMallGoodsProductDOS().add(mallGoodsProductDO);
          }
        }
      }
    }
    // 转换对象
    List<OrderGoodsIndstDTO> collect =
        mallGoodsInfoSimpleDOS.stream()
            .map(d -> d.buildOrderGoodsIndstDTO())
            .collect(Collectors.toList());
    // 拼装商品服务
    List<Integer> goodsIds =
        collect.stream().map(OrderGoodsIndstDTO::getGoodsInfoId).collect(Collectors.toList());
    List<GoodsServiceDTO> services =
        goodsInfoDao.listGoodsService(goodsIds).stream()
            .map(d -> d.buildGoodsServiceDTO())
            .collect(Collectors.toList());
    Map<Integer, List<GoodsServiceDTO>> mapGoodsService =
        org.springframework.util.CollectionUtils.isEmpty(services)
            ? null
            : services.stream().collect(Collectors.groupingBy(GoodsServiceDTO::getGoodsInfoId));
    List<OrderGoodsIndstDTO> resultList =
        collect.stream()
            .map(
                d -> {
                  OrderGoodsIndstDTO orderGoodsIndstDTO = d;
                  if (mapGoodsService != null && mapGoodsService.containsKey(d.getGoodsInfoId())) {
                    orderGoodsIndstDTO.setServices(mapGoodsService.get(d.getGoodsInfoId()));
                  }
                  return orderGoodsIndstDTO;
                })
            .collect(Collectors.toList());
    return resultList;
  }

  private List<MallGoodsInfoSimpleDO> fillOrderGoodsIndstPrice(
      List<MallGoodsInfoSimpleDO> mallGoodsInfoSimpleDOS,
      List<IndustrySpecPriceDO> industrySpecPriceDOS,
      List<Integer> industrySpecIds) {
    // 配置商品行业规格价格
    for (MallGoodsInfoSimpleDO mallGoodsInfoSimpleDO : mallGoodsInfoSimpleDOS) {
      for (MallGoodsSpecSimpleDO mallGoodsSpecSimpleDO :
          mallGoodsInfoSimpleDO.getMallGoodsSpecSimpleDOS()) {
        // 查询规格进行配置
        for (IndustrySpecPriceDO industrySpecPriceDO : industrySpecPriceDOS) {
          if (mallGoodsSpecSimpleDO
              .getIndustrySpecId()
              .toString()
              .equals(industrySpecPriceDO.getIndustrySpecId().toString())) {
            mallGoodsSpecSimpleDO.setUnitPrice(industrySpecPriceDO.getPrice());
            industrySpecIds.removeIf(
                d -> d.toString().equals(industrySpecPriceDO.getIndustrySpecId().toString()));
            break;
          }
        }
      }
    }
    return mallGoodsInfoSimpleDOS;
  }

  @Override
  public PageResult listPageGoodsInfo(MallGoodsQO param) {
    int count = goodsInfoDao.countListGoodsInfo(param);
    if (count == 0) {
      return PageResult.buildPage(param.getPageNo(), param.getPageSize(), count);
    }
    Integer pageNo = param.getPageNo();
    param.buildCurrentPage();
    List<GoodsInfo> goodsInfo = goodsInfoDao.listGoodsInfo(param);
    List<GoodsInfoListDTO> pageList =
        goodsInfo.stream().map(GoodsInfo::buildGoodsInfoListDTO).collect(Collectors.toList());
    return PageResult.buildPage(pageNo, param.getPageSize(), count, pageList);
  }

  @Override
  public ResultBody batchOnShelfOrTakeDown(List<Integer> ids, Integer status) {
    goodsInfoDao.batchUpOrDownWare(ids, status);
    return ResultBody.success();
  }

  @Override
  public ResultBody batchRemoveWareInfo(List<Integer> ids) {
    goodsInfoDao.removeWareInfo(ids);
    return ResultBody.success();
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public ResultBody exchangeGoodsInfo(Integer firstId, Integer secondId) {
    GoodsInfo firstGoodsInfoDO = goodsInfoDao.getGoodsSimpleInfo(firstId);
    GoodsInfo secondGoodsInfoDO = goodsInfoDao.getGoodsSimpleInfo(secondId);
    int updateCount1 = goodsInfoDao.updateGoodsInfoSort(firstId, secondGoodsInfoDO.getSort());
    int updateCount2 = goodsInfoDao.updateGoodsInfoSort(secondId, firstGoodsInfoDO.getSort());
    if (updateCount1 == updateCount2) {
      return ResultBody.success();
    } else {
      TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
      return ResultBody.error("排序失败");
    }
  }
}
