package com.mmc.pms.util;

import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

/**
 * @Author small @Date 2023/5/23 10:16 @Version 1.0
 */
@Slf4j
public class FileLoadUtil {

  /**
   * 批量生成文件
   *
   * @throws IOException
   * @throws IllegalStateException
   */
  public static ResultBody bathCreateFile(String directory, MultipartFile[] files)
      throws IllegalStateException, IOException {
    if (files == null || files.length == 0) {
      return ResultBody.error(
          ResultEnum.FILE_UPLOAD_NULL_ERROR.getResultCode(),
          ResultEnum.FILE_UPLOAD_NULL_ERROR.getResultMsg());
    }
    List<String> downloadPaths = new ArrayList<>();
    for (int i = 0; i < files.length; i++) {
      String filename = FileLoadUtil.createNewFileName(files[i].getOriginalFilename());
      // 创建新路径
      String newFileDir = FileLoadUtil.createFileDirctory(directory);
      // 创建文件对象
      File uploadFile = new File(newFileDir + filename);
      if (!uploadFile.getParentFile().exists()) {
        uploadFile.getParentFile().mkdirs();
      }
      while (uploadFile.exists()) { // 文件名重复
        filename = filename + "(" + i + ")";
        uploadFile = new File(newFileDir + filename);
        i++;
      }
      uploadFile.createNewFile();
      files[i].transferTo(uploadFile);
      downloadPaths.add(FileLoadUtil.createDownLoadPath(directory, filename));
    }
    return ResultBody.success(downloadPaths);
  }

  /**
   * 文件名
   *
   * @param originalFilename
   * @return
   */
  public static String createNewFileName(String originalFilename) {
    int i = originalFilename.lastIndexOf('.');
    String endWith = "";
    StringBuffer sb = new StringBuffer();
    if (i >= 0) {
      endWith = originalFilename.substring(i);
    }
    sb.append(System.currentTimeMillis());
    sb.append(endWith);
    return sb.toString();
  }

  /** 文件夹路径 */
  public static String createFileDirctory(String directory) {
    return FileServletConstant.MOUNTVOLUME
        + directory
        + File.separator
        + TDateUtil.getCurrentDateByType("yyyyMMdd")
        + File.separator;
  }

  /** 解析下载路径 */
  public static String explainLoadPath(String path) {
    return FileServletConstant.MOUNTVOLUME + path.replaceAll("@", FileServletConstant.separator);
  }

  /** 生成下载地址 */
  public static String createDownLoadPath(String directory, String filename) {
    return FileServletConstant.DOWNLOADPATH
        + directory
        + "@"
        + TDateUtil.getCurrentDateByType("yyyyMMdd")
        + "@"
        + filename;
  }

  /**
   * 判断文件大小
   *
   * @param len
   * @param size
   * @param unit
   */
  public static boolean checkFileSize(Long len, int size, String unit) {
    double fileSize = 0;
    if ("B".equals(unit.toUpperCase())) {
      fileSize = (double) len;
    } else if ("K".equals(unit.toUpperCase())) {
      fileSize = (double) len / 1024;
    } else if ("M".equals(unit.toUpperCase())) {
      fileSize = (double) len / 1048576;
    } else if ("G".equals(unit.toUpperCase())) {
      fileSize = (double) len / 1073741824;
    }
    if (fileSize > size) {
      return false;
    }
    return true;
  }

  /**
   * 将文件头转换成16进制字符串
   *
   * @return 16进制字符串
   */
  private static String bytesToHexString(byte[] src) {

    StringBuilder stringBuilder = new StringBuilder();
    if (src == null || src.length <= 0) {
      return null;
    }
    for (int i = 0; i < src.length; i++) {
      int v = src[i] & 0xFF;
      String hv = Integer.toHexString(v);
      if (hv.length() < 2) {
        stringBuilder.append(0);
      }
      stringBuilder.append(hv);
    }
    return stringBuilder.toString();
  }

  /**
   * 得到文件头
   *
   * @return 文件头
   * @throws IOException
   */
  private static String getFileContent(InputStream is) throws IOException {
    final int len = 20;
    byte[] b = new byte[len];
    final boolean isMark = is.markSupported();
    if (isMark) {
      is.mark(len);
    }
    try {
      is.read(b, 0, len);
    } catch (IOException e) {
      log.warn("读取文件头信息失败: ", e);
      throw e;
    } finally {
      if (isMark) {
        is.reset();
      }
    }
    return bytesToHexString(b);
  }

  public static FileTypeConstant getType(InputStream is) throws IOException {
    return getType(getFileContent(is));
  }

  public static FileTypeConstant getType(final byte[] bytes) throws IOException {
    return getType(bytesToHexString(bytes));
  }

  public static FileTypeConstant getType(String fileHead) throws IOException {
    if (fileHead == null || fileHead.length() == 0) {
      return null;
    }
    fileHead = fileHead.toUpperCase();
    FileTypeConstant[] fileTypes = FileTypeConstant.values();

    for (FileTypeConstant type : fileTypes) {
      String s = type.getValue().toUpperCase();
      if (fileHead.contains(type.getValue().toUpperCase())) {
        return type;
      }
    }
    return null;
  }

  public static Integer checkFileType(FileTypeConstant value) {
    Integer type = 4; // 其他
    // 图片
    FileTypeConstant[] pics = {
      FileTypeConstant.JPEG, FileTypeConstant.PNG, FileTypeConstant.BMP, FileTypeConstant.GIF
    };

    FileTypeConstant[] docs = {
      FileTypeConstant.PST,
      FileTypeConstant.XLS_DOC,
      FileTypeConstant.XLSX_DOCX,
      FileTypeConstant.WPS,
      FileTypeConstant.PDF,
      FileTypeConstant.PPTX_DOCX
    };

    FileTypeConstant[] videos = {
      FileTypeConstant.AVI,
      FileTypeConstant.RAM,
      FileTypeConstant.RM,
      FileTypeConstant.MPG,
      FileTypeConstant.MOV,
      FileTypeConstant.ASF,
      FileTypeConstant.MP4,
      FileTypeConstant.FLV,
      FileTypeConstant.MID
    };

    // 图片
    for (FileTypeConstant fileType : pics) {
      if (fileType.equals(value)) {
        type = 1;
      }
    }
    // 文档
    for (FileTypeConstant fileType : docs) {
      if (fileType.equals(value)) {
        type = 2;
      }
    }
    // 视频
    for (FileTypeConstant fileType : videos) {
      if (fileType.equals(value)) {
        type = 3;
      }
    }
    return type;
  }
}
