package com.mmc.pms.service.mall.impl;

import com.mmc.pms.auth.dto.CompanyInfoVO;
import com.mmc.pms.auth.dto.LoginSuccessDTO;
import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.lease.LeaseGoodsDao;
import com.mmc.pms.dao.mall.MallGoodsDao;
import com.mmc.pms.entity.lease.LeasePriceStockDO;
import com.mmc.pms.entity.lease.LeaseSpecAttrDO;
import com.mmc.pms.entity.lease.LeaseSpecAttrValueDO;
import com.mmc.pms.entity.mall.*;
import com.mmc.pms.feign.UserAppApi;
import com.mmc.pms.model.lease.vo.LeaseGoodsVO;
import com.mmc.pms.model.lease.vo.LeasePriceStockVO;
import com.mmc.pms.model.mall.GoodsResourcesVO;
import com.mmc.pms.model.mall.MallGoodsVO;
import com.mmc.pms.model.mall.PriceStockVO;
import com.mmc.pms.model.mall.SpecAttrVO;
import com.mmc.pms.model.sale.qo.MallGoodsInfoQO;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.inspection.CompanyInspectionService;
import com.mmc.pms.service.mall.MallGoodsService;
import com.mmc.pms.util.CodeUtil;
import com.mmc.pms.util.TDateUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @Author LW
 * @date 2023/7/24 17:37 概要：
 */
@Service
public class MallGoodsServiceImpl implements MallGoodsService {
    @Resource
    private MallGoodsDao mallGoodsDao;
    @Resource
    private LeaseGoodsDao leaseGoodsDao;
    @Autowired
    private CompanyInspectionService companyInspectionService;
    @Autowired
    private UserAppApi userAppApi;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody addMallGoods(MallGoodsVO mallGoodsVO, Integer userAccountId) {
        ResultBody resultError = checkInformation(null, mallGoodsVO, userAccountId);
        if (resultError != null) return resultError;
        int count = mallGoodsDao.countMallGoods(userAccountId);
        MallGoodsDO mallGoodsDO = new MallGoodsDO(mallGoodsVO);
        mallGoodsDO.setGoodsNo("MG" + TDateUtil.getDateStr(new Date(), "yyyyMMddHHmmss") + CodeUtil.getRandomNum(4));
        mallGoodsDO.setUserAccountId(userAccountId);
        mallGoodsDO.setSort(count + 1);
        // 将基础信息存储入库
        mallGoodsDao.insertMallGoodsBaseInfo(mallGoodsDO);
        // 将商品图片等资源存入数据库中
        insertMallGoodsResources(null, mallGoodsVO, mallGoodsDO.getId());
        // 将商品规格存入数据库
        addMallGoodsSpec(mallGoodsVO.getSpecAttrList(), mallGoodsDO.getId(), mallGoodsVO.getPriceStock(), null, 0);
        return ResultBody.success();
    }

    /**
     * 添加商品规格
     *
     * @param specAttrList
     * @param id                id
     * @param priceStock
     * @param leasePriceStockVO
     * @param flag              0:商城 其他值：租赁
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void addMallGoodsSpec(List<SpecAttrVO> specAttrList, Integer id, List<PriceStockVO> priceStock, List<LeasePriceStockVO> leasePriceStockVO, Integer flag) {
        if (CollectionUtils.isNotEmpty(specAttrList)) {
            for (SpecAttrVO specAttrVO : specAttrList) {
                SpecAttrDO specAttrDO = new SpecAttrDO();
                specAttrDO.setMallGoodsId(id);
                specAttrDO.setSpecName(specAttrVO.getSpecName());
                if (flag.equals(0)) {
                    mallGoodsDao.insertSpecAttr(specAttrDO);
                } else {
                    leaseGoodsDao.insertSpecAttr(specAttrDO);
                }
                List<SpecAttrVO> specValuesList = specAttrVO.getSpecValuesList();
                List<SpecAttrValueDO> list = specValuesList.stream().map(d -> {
                    SpecAttrValueDO specAttrValueDO = new SpecAttrValueDO();
                    specAttrValueDO.setSpecValueName(d.getSpecName());
                    specAttrValueDO.setSpecAttrId(specAttrDO.getId());
                    return specAttrValueDO;
                }).collect(Collectors.toList());
                if (flag.equals(0)) {
                    mallGoodsDao.batchInsertSpecAttr(list);
                } else {
                    leaseGoodsDao.batchInsertSpecAttr(list);
                }
            }
        }
        if (CollectionUtils.isNotEmpty(priceStock)) {
            List<PriceStockDO> priceStockDOList = priceStock.stream().map(d -> new PriceStockDO(d, id)).collect(Collectors.toList());
            mallGoodsDao.batchInsertPriceStock(priceStockDOList);
        }
        if (CollectionUtils.isNotEmpty(leasePriceStockVO)) {
            List<LeasePriceStockDO> priceStockDOList = leasePriceStockVO.stream().map(d -> new LeasePriceStockDO(d, id)).collect(Collectors.toList());
            leaseGoodsDao.batchInsertPriceStock(priceStockDOList);
        }
    }


    @Transactional(rollbackFor = Exception.class)
    @Override
    public void insertMallGoodsResources(LeaseGoodsVO leaseGoodsVO, MallGoodsVO mallGoodsVO, Integer id) {
        List<MallGoodsResourcesDO> mallGoodsResourcesList;
        if (mallGoodsVO != null) {
            mallGoodsResourcesList = mallGoodsVO.getResourcesList().stream().map(d -> {
                MallGoodsResourcesDO mallGoodsResourcesDO = new MallGoodsResourcesDO(d);
                mallGoodsResourcesDO.setMallGoodsId(id);
                return mallGoodsResourcesDO;
            }).collect(Collectors.toList());
            mallGoodsDao.batchInsertMallGoodsResources(mallGoodsResourcesList);
        }

        if (leaseGoodsVO != null) {
            mallGoodsResourcesList = leaseGoodsVO.getResourcesList().stream().map(d -> {
                MallGoodsResourcesDO mallGoodsResourcesDO = new MallGoodsResourcesDO(d);
                mallGoodsResourcesDO.setMallGoodsId(id);
                return mallGoodsResourcesDO;
            }).collect(Collectors.toList());
            leaseGoodsDao.batchInsertLeaseGoodsResources(mallGoodsResourcesList);
        }
    }

    @Override
    public ResultBody checkInformation(LeaseGoodsVO leaseGoodsVO, MallGoodsVO mallGoodsVO, Integer userAccountId) {
        if (mallGoodsVO != null) {
            // 查询该账账号下是否有相同的商品名称存在
            if (mallGoodsDao.countMallGoodsByName(mallGoodsVO, userAccountId) > 0) {
                return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
            }
            // 判断主图是否为空
            List<GoodsResourcesVO> resourcesList = mallGoodsVO.getResourcesList()
                    .stream().filter(d -> d.getType().equals(0))
                    .collect(Collectors.toList());
            if (resourcesList.size() == 0) {
                return ResultBody.error(ResultEnum.GOODS_PIC_IS_NOT_NULL);
            }
        }

        if (leaseGoodsVO != null) {
            // 查询该账账号下是否有相同的租赁商品名称存在
            if (leaseGoodsDao.countLeaseGoodsByName(leaseGoodsVO, userAccountId) > 0) {
                return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
            }
            // 判断主图是否为空
            List<GoodsResourcesVO> resourcesList = leaseGoodsVO.getResourcesList()
                    .stream().filter(d -> d.getType().equals(0))
                    .collect(Collectors.toList());
            if (resourcesList.size() == 0) {
                return ResultBody.error(ResultEnum.GOODS_PIC_IS_NOT_NULL);
            }
        }
        return null;
    }

    @Override
    public ResultBody<MallGoodsVO> mallGoodsDetails(Integer id) {
        MallGoodsDO mallGoodsDO = mallGoodsDao.getMallGoodsBaseInfo(id);
        if (mallGoodsDO == null) {
            return ResultBody.error("商品不存在或已删除！");
        }
        MallGoodsVO mallGoodsVO = mallGoodsDO.buildMallGoodsVO();
        // 获取图片及其他资源信息
        List<MallGoodsResourcesDO> mallGoodsResourcesList = mallGoodsDao.getMallGoodsResources(id);
        mallGoodsVO.setResourcesList(mallGoodsResourcesList.stream()
                .map(MallGoodsResourcesDO::buildGoodsResourcesVO).collect(Collectors.toList()));
        // 获取规格及规格值
        List<SpecAttrDO> specAttrDOList = mallGoodsDao.listSpecAttr(id);
        List<Integer> specIds = specAttrDOList.stream().map(SpecAttrDO::getId).collect(Collectors.toList());
        List<SpecAttrValueDO> specAttrValueDOList = mallGoodsDao.listSpecAttrValue(specIds);
        // 转成vo
        List<SpecAttrVO> specAttrVOList = specAttrDOList.stream().map(SpecAttrDO::buildSpecAttr).collect(Collectors.toList());
        // 拼装数据
        Map<Integer, List<SpecAttrValueDO>> specValueMap = specAttrValueDOList.stream().collect(Collectors.groupingBy(SpecAttrValueDO::getSpecAttrId));
        specAttrVOList = specAttrVOList.stream().peek(d -> {
            List<SpecAttrValueDO> attrValueList = specValueMap.get(d.getId());
            d.setSpecValuesList(attrValueList.stream().map(SpecAttrValueDO::buildSpecAttr).collect(Collectors.toList()));
        }).collect(Collectors.toList());
        mallGoodsVO.setSpecAttrList(specAttrVOList);
        // 获取价格库存排列组合
        List<PriceStockDO> priceStockDOList = mallGoodsDao.listPriceStock(id);
        mallGoodsVO.setPriceStock(priceStockDOList.stream().map(PriceStockDO::buildPriceStockVO).collect(Collectors.toList()));
        return ResultBody.success(mallGoodsVO);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody editMallGoods(MallGoodsVO mallGoodsVO, Integer userAccountId) {
        ResultBody resultError = checkInformation(null, mallGoodsVO, userAccountId);
        if (resultError != null) return resultError;
        MallGoodsDO mallGoodsDO = new MallGoodsDO(mallGoodsVO);
        // 修改商城商品的基本信息
        mallGoodsDao.updateMallGoods(mallGoodsDO);
        // 修改商城商品的图片等资源信息,先删除后新增
        mallGoodsDao.deleteMallGoodsResources(mallGoodsVO.getId());
        this.insertMallGoodsResources(null, mallGoodsVO, mallGoodsVO.getId());
        // 修改规格
        updateSpecInfo(mallGoodsVO, null);
        return ResultBody.success();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateSpecInfo(MallGoodsVO mallGoodsVO, LeaseGoodsVO leaseGoodsVO) {
        List<SpecAttrVO> specAttrList;
        List<SpecAttrVO> updateSpecAttr;
        List<SpecAttrVO> addSpecAttr;
        if (mallGoodsVO != null) {
            specAttrList = mallGoodsVO.getSpecAttrList();
            // 从数据库获取该商品下的所有规格
            deleteSpecAttr(mallGoodsVO, null, specAttrList);
            // 获取id不为空的规格，进行修改
            updateSpecAttr = specAttrList.stream().filter(spec -> spec.getId() != null).collect(Collectors.toList());
            // 从数据库获取规格下的规格值信息,获取所有，避免重复查询数据库
            Map<Integer, List<SpecAttrValueDO>> specValueMap = mallGoodsDao.listSpecAttrValue(updateSpecAttr.stream().map(SpecAttrVO::getId).collect(Collectors.toList()))
                    .stream().collect(Collectors.groupingBy(SpecAttrValueDO::getSpecAttrId));
            if (CollectionUtils.isNotEmpty(updateSpecAttr)) {
                for (SpecAttrVO specAttrVO : updateSpecAttr) {
                    // 修改规格信息
                    mallGoodsDao.updateSpecAttr(specAttrVO);
                    // 获取该规格下数据库里的规格值信息
                    List<SpecAttrValueDO> dbSpecAttrValue = specValueMap.get(specAttrVO.getId());
                    deleteSpecValue(specAttrVO, null, dbSpecAttrValue);
                    // 获取规格值id不为空的就进行规格值修改
                    List<SpecAttrVO> updateSpecValue = specAttrVO.getSpecValuesList().stream().filter(var -> var.getId() != null).collect(Collectors.toList());
                    if (CollectionUtils.isNotEmpty(updateSpecValue)) {
                        for (SpecAttrVO attrVO : updateSpecValue) {
                            mallGoodsDao.updateSpecAttrValue(attrVO);
                        }
                    }
                    // 新增规格值
                    addNewSpecAttrValues(specAttrVO, 0);
                }
            }
            // 获取规格id为空的值，进行新增
            addSpecAttr = specAttrList.stream().filter(spec -> spec.getId() == null).collect(Collectors.toList());
            // 修改排列组合数据
            mallGoodsDao.deleteMallPriceStock(mallGoodsVO.getId());
            addMallGoodsSpec(addSpecAttr, mallGoodsVO.getId(), mallGoodsVO.getPriceStock(), null, 0);
        } else {
            specAttrList = leaseGoodsVO.getSpecAttrList();
            // 从数据库获取该租赁商品下的所有规格,并删除多余规格
            deleteSpecAttr(null, leaseGoodsVO, specAttrList);
            // 获取id不为空的规格，进行修改
            updateSpecAttr = specAttrList.stream().filter(spec -> spec.getId() != null).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(updateSpecAttr)) {
                // 获取规格及规格值
                List<LeaseSpecAttrDO> leaseSpecAttrDO = leaseGoodsDao.getLeaseSpecAttr(updateSpecAttr.stream().map(SpecAttrVO::getId).collect(Collectors.toList()));
                for (SpecAttrVO specAttrVO : updateSpecAttr) {
                    // 修改规格
                    leaseGoodsDao.updateSpecAttr(specAttrVO);
                    // 删除多余规格
                    for (LeaseSpecAttrDO specAttrDO : leaseSpecAttrDO) {
                        if (specAttrVO.getId().equals(specAttrDO.getId())) {
                            List<LeaseSpecAttrValueDO> dbLeaseSpecAttrValues = specAttrDO.getLeaseSpecAttrValues();
                            deleteSpecValue(specAttrVO, dbLeaseSpecAttrValues, null);
                        }
                    }
                    // 新增规格值的信息
                    addNewSpecAttrValues(specAttrVO, 1);
                }
            }
            // 获取规格id为空的值，进行新增
            addSpecAttr = specAttrList.stream().filter(spec -> spec.getId() == null).collect(Collectors.toList());
            // 修改排列组合数据
            leaseGoodsDao.deleteLeasePriceStock(leaseGoodsVO.getId());
            addMallGoodsSpec(addSpecAttr, leaseGoodsVO.getId(), null, leaseGoodsVO.getPriceStock(), 1);
        }
    }

    @Override
    public PageResult listMallGoodsBySearch(MallGoodsInfoQO param, LoginSuccessDTO userLoginInfoFromRedis) {
        param.setShelfStatus(1);
        PageResult pageResult = getMallGoodsInfo(param);
        return pageResult;
    }

    /**
     * 添加新规格值
     *
     * @param specAttrVO
     * @param flag       标志：0商城 1租赁
     */
    @Transactional(rollbackFor = Exception.class)
    public void addNewSpecAttrValues(SpecAttrVO specAttrVO, Integer flag) {
        List<SpecAttrValueDO> specValueList;
        List<SpecAttrVO> addSpecValue = specAttrVO.getSpecValuesList().stream().filter(var -> var.getId() == null).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(addSpecValue)) {
            specValueList = addSpecValue.stream().map(d -> {
                SpecAttrValueDO specAttrValueDO = new SpecAttrValueDO();
                specAttrValueDO.setSpecAttrId(specAttrVO.getId());
                specAttrValueDO.setSpecValueName(d.getSpecName());
                return specAttrValueDO;
            }).collect(Collectors.toList());
            if (flag.equals(0)) {
                mallGoodsDao.batchInsertSpecAttr(specValueList);
            } else {
                leaseGoodsDao.batchInsertSpecAttr(specValueList);
            }
        }
    }

    /**
     * 删除规格值
     *
     * @param specAttrVO
     * @param dbLeaseSpecAttrValues
     * @param dbSpecAttrValue
     */
    @Transactional(rollbackFor = Exception.class)
    public void deleteSpecValue(SpecAttrVO specAttrVO, List<LeaseSpecAttrValueDO> dbLeaseSpecAttrValues, List<SpecAttrValueDO> dbSpecAttrValue) {
        List<Integer> delValueIds;
        if (CollectionUtils.isNotEmpty(dbSpecAttrValue)) {
            delValueIds = dbSpecAttrValue.stream()
                    .map(SpecAttrValueDO::getId)
                    .filter(id -> !specAttrVO.getSpecValuesList().stream().map(SpecAttrVO::getId).collect(Collectors.toList()).contains(id))
                    .collect(Collectors.toList());
            if (delValueIds.size() != 0) {
                // 删除多余规格值
                mallGoodsDao.deleteSpecAttrValue(delValueIds);
            }
        } else {
            delValueIds = dbLeaseSpecAttrValues.stream()
                    .map(LeaseSpecAttrValueDO::getId)
                    .filter(id -> !specAttrVO.getSpecValuesList().stream().map(SpecAttrVO::getId).collect(Collectors.toList()).contains(id))
                    .collect(Collectors.toList());
            if (delValueIds.size() != 0) {
                // 删除多余规格值
                leaseGoodsDao.deleteSpecAttrValue(delValueIds);
            }
        }
    }

    /**
     * 删除规格
     *
     * @param mallGoodsVO
     * @param leaseGoodsVO
     * @param specAttrList
     */
    @Transactional(rollbackFor = Exception.class)
    public void deleteSpecAttr(MallGoodsVO mallGoodsVO, LeaseGoodsVO leaseGoodsVO, List<SpecAttrVO> specAttrList) {
        List<Integer> delIds;
        if (leaseGoodsVO != null) {
            List<LeaseSpecAttrDO> dbSpecList = leaseGoodsDao.listSpecAttr(leaseGoodsVO.getId());
            delIds = dbSpecList.stream().map(LeaseSpecAttrDO::getId)
                    .filter(id -> !specAttrList.stream().map(SpecAttrVO::getId).collect(Collectors.toList()).contains(id))
                    .collect(Collectors.toList());
            if (delIds.size() != 0) {
                // 删除多余规格
                leaseGoodsDao.deleteSpecAttr(delIds);
            }
        } else {
            List<SpecAttrDO> dbSpecList = mallGoodsDao.listSpecAttr(mallGoodsVO.getId());
            delIds = dbSpecList.stream().map(SpecAttrDO::getId)
                    .filter(id -> !specAttrList.stream().map(SpecAttrVO::getId).collect(Collectors.toList()).contains(id))
                    .collect(Collectors.toList());
            if (delIds.size() != 0) {
                // 删除多余规格
                mallGoodsDao.deleteSpecAttr(delIds);
            }
        }
    }

//    @Override
//    public ResultBody getSkuUnit() {
//        List<SkuUnitDO> skuUnitList = mallGoodsDao.getSkuUnit();
//        List<SkuUnitDTO> list =
//                skuUnitList.stream().map(SkuUnitDO::buildSkuUnitDTO).collect(Collectors.toList());
//        return ResultBody.success(list);
//    }


    @Override
    public PageResult listPageGoodsInfo(MallGoodsInfoQO param, LoginSuccessDTO loginSuccessDTO) {
        if (loginSuccessDTO.getRoleInfo().getSuperAdmin().equals(1)) {
            // 超级管理员获取所有商品信息
            return getMallGoodsInfo(param);
        } else {
            // 非超级管理员获取自家的商品信息
            param.setUserAccountId(loginSuccessDTO.getUserAccountId());
            return getMallGoodsInfo(param);
        }
    }

    private PageResult getMallGoodsInfo(MallGoodsInfoQO param) {
        int count;
        List<MallGoodsDO> mallGoodsList;
        count = mallGoodsDao.countListMallGoods(param);
        if (count == 0) {
            return PageResult.buildPage(param.getPageNo(), param.getPageSize(), count);
        }
        Integer pageNo = param.getPageNo();
        param.buildCurrentPage();
        mallGoodsList = mallGoodsDao.listMallGoods(param);
        for(MallGoodsDO mallGoodsDO : mallGoodsList) {
            List<PriceStockDO> priceStockDOS = mallGoodsDao.listPriceStock(mallGoodsDO.getId());
            if(priceStockDOS != null) {
                mallGoodsDO.setPriceStockDOS(priceStockDOS);
            }
        }
        List<MallGoodsVO> pageList = mallGoodsList.stream().map(MallGoodsDO::buildListMallGoodsVO).collect(Collectors.toList());
        setCompanyName(pageList);
        return PageResult.buildPage(pageNo, param.getPageSize(), count, pageList);
    }

    private void setCompanyName(List<MallGoodsVO> pageList) {
        List<Integer> userAccountIds = pageList.stream().map(MallGoodsVO::getUserAccountId).collect(Collectors.toList());
        List<CompanyInfoVO> companyInfoVOS = userAppApi.listCompanyInfoByBUIds(userAccountIds);
        if (CollectionUtils.isNotEmpty(companyInfoVOS)) {
            Map<Integer, String> collect = companyInfoVOS.stream().collect(Collectors.toMap(CompanyInfoVO::getBackUserAccountId, CompanyInfoVO::getCompanyName));
            pageList.forEach(i -> i.setCompanyName(collect.get(i.getUserAccountId())));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody exchange(Integer firstId, Integer secondId) {
        MallGoodsDO firstMallGoodsBaseInfo = mallGoodsDao.getMallGoodsBaseInfo(firstId);
        MallGoodsDO secondMallGoodsBaseInfo = mallGoodsDao.getMallGoodsBaseInfo(secondId);
        int updateCount1 = mallGoodsDao.updateMallGoodsSort(firstId, secondMallGoodsBaseInfo.getSort());
        int updateCount2 = mallGoodsDao.updateMallGoodsSort(secondId, firstMallGoodsBaseInfo.getSort());
        if (updateCount1 == updateCount2) {
            return ResultBody.success();
        } else {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            return ResultBody.error("排序失败");
        }
    }

    @Override
    public ResultBody upOrDownShelf(Integer id, Integer status) {
        mallGoodsDao.updateMallGoodsByShelf(id, status);
        return ResultBody.success();
    }

    @Override
    public ResultBody removeMallGoods(Integer id) {
        mallGoodsDao.removeMallGoods(id);
        return ResultBody.success();
    }

    @Override
    public ResultBody feignRemoveGoodsByBackUserAccountId(Integer id, Integer companyInfoId) {
        mallGoodsDao.feignRemoveGoodsByBackUserAccountId(id);
        // 删除团队服务
        companyInspectionService.removeCompanyInspectionByCompanyInfoId(companyInfoId);
        return ResultBody.success();
    }
}
