package com.mmc.pms.service.impl;

import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.BrandManageDao;
import com.mmc.pms.dao.ProductDao;
import com.mmc.pms.entity.BrandInfoDO;
import com.mmc.pms.entity.category.DeviceModeDO;
import com.mmc.pms.model.category.qo.ModeInfoQO;
import com.mmc.pms.model.category.vo.ModeInfoVO;
import com.mmc.pms.model.sale.dto.BrandInfoDTO;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.BrandManageService;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @Author LW
 * @date 2023/3/14 13:27 概要：
 */
@Service
public class BrandManageServiceImpl implements BrandManageService {

    @Autowired
    private BrandManageDao brandManageDao;
    @Resource
    private ProductDao productDao;

    /**
     * 添加品牌信息
     *
     * @param brandName 品牌名称
     * @return {@link ResultBody}
     */
    @Override
    public ResultBody addBrandInfo(String brandName) {
        // 根据品牌名称判断是否存在
        int count = brandManageDao.countBrandInfoByName(null, brandName);
        if (count > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        BrandInfoDO brandInfoDO = new BrandInfoDO(brandName);
        brandManageDao.insertBrandInfo(brandInfoDO);
        return ResultBody.success();
    }

    /**
     * 品牌列表信息
     *
     * @return {@link ResultBody}
     */
    @Override
    public PageResult listBrandInfo(Integer pageNo, Integer pageSize) {
        int count = brandManageDao.countBrandInfo();
        if (count == 0) {
            return PageResult.buildPage(pageNo, pageSize, count);
        }
        int itemIndex = (pageNo - 1) * pageSize;
        List<BrandInfoDO> brandInfo = brandManageDao.listBrandInfo(itemIndex, pageSize);
        List<Integer> ids = brandInfo.stream().map(BrandInfoDO::getId).collect(Collectors.toList());
        List<DeviceModeDO> deviceModeDOList = brandManageDao.listDeviceModeByBrandId(ids);
        Map<Integer, List<DeviceModeDO>> map = deviceModeDOList.stream().collect(Collectors.groupingBy(DeviceModeDO::getBrandInfoId));
        List<BrandInfoDTO> brandInfoList =
                brandInfo.stream().map(BrandInfoDO::buildBrandInfoDTO).collect(Collectors.toList());
        brandInfoList = brandInfoList.stream().peek(d -> {
            List<DeviceModeDO> deviceModeList = map.get(d.getId());
            if (CollectionUtils.isNotEmpty(deviceModeList)) {
                d.setModeInfoList(deviceModeList.stream().map(DeviceModeDO::buildModeInfoVO).collect(Collectors.toList()));
            }
        }).collect(Collectors.toList());
        return PageResult.buildPage(pageNo, pageSize, count, brandInfoList);
    }

    /**
     * 删除品牌信息
     *
     * @param id id
     * @return {@link ResultBody}
     */
    @Override
    public ResultBody deleteBrandInfo(Integer id) {
        // 判断该品牌是否绑定产品信息
        int count = productDao.countProductSpecByBrandId(id);
        if (count > 0) {
            return ResultBody.error(ResultEnum.BRAND_DELETE_FAIL);
        }
        brandManageDao.removeBrandInfoById(id);
        return ResultBody.success();
    }

    @Override
    public ResultBody editBrandInfo(Integer id, String brandName) {
        // 根据品牌名称判断是否存在
        int count = brandManageDao.countBrandInfoByName(id, brandName);
        if (count > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        BrandInfoDO brandInfoDO = new BrandInfoDO(brandName);
        brandInfoDO.setId(id);
        brandManageDao.updateBrandInfo(brandInfoDO);
        return ResultBody.success();
    }

    @Override
    public ResultBody addMode(ModeInfoVO param) {
        int count = brandManageDao.countModeInfoByName(param);
        if (count > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        DeviceModeDO modeInfoDO = new DeviceModeDO(param);
        brandManageDao.insertModeInfo(modeInfoDO);
        return ResultBody.success();
    }

    @Override
    public PageResult modeList(ModeInfoQO param) {
        int count = brandManageDao.countModeInfo(param);
        if (count == 0) {
            return PageResult.buildPage(param.getPageNo(), param.getPageSize(), count);
        }
        Integer pageNo = param.getPageNo();
        param.buildCurrentPage();
        List<DeviceModeDO> deviceModeDOList = brandManageDao.listDeviceMode(param);
        List<ModeInfoVO> list = deviceModeDOList.stream().map(DeviceModeDO::buildModeInfoVO).collect(Collectors.toList());
        return PageResult.buildPage(pageNo, param.getPageSize(), count, list);
    }

    @Override
    public ResultBody editMode(ModeInfoVO param) {
        int count = brandManageDao.countModeInfoByName(param);
        if (count > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        DeviceModeDO modeInfoDO = new DeviceModeDO(param);
        brandManageDao.updateModeInfo(modeInfoDO);
        return ResultBody.success();
    }


}
