package com.mmc.oms.service.uav.impl;

import com.mmc.oms.client.PmsClient;
import com.mmc.oms.common.json.JsonUtil;
import com.mmc.oms.common.result.PageResult;
import com.mmc.oms.common.result.ResultBody;
import com.mmc.oms.common.util.CodeUtil;
import com.mmc.oms.common.util.KdnExpressUtil;
import com.mmc.oms.dao.UserAddressDao;
import com.mmc.oms.dao.uav.UavOrderDao;
import com.mmc.oms.dao.uav.UavOrderExpressDao;
import com.mmc.oms.entity.mall.UserAddressDO;
import com.mmc.oms.entity.uav.*;
import com.mmc.oms.enums.UavOrderStatus;
import com.mmc.oms.feign.UserAppApi;
import com.mmc.oms.model.dto.kdn.KdnExpDTO;
import com.mmc.oms.model.dto.order.ExpStationsDTO;
import com.mmc.oms.model.dto.uav.UavOrderDTO;
import com.mmc.oms.model.dto.uav.UavOrderPayDTO;
import com.mmc.oms.model.dto.uav.UavOrderSkuDTO;
import com.mmc.oms.model.dto.uav.UavOrderStatusDTO;
import com.mmc.oms.model.dto.user.BaseAccountDTO;
import com.mmc.oms.model.qo.uav.UavOrderQO;
import com.mmc.oms.model.vo.demand.CommonPaymentVO;
import com.mmc.oms.model.vo.uav.*;
import com.mmc.oms.service.demand.DemandReleaseOrderService;
import com.mmc.oms.service.uav.UavOrderService;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author: zj
 * @Date: 2023/9/4 15:31
 */
@Service
public class UavOrderServiceImpl implements UavOrderService {
    @Autowired
    private UavOrderDao uavOrderDao;

    @Autowired
    private UavOrderExpressDao uavOrderExpressDao;

    @Autowired
    private UserAddressDao userAddressDao;

    @Autowired
    private DemandReleaseOrderService demandReleaseOrderService;

    @Autowired
    private UserAppApi userAppApi;

    @Autowired
    private PmsClient pmsClient;

    @Override
    @Transactional
    public ResultBody addOrder(AddUavOrderVO param, BaseAccountDTO currentAccount) {
        if (CollectionUtils.isEmpty(param.getSkuVOS())){
            return ResultBody.error("至少选择一个规格！");
        }
        MallGoodsVO mallGoodsVO = null;
        try {
            mallGoodsVO = pmsClient.mallGoodsDetails(param.getMallGoodsId());
            // 检验商品是否在卖
            if (mallGoodsVO == null || !mallGoodsVO.getShelfStatus().equals(1)){
                return ResultBody.error("商品已下架");
            }
            // 找到对应的规格
            for (UavOrderSkuVO skuVO : param.getSkuVOS()) {
                Boolean flag = false;
                for (PriceStockVO priceStockVO : mallGoodsVO.getPriceStock()) {
                    if (skuVO.getProductSpec().equals(priceStockVO.getProductSpec())) {
                        // 获取最新的规格单价
                        skuVO.setUnitPrice(priceStockVO.getSalePrice());
                        flag = true;
                        break;
                    }
                }
                if (!flag){
                    return ResultBody.error("选择规格不存在");
                }
            }
        }catch (Exception e){
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            e.printStackTrace();
        }
        String orderNo = CodeUtil.uavOrderCode();
        BigDecimal orderTotalAmount = BigDecimal.ZERO;
        for (UavOrderSkuVO skuVO : param.getSkuVOS()) {
            // 计算规格金额
            BigDecimal skuAmount = BigDecimal.valueOf(skuVO.getOrderNum()).multiply(skuVO.getUnitPrice());
            orderTotalAmount = orderTotalAmount.add(skuAmount);
        }
        if (orderTotalAmount.compareTo(param.getOrderTotalAmount()) != 0) {
            return ResultBody.error("价格有变动，请重新选择规格后下单！");
        }
        // TODO 校验云享金、钱包余额是否足够抵扣
        param.getShareAmount();
        param.getWalletAmount();

        // 添加总订单表
        UavOrderDO uavOrderDO = new UavOrderDO();
        uavOrderDO.setOrderNo(orderNo);
        uavOrderDO.setOrderTotalAmount(orderTotalAmount);
        uavOrderDO.setOrderType(param.getOrderType());
        if (param.getOrderType().equals(0)){
            uavOrderDO.setPayType("wechatpay");
            uavOrderDO.setStatusCode(UavOrderStatus.PAYING.getCode());
        }else {
            uavOrderDO.setStatusCode(UavOrderStatus.CONTACT.getCode());
        }
        uavOrderDO.setUserAccountId(currentAccount.getUserAccountId());
        uavOrderDO.setThirdBackUserAccountId(param.getThirdBackUserAccountId());
        uavOrderDO.setCompanyName(param.getCompanyName());
        uavOrderDO.setWalletAmount(param.getWalletAmount());
        uavOrderDO.setShareAmount(param.getShareAmount());
        uavOrderDO.setOtherAmount(param.getOtherAmount());
        uavOrderDO.setUserAddressId(param.getUserAddressId());
        uavOrderDO.setUserRemark(param.getUserRemark());
        uavOrderDao.addOrder(uavOrderDO);
        List<UavOrderSkuDO> skuDOS = new ArrayList<>();
        for (UavOrderSkuVO skuVO : param.getSkuVOS()) {
            UavOrderSkuDO uavOrderSkuDO = new UavOrderSkuDO();
            // 添加订单规格信息
            uavOrderSkuDO.setUavOrderId(uavOrderDO.getId());
            uavOrderSkuDO.setMallGoodsId(param.getMallGoodsId());
            uavOrderSkuDO.setTradeName(mallGoodsVO.getTradeName());
            uavOrderSkuDO.setPriceStockId(skuVO.getPriceStockId());
            uavOrderSkuDO.setProductSpec(skuVO.getProductSpec());
            uavOrderSkuDO.setOrderNum(skuVO.getOrderNum());
            uavOrderSkuDO.setUnitPrice(skuVO.getSalePrice());
            uavOrderSkuDO.setSkuNo(skuVO.getSkuNo());
            uavOrderSkuDO.setSkuImage(skuVO.getSkuImage());
            skuDOS.add(uavOrderSkuDO);
        }
        uavOrderDao.addOrderSku(skuDOS);
        // 添加收货地址
        UserAddressDO userAddressDO = userAddressDao.selectById(param.getUserAddressId());
        UavOrderExpressDO uavOrderExpressDO = new UavOrderExpressDO(userAddressDO);
        uavOrderExpressDO.setUavOrderId(uavOrderDO.getId());
        uavOrderExpressDao.addAddress(uavOrderExpressDO);
        return ResultBody.success();
    }

    @Override
    public ResultBody detail(Integer id) throws Exception {
        UavOrderDO uavOrderDO = uavOrderDao.detail(id);
        UavOrderDTO uavOrderDTO = uavOrderDO.buildUavOrderDTO();
        // 收货信息、付款信息、规格信息、快递信息、签署信息
        UavOrderExpressDO uavOrderExpressDO = uavOrderExpressDao.getUavOrderExpressDO(id);
        if (uavOrderExpressDO != null) {
            uavOrderDTO.setUavOrderExpressDTO(uavOrderExpressDO.buildUavOrderExpressDTO());
        }
        List<UavOrderPayDO> payDOS = uavOrderDao.listUavOrderPay(id);
        if (CollectionUtils.isNotEmpty(payDOS)) {
            List<UavOrderPayDTO> collect = payDOS.stream().map(UavOrderPayDO::buildUavOrderPayDTO).collect(Collectors.toList());
            uavOrderDTO.setPayDTOList(collect);
        }
        List<UavOrderSkuDO> skuDOS =uavOrderDao.listUavOrderSkuDO(id);
        if (CollectionUtils.isNotEmpty(skuDOS)) {
            List<UavOrderSkuDTO> uavOrderSkuDTOS = skuDOS.stream().map(UavOrderSkuDO::buildUavOrderSkuDTO).collect(Collectors.toList());
            uavOrderDTO.setSkuDTOList(uavOrderSkuDTOS);
        }
        // 快递信息
        KdnExpDTO kdnExpDTO = this.getOrderExpInfo(uavOrderExpressDO);
        if (kdnExpDTO != null) {
            uavOrderDTO.setKdnExpDTO(kdnExpDTO);
        }
        return ResultBody.success(uavOrderDTO);
    }

    private KdnExpDTO getOrderExpInfo(UavOrderExpressDO exp) throws Exception {
        String json = "";
        if ("SF".equals(exp.getSendExpCode())) {
            json = KdnExpressUtil.SF(exp.getTakePhone(), exp.getSendExpNo());
        } else if ("JD".equals(exp.getSendExpCode())) {
            json = KdnExpressUtil.JD("", exp.getSendExpNo());
        } else {
            json = KdnExpressUtil.others(exp.getSendExpCode(), exp.getSendExpNo());
        }
        KdnExpDTO kdn = JsonUtil.parseJsonToObj(json, KdnExpDTO.class);
        kdn.setLogisticCode(exp.getSendExpNo());
        if (!org.springframework.util.CollectionUtils.isEmpty(kdn.getTraces())) {
            Collections.sort(kdn.getTraces(), Comparator.comparing(ExpStationsDTO::getAcceptTime).reversed());// 升序
        }
        return kdn;
    }

    @Override
    public ResultBody close(Integer id) {
        // TODO 退款、修改订单状态
        this.updateUavOrderStatus(id, UavOrderStatus.CLOSE);
        return ResultBody.success();
    }

    @Override
    public ResultBody list(UavOrderQO uavOrderQO, BaseAccountDTO currentAccount) {
        if (!currentAccount.getRoleInfo().getSuperAdmin().equals(1)){
            uavOrderQO.setThirdBackUserAccountId(currentAccount.getUserAccountId());
        }
        int count = uavOrderDao.countList(uavOrderQO);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(uavOrderQO.getPageNo(), uavOrderQO.getPageSize(), count));
        }
        Integer pageNo = uavOrderQO.getPageNo();
        uavOrderQO.buildCurrentPage();
        List<UavOrderDO> uavOrderDOS = uavOrderDao.list(uavOrderQO);
        List<UavOrderDTO> uavOrderDTOS = uavOrderDOS.stream().map(UavOrderDO::buildUavOrderDTO).collect(Collectors.toList());
        return ResultBody.success(PageResult.buildPage(pageNo, uavOrderQO.getPageSize(), count, uavOrderDTOS));
    }

    @Override
    public ResultBody listByUid(UavOrderQO uavOrderQO, BaseAccountDTO currentAccount) {
        uavOrderQO.setUserAccountId(currentAccount.getUserAccountId());
        int count = uavOrderDao.countList(uavOrderQO);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(uavOrderQO.getPageNo(), uavOrderQO.getPageSize(), count));
        }
        Integer pageNo = uavOrderQO.getPageNo();
        uavOrderQO.buildCurrentPage();
        List<UavOrderDO> uavOrderDOS = uavOrderDao.list(uavOrderQO);
        List<UavOrderDTO> uavOrderDTOS = uavOrderDOS.stream().map(UavOrderDO::buildUavOrderDTO).collect(Collectors.toList());
        return ResultBody.success(PageResult.buildPage(pageNo, uavOrderQO.getPageSize(), count, uavOrderDTOS));
    }

    @Override
    public ResultBody pay(Integer id, BaseAccountDTO currentAccount, HttpServletRequest request) {
        // 发起微信支付
        UavOrderDO uavOrderDO = uavOrderDao.detail(id);
        if (uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.PAYING.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        CommonPaymentVO commonPaymentVO = new CommonPaymentVO();
        ResultBody resultBody = demandReleaseOrderService.orderPayment(commonPaymentVO, request, currentAccount.getUserAccountId());
        return resultBody;
    }

    @Transactional
    @Override
    public ResultBody send(UavOrderExpressVO param) {
        // 修改订单状态
        UavOrderDO uavOrderDO = uavOrderDao.detail(param.getUavOrderId());
        if (uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.WAITING_DELIVER_GOODS.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        this.updateUavOrderStatus(param.getUavOrderId(), UavOrderStatus.WAITING_DELIVER_GOODS);
        // 修改收货信息
        UavOrderExpressDO uavOrderExpressDO = new UavOrderExpressDO(param);
        uavOrderExpressDao.updateUavOrderExpressDO(uavOrderExpressDO);
        return ResultBody.success();
    }

    @Override
    public ResultBody receive(Integer id) {
        // 修改订单状态
        UavOrderDO uavOrderDO = uavOrderDao.detail(id);
        if (uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.WAITING_RECEIVE_GOODS.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        this.updateUavOrderStatus(id, UavOrderStatus.WAITING_RECEIVE_GOODS);
        // 修改收货信息
        UavOrderExpressDO uavOrderExpressDO = uavOrderExpressDao.getUavOrderExpressDO(id);
        uavOrderExpressDO.setReceive(1);
        uavOrderExpressDO.setReceiveTime(new Date());
        uavOrderExpressDao.updateUavOrderExpressDO(uavOrderExpressDO);
        // TODO 订单抽成设置
        return ResultBody.success();
    }

    @Override
    public ResultBody remarkOrder(UavOrderRemarkVO uavOrderRemarkVO) {
        // 填入评价内容
        // 修改订单状态
        return null;
    }

    @Transactional
    public void updateUavOrderStatus(Integer id, UavOrderStatus uavOrderStatus){
        uavOrderDao.updateUavOrderStatus(id, uavOrderStatus.getNextCode());
    }

    @Override
    public ResultBody sellerRemark(Integer id, String content) {
        uavOrderDao.sellerRemark(id, content);
        return ResultBody.success();
    }

    @Override
    public ResultBody statusList() {
        List<UavOrderStatusDO> statusDOS = uavOrderDao.statusList();
        List<UavOrderStatusDTO> uavOrderStatusDTOS = statusDOS.stream().map(UavOrderStatusDO::buildUavOrderStatusDTO).collect(Collectors.toList());
        return ResultBody.success(uavOrderStatusDTOS);
    }

    @Override
    public ResultBody upLoadPay(UavOrderPayVO uavOrderPayVO) {
        return null;
    }

    @Override
    public ResultBody checkPay(UavOrderPayVO uavOrderPayVO) {
        return null;
    }
}
