package com.mmc.oms.controller;

import com.mmc.oms.common.publicinterface.Update;
import com.mmc.oms.common.result.ResultBody;
import com.mmc.oms.model.dto.coupon.CouponListDTO;
import com.mmc.oms.model.dto.coupon.CouponUsageDetailsDTO;
import com.mmc.oms.model.dto.coupon.CouponUserDTO;
import com.mmc.oms.model.dto.coupon.CouponUserOrderDTO;
import com.mmc.oms.model.dto.goods.AppGoodsInfoDTO;
import com.mmc.oms.model.dto.order.OrderCouponDTO;
import com.mmc.oms.model.dto.user.BaseAccountDTO;
import com.mmc.oms.model.vo.coupon.CouponUserExchangeVO;
import com.mmc.oms.model.vo.coupon.CouponUserVO;
import com.mmc.oms.service.CouponUserService;
import io.swagger.annotations.*;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import springfox.documentation.annotations.ApiIgnore;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.List;

/**
 * @Author small
 * @Date 2023/5/24 15:56
 * @Version 1.0
 */
@RestController
@RequestMapping("/couponUser/app")
@Api(tags = {"优惠券小程序-相关接口"})
public class CouponUserController extends BaseController {

    @Resource
    private CouponUserService couponUserService;


    @ApiOperation(value = "查看我的优惠卷")
    @ApiResponses({@ApiResponse(code = 200, message = "OK", response = CouponUserDTO.class)})
    @GetMapping("/checkMyCoupon")
    public ResultBody<CouponUserDTO> getMyCoupon(@ApiParam(value = "状态 1：已使用, 2:已失效，3:已转赠")
                                                 @RequestParam(value = "status", required = false) Integer status,
                                                 @ApiParam(value = "页码", required = true)
                                                 @RequestParam(value = "pageNo") Integer pageNo,
                                                 @ApiParam(value = "每页显示数", required = true)
                                                 @RequestParam(value = "pageSize") Integer pageSize,
                                                 HttpServletRequest request) {
        return couponUserService.getMyList(status, pageNo, pageSize, this.getCurrentAccount(request));
    }


    @ApiOperation(value = "领取优惠券")
    @ApiResponses({@ApiResponse(code = 200, message = "OK")})
    @GetMapping("/receive")
    public ResultBody receive(@ApiParam(value = "优惠券id", required = true)
                              @RequestParam(value = "id") Integer id,
                              HttpServletRequest request) {
        return couponUserService.receiveCoupon(id, this.getCurrentAccount(request));
    }


    @ApiOperation(value = "转赠优惠券")
    @ApiResponses({@ApiResponse(code = 200, message = "OK")})
    @PostMapping("/presented")
    public ResultBody presented(@Validated(Update.class)
                                @ApiParam(value = "优惠券id", required = true)
                                @RequestParam(value = "id") Integer id,
                                @ApiParam(value = "用户uid", required = true)
                                @RequestParam(value = "uid") String uid,
                                HttpServletRequest request) {
        return couponUserService.presentedCoupon(id, uid, this.getCurrentAccount(request));
    }


    @ApiOperation(value = "获取可领取优惠券列表")
    @ApiResponses({@ApiResponse(code = 200, message = "OK", response = CouponListDTO.class)})
    @GetMapping("/webPageList")
    public ResultBody<CouponListDTO> getList(@ApiParam(value = "品牌id 搜索品牌指定优惠卷", required = false)
                                             @RequestParam(value = "productId", defaultValue = "0") List<Integer> productId,
                                             @ApiParam(value = "页码", required = true)
                                             @RequestParam(value = "pageNo") Integer pageNo,
                                             @ApiParam(value = "每页显示数", required = true)
                                             @RequestParam(value = "pageSize") Integer pageSize,
                                             HttpServletRequest request) {
        return couponUserService.getList(productId, pageNo, pageSize, this.getCurrentAccount(request));
    }


    @ApiOperation(value = "feign-根据订单品牌获取可用优惠券", hidden = true)
    @ApiResponses({@ApiResponse(code = 200, message = "OK", response = CouponUserOrderDTO.class)})
    @GetMapping("/feignCoupons")
    public List<CouponUserOrderDTO> couponByOrder(@ApiParam(value = "品牌id 搜索品牌指定优惠卷", required = true)
                                                  @RequestParam(value = "primaryKey") List<Integer> primaryKey,
                                                  @ApiParam(value = "用户id", required = true)
                                                  @RequestParam Integer uid) {
        return couponUserService.getUsableCoupon(primaryKey, uid);
    }


    @ApiOperation(value = "feign-根据id查询我的优惠券详情", hidden = true)
    @ApiResponses({@ApiResponse(code = 200, message = "OK", response = CouponUserDTO.class)})
    @GetMapping("/feignMyCoupons")
    private List<CouponUserDTO> feignGetByIds(@ApiParam(value = "id", required = true)
                                              @RequestParam List<Integer> couponIds) {
        return couponUserService.feignGetById(couponIds);
    }

    @ApiOperation(value = "feign-下单修改我的优惠券", hidden = true)
    @ApiResponses({@ApiResponse(code = 200, message = "OK")})
    @PostMapping("/feignUpMine")
    private Boolean feignUpMyCoupon(@RequestBody List<CouponUserVO> couponUserVO) {
        return couponUserService.feignUpMyCoupon(couponUserVO);
    }

    @ApiOperation(value = "feign-积分兑换优惠券", hidden = true)
    @ApiResponses({@ApiResponse(code = 200, message = "OK")})
    @PostMapping("/feignExchange")
    private Boolean feignExchangeCoupon(@RequestBody CouponUserExchangeVO couponUserExchangeVO) {
        return couponUserService.feignExchangeCoupon(couponUserExchangeVO);
    }

    @ApiOperation(value = "feign-领取优惠券", hidden = true)
    @ApiResponses({@ApiResponse(code = 200, message = "OK")})
    @GetMapping("/feignReceive")
    public ResultBody feignReceive(@ApiParam(value = "优惠券id", required = true)
                                   @RequestParam(value = "id") Integer id,
                                   @RequestParam Integer uid,
                                   @RequestParam String uuid,
                                   @RequestParam String phone) {
        BaseAccountDTO countDTO = new BaseAccountDTO();
        countDTO.setUserAccountId(uid);
        countDTO.setUid(uuid);
        countDTO.setPhoneNum(phone);
        return couponUserService.receiveCoupon(id, countDTO);
    }

    @ApiOperation(value = "feign-根据优惠券id查询优惠券", hidden = true)
    @ApiResponses({@ApiResponse(code = 200, message = "OK")})
    @PostMapping("/feignListUserCouponsByIds")
    @ApiIgnore
    public List<CouponUserOrderDTO> feignListUserCouponsByIds(@RequestBody List<Integer> couponIds) {
        return couponUserService.feignListUserCouponsByIds(couponIds);
    }

    @ApiOperation(value = "feign-根据订单id查询使用优惠券", hidden = true)
    @ApiResponses({@ApiResponse(code = 200, message = "OK", response = CouponUserDTO.class)})
    @GetMapping("/feignOrderCoupon")
    public List<CouponUserDTO> feignOrderCoupon(@ApiParam(value = "orderId", required = true)
                                                @RequestParam String orderId) {
        return couponUserService.feignOrderCoupon(orderId);
    }

    @ApiOperation(value = "feign-优惠券定时过期", hidden = true)
    @ApiResponses({@ApiResponse(code = 200, message = "OK")})
    @GetMapping("/feignExpire")
    public void feignExpire() {
        couponUserService.overdueTask();
    }

    @ApiOperation(value = "feign-优惠券回退", hidden = true)
    @ApiResponses({@ApiResponse(code = 200, message = "OK")})
    @PostMapping("/feignRollbackCoupons")
    public Boolean feignRollbackCoupons(@RequestBody List<OrderCouponDTO> orderCouponDTOS) {
        return couponUserService.feignRollbackCoupons(orderCouponDTOS);
    }

    @ApiOperation(value = "feign-批量领取优惠券")
    @ApiResponses({@ApiResponse(code = 200, message = "OK")})
    @GetMapping("/feignReceiveByList")
    public ResultBody feignReceiveByList(@ApiParam(value = "优惠券id", required = true)
                                         @RequestParam(value = "id") List<Integer> id,
                                         @RequestParam Integer uid,
                                         @RequestParam String uuid,
                                         @RequestParam String phone) {
        BaseAccountDTO countDTO = new BaseAccountDTO();
        countDTO.setUserAccountId(uid);
        countDTO.setUid(uuid);
        countDTO.setPhoneNum(phone);
        return couponUserService.receiveCouponList(id, countDTO);
    }

    @ApiOperation(value = "优惠券拥有的商品")
    @ApiResponses({@ApiResponse(code = 200, message = "OK", response = AppGoodsInfoDTO.class)})
    @ApiImplicitParams({
            @ApiImplicitParam(name = "couponId", required = false, value = "优惠券id"),
            @ApiImplicitParam(name = "productName", required = false, value = "商品名称"),
            @ApiImplicitParam(name = "useType", required = false, value = "1vip券, 2品牌券"),
    })
    @GetMapping("/couponMerchandise")
    public ResultBody<AppGoodsInfoDTO> couponMerchandise(@RequestParam(value = "couponId", required = false) Integer couponId,
                                                         @RequestParam(value = "productName", required = false) String productName,
                                                         @RequestParam(value = "useType", required = false) Integer useType) {
        return couponUserService.couponMerchandise(couponId, productName, useType);
    }

    @ApiOperation(value = "优惠券使用明细 ")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "uid", required = true, value = "领劵人id"),
    })
    @ApiResponses({@ApiResponse(code = 200, message = "OK", response = CouponUsageDetailsDTO.class)})
    @GetMapping("/couponUsageDetails")
    public ResultBody<CouponUsageDetailsDTO> couponUsageDetails(@RequestParam(value = "uid", required = true) Integer uid) {
        return couponUserService.couponUsageDetails(uid);
    }

}
