package com.mmc.oms.service.mall.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.mmc.oms.client.PmsClient;
import com.mmc.oms.client.UserClient;
import com.mmc.oms.common.constant.TokenConstant;
import com.mmc.oms.common.result.PageResult;
import com.mmc.oms.common.result.ResultBody;
import com.mmc.oms.common.util.CodeUtil;
import com.mmc.oms.dao.mall.ServiceOrderProvinceCityDao;
import com.mmc.oms.dao.mall.ServiceOrderTaskDao;
import com.mmc.oms.dao.mall.ServiceOrderTaskImageDao;
import com.mmc.oms.entity.mall.ServiceOrderProvinceCityDO;
import com.mmc.oms.entity.mall.ServiceOrderTaskDO;
import com.mmc.oms.entity.mall.ServiceOrderTaskImageDO;
import com.mmc.oms.feign.UserAppApi;
import com.mmc.oms.feign.service.WorkServiceUrl;
import com.mmc.oms.model.dto.mall.ServiceOrderTaskDTO;
import com.mmc.oms.model.dto.service.ServiceDTO;
import com.mmc.oms.model.dto.user.UserAccountSimpleDTO;
import com.mmc.oms.model.qo.mall.ServiceOrderTaskQO;
import com.mmc.oms.model.vo.mall.ServiceOrderTaskVO;
import com.mmc.oms.service.mall.ServiceOrderTaskService;
import com.mmc.oms.util.BeanCopyUtils;
import com.mmc.oms.util.HttpUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 云享飞服务订单-主表 服务实现类
 *
 * @author Pika
 * @since 2023-06-10
 */
@Service
public class ServiceOrderTaskServiceImpl
    extends ServiceImpl<ServiceOrderTaskDao, ServiceOrderTaskDO>
    implements ServiceOrderTaskService {

  @Autowired private ServiceOrderTaskDao serviceOrderTaskDao;

  @Autowired private ServiceOrderProvinceCityDao serviceOrderProvinceCityDao;

  @Autowired private ServiceOrderTaskImageDao serviceOrderTaskImageDao;

  @Autowired private PmsClient pmsClient;

  @Autowired private UserClient userClient;

  @Autowired private HttpUtil httpUtil;

  @Override
  @Transactional(rollbackFor = Exception.class)
  public ResultBody insertOrderTask(ServiceOrderTaskVO param, Integer userAccountId,HttpServletRequest request) {
    ServiceDTO serviceDO = pmsClient.feignQueryWorkService(param.getInspectionId(),request.getHeader(TokenConstant.TOKEN));
    if (serviceDO == null || serviceDO.getId()==null) {
      return ResultBody.error("服务id不存在");
    }
    // 保存作业服务订单
    ServiceOrderTaskDO orderTaskDO =
        ServiceOrderTaskDO.builder()
            .orderNo(CodeUtil.createOrderTaskNO())
            .orderName(serviceDO.getServiceName())
            .inspectionId(param.getInspectionId())
            .inspectionName(serviceDO.getServiceName())
            .taskDescription(param.getTaskDescription())
            .startTime(param.getStartTime())
            .endTime(param.getEndTime())
            .userAccountId(userAccountId)
            .build();
    serviceOrderTaskDao.insert(orderTaskDO);

    // 保存作业服务地址信息
    ServiceOrderProvinceCityDO provinceCityDO =
        ServiceOrderProvinceCityDO.builder()
            .serviceOrderTaskId(orderTaskDO.getId())
            .province(param.getProvince())
            .city(param.getCity())
            .address(param.getAddress())
            .lat(param.getLat())
            .lon(param.getLon())
            .build();
    serviceOrderProvinceCityDao.insert(provinceCityDO);

    // 保存作业服务图片
    List<String> images = param.getImages();
    if (CollectionUtils.isNotEmpty(images)) {
      ServiceOrderTaskImageDO imageDO =
          ServiceOrderTaskImageDO.builder()
              .orderTaskId(orderTaskDO.getId())
              .picture(String.join(",", images))
              .build();
      serviceOrderTaskImageDao.insert(imageDO);
    }
    return ResultBody.success();
  }

  @Override
  public PageResult queryOrderTaskList(ServiceOrderTaskQO param, Integer userAccountId, HttpServletRequest request) {
    Integer pageNo = param.getPageNo();
    param.buildCurrentPage();
    Integer count = serviceOrderTaskDao.conditionCount(param, userAccountId);
    if (count == null || count == 0) {
      return PageResult.buildPage(param.getPageNo(), param.getPageSize(), count);
    }
    List<ServiceOrderTaskDO> orderTaskList =
        serviceOrderTaskDao.queryPageByLimit(param, userAccountId);
    List<Integer> orderTaskIds =
        orderTaskList.stream().map(ServiceOrderTaskDO::getId).collect(Collectors.toList());

    Map<Integer, ServiceOrderProvinceCityDO> provinceCityMap = getOrderProvinceCityDOMapById(orderTaskIds);

    Map<Integer, ServiceOrderTaskImageDO> imageMap = getImageListMapById(orderTaskIds);

    List<Integer> userIds = orderTaskList.stream().map(ServiceOrderTaskDO::getUserAccountId).collect(Collectors.toList());
    Map<Integer, UserAccountSimpleDTO> userMap = getUserDTOMap(request, userIds);

    List<Integer> inspectionIds = orderTaskList.stream().map(ServiceOrderTaskDO::getInspectionId).collect(Collectors.toList());
    Map<Integer, ServiceDTO> serviceDTOMap = getServiceDTOMap(inspectionIds);

    List<ServiceOrderTaskDTO> pageList = new ArrayList<>();
    for (ServiceOrderTaskDO item : orderTaskList) {
      ServiceOrderTaskDTO orderTaskDTO = getServiceOrderTaskDTO(provinceCityMap, imageMap, item,userMap,serviceDTOMap);
      pageList.add(orderTaskDTO);
    }
    return PageResult.buildPage(pageNo, param.getPageSize(), count, pageList);
  }

  private Map<Integer, ServiceDTO> getServiceDTOMap(List<Integer> inspectionIds) {
    String jsonStr = httpUtil.sendPost(WorkServiceUrl.feignQueryWorkServiceListById, null, JSONObject.toJSONString(inspectionIds));
    List<ServiceDTO> serviceDTOS = JSONObject.parseArray(jsonStr, ServiceDTO.class);
    Map<Integer, ServiceDTO> serviceDTOMap = new HashMap<>();
    if(CollectionUtils.isNotEmpty(serviceDTOS)){
      serviceDTOMap = serviceDTOS.stream().collect(Collectors.toMap(ServiceDTO::getId, d -> d, (k1, k2) -> k1));
    }
    return serviceDTOMap;
  }

  @Override
  public ResultBody<ServiceOrderTaskDTO> queryById(Integer id,HttpServletRequest request) {
    ServiceOrderTaskDO order = serviceOrderTaskDao.selectById(id);
    if(order==null){
      return ResultBody.success();
    }
    Map<Integer, ServiceOrderProvinceCityDO> provinceCityMap = getOrderProvinceCityDOMapById(Arrays.asList(id));
    Map<Integer, ServiceOrderTaskImageDO> imageMap = getImageListMapById(Arrays.asList(id));
    Map<Integer, UserAccountSimpleDTO> userMap = getUserDTOMap(request, Arrays.asList(order.getUserAccountId()));
    Map<Integer, ServiceDTO> serviceDTOMap = getServiceDTOMap(Arrays.asList(order.getInspectionId()));
    ServiceOrderTaskDTO serviceOrderTaskDTO = getServiceOrderTaskDTO(provinceCityMap, imageMap, order,userMap,serviceDTOMap);
    return ResultBody.success(serviceOrderTaskDTO);
  }

  private Map<Integer, UserAccountSimpleDTO> getUserDTOMap(HttpServletRequest request, List<Integer> userIds) {
    List<UserAccountSimpleDTO> userList = userClient.feignListUserAccountByIds(userIds, request.getHeader(TokenConstant.TOKEN));
    Map<Integer, UserAccountSimpleDTO> userMap = new HashMap<>();
    if(CollectionUtils.isNotEmpty(userList)){
      userMap = userList.stream().collect(Collectors.toMap(UserAccountSimpleDTO::getId, d -> d, (k1, k2) -> k1));
    }
    return userMap;
  }

  private ServiceOrderTaskDTO getServiceOrderTaskDTO(Map<Integer, ServiceOrderProvinceCityDO> provinceCityMap, Map<Integer, ServiceOrderTaskImageDO> imageMap, ServiceOrderTaskDO item,Map<Integer, UserAccountSimpleDTO> userMap,Map<Integer, ServiceDTO> serviceDTOMap) {
    ServiceOrderTaskDTO orderTaskDTO = BeanCopyUtils.properties(item, new ServiceOrderTaskDTO());
    // 订单省市信息
    BeanCopyUtils.properties(provinceCityMap.get(item.getId()), orderTaskDTO);
    // 用户信息
    UserAccountSimpleDTO userAccountSimpleDTO = userMap.get(item.getUserAccountId());
    BeanCopyUtils.properties(userAccountSimpleDTO, orderTaskDTO);
    orderTaskDTO.setUserId(userAccountSimpleDTO==null?null:userAccountSimpleDTO.getId());
    ServiceOrderTaskImageDO serviceOrderTaskImageDO = imageMap.get(item.getId());
    // 行业服务信息
    ServiceDTO serviceDTO = serviceDTOMap.get(item.getInspectionId());
    BeanCopyUtils.properties(serviceDTO, orderTaskDTO);
    List<String> images = null;
    if (serviceOrderTaskImageDO!=null &&serviceOrderTaskImageDO.getPicture() !=null) {
      String pictureStr = serviceOrderTaskImageDO.getPicture();
      String[] pictures = pictureStr.split(",");
      images = Arrays.asList(pictures);
    }
    orderTaskDTO.setImages(images);
    orderTaskDTO.setId(item.getId());
    return orderTaskDTO;
  }

  /**
   * 根据id集合批量查询服务作业图片
   */
  private Map<Integer, ServiceOrderTaskImageDO> getImageListMapById(List<Integer> orderTaskIds) {
    LambdaQueryWrapper<ServiceOrderTaskImageDO> imageWrapper = new LambdaQueryWrapper<>();
    imageWrapper.in(ServiceOrderTaskImageDO::getOrderTaskId, orderTaskIds);
    List<ServiceOrderTaskImageDO> imageList = serviceOrderTaskImageDao.selectList(imageWrapper);
    Map<Integer, ServiceOrderTaskImageDO> imageMap = new HashMap<>();
    if (CollectionUtils.isNotEmpty(imageList)) {
      imageMap =
          imageList.stream()
              .collect(Collectors.toMap(ServiceOrderTaskImageDO::getOrderTaskId,d->d,(k1,k2)->k1));
    }
    return imageMap;
  }

  /**
   * 根据id集合批量查询服务作业订单地址
   */
  private Map<Integer, ServiceOrderProvinceCityDO> getOrderProvinceCityDOMapById(List<Integer> orderTaskIds) {
    LambdaQueryWrapper<ServiceOrderProvinceCityDO> provinceCityWrapper = new LambdaQueryWrapper<>();
    provinceCityWrapper.in(ServiceOrderProvinceCityDO::getServiceOrderTaskId, orderTaskIds);
    List<ServiceOrderProvinceCityDO> orderProvinceCityDOList =
        serviceOrderProvinceCityDao.selectList(provinceCityWrapper);
    Map<Integer, ServiceOrderProvinceCityDO> provinceCityMap =
        orderProvinceCityDOList.stream()
            .collect(
                Collectors.toMap(
                    ServiceOrderProvinceCityDO::getServiceOrderTaskId, d -> d, (k1, k2) -> k1));
    return provinceCityMap;
  }
}
