import { useEffect, useState } from 'react';
import SearchBox from '~/components/search-box';
import { Button, message, Modal, Table } from 'antd';
import { PlusOutlined } from '@ant-design/icons';
import { InterListType, InterReqListType } from '~/api/interface';
import { ResourceManageAPI } from '~/api';
import { industryCaseListCasePage } from '~/api/interface/resourceManageType';
import { ColumnsType } from 'antd/es/table';
import AddEditModal from '~/pages/resourceManage/businessCaseManage/comp/addEditModal';
import PreviewModal from '~/pages/resourceManage/businessCaseManage/comp/previewModal';

// 列表类型
type TableType = InterListType<industryCaseListCasePage>;
// 请求的参数
type ReqType = InterReqListType<industryCaseListCasePage>;
// 搜索表单的数据
let query: ReqType = {};

const BusinessCaseManage = () => {
  const { confirm } = Modal;
  // 新增弹窗
  const [addModalVisible, setAddModalVisible] = useState<boolean>(false);
  // 预览弹窗
  const [previewVisible, setPreviewVisible] = useState<boolean>(false);
  // 需要编辑的数据
  const [editData, setEditData] = useState<TableType[0]>();
  // 表格数据
  const [tableData, setTableData] = useState<TableType>([]);
  // 表格分页配置
  const [pagination, setPagination] = useState({
    total: 0,
    pageSize: 10,
    current: 1,
    totalPage: 0,
  });
  // 加载列表
  const getTableList = async (value = {}) => {
    // 只需要修改这个地方的接口即可
    const res = await ResourceManageAPI.industryCaseListCasePage({
      pageNo: pagination.current,
      pageSize: pagination.pageSize,
      ...value,
      ...query,
    });
    if (res && res.code === '200') {
      const { list, pageNo, totalCount, pageSize, totalPage } = res.result; // 解构
      setPagination({
        total: totalCount,
        current: pageNo,
        pageSize,
        totalPage,
      });
      setTableData(list);
    }
  };
  // 翻页
  const paginationChange = (pageNo: number, pageSize: number) => {
    getTableList({ pageNo, pageSize }).then();
  };
  // 表单提交
  const onFinish = (data: ReqType) => {
    pagination.current = 1;
    query = data;
    getTableList(data).then();
  };
  // 删除数据
  const handleDelete = (record: TableType[0]) => {
    confirm({
      title: '提示',
      content: '是否删除该记录？',
      onOk: async () => {
        const res = await ResourceManageAPI.industryCaseDeleteDetails({ id: record.id });
        if (res && res.code === '200') {
          message.success('删除成功');
          paginationChange(
            tableData.length === 1 ? pagination.current - 1 : pagination.current,
            pagination.pageSize,
          );
        }
      },
    });
  };
  // componentDidMount
  useEffect(() => {
    query = {};
    getTableList().then();
  }, []);
  // 表格结构
  const columns: ColumnsType<TableType[0]> = [
    {
      title: '序号',
      dataIndex: 'accountNo',
      align: 'center',
      width: '50px',
      render: (_text, _record, index) => (pagination.current - 1) * pagination.pageSize + index + 1,
    },
    {
      title: '文章名称',
      dataIndex: 'caseTitle',
      align: 'center',
      ellipsis: true,
    },
    {
      title: '发布时间',
      dataIndex: 'updateTime',
      align: 'center',
      render: (text, record) => text || record.createTime,
    },
    {
      title: '来源',
      dataIndex: 'origin',
      align: 'center',
    },
    {
      title: '操作',
      dataIndex: 'action',
      align: 'center',
      width: '150px',
      fixed: 'right',
      render: (_text, record) => (
        <>
          <Button
            type={'link'}
            onClick={() => {
              setEditData(JSON.parse(JSON.stringify(record)));
              setAddModalVisible(true);
            }}
          >
            编辑
          </Button>
          <Button
            type={'link'}
            onClick={() => {
              setEditData(JSON.parse(JSON.stringify(record)));
              setPreviewVisible(true);
            }}
          >
            预览
          </Button>
          <Button type={'link'} danger onClick={() => handleDelete(record)}>
            删除
          </Button>
        </>
      ),
    },
  ];
  return (
    <>
      <SearchBox
        search={[
          { name: 'keyword', label: '文章名称', type: 'input', placeholder: '请输入文章名称' },
        ]}
        child={
          <>
            <Button
              type={'primary'}
              icon={<PlusOutlined />}
              onClick={() => {
                setAddModalVisible(true);
              }}
            >
              新增
            </Button>
          </>
        }
        searchData={onFinish}
      />
      <Table
        size='small'
        dataSource={tableData}
        columns={columns}
        rowKey='id'
        // scroll={{ x: 1500 }}
        bordered
        pagination={{
          total: pagination.total,
          pageSize: pagination.pageSize,
          current: pagination.current,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
      <AddEditModal
        open={addModalVisible}
        title={editData?.id ? '编辑' : '新增'}
        data={editData}
        closed={() => {
          setAddModalVisible(false);
          setEditData(undefined);
          paginationChange(pagination.current, pagination.pageSize);
        }}
      />
      <PreviewModal
        open={previewVisible}
        title={editData?.id ? '预览' : '新增'}
        data={editData}
        closed={() => {
          setPreviewVisible(false);
          setEditData(undefined);
          paginationChange(pagination.current, pagination.pageSize);
        }}
      />
    </>
  );
};

export default BusinessCaseManage;
