package com.mmc.tmj.controller;

import com.mmc.tmj.entity.param.RouteDataDto;
import com.mmc.tmj.entity.result.BaseResponse;
import com.mmc.tmj.entity.result.RouteDetail;
import com.mmc.tmj.entity.result.RouteList;
import com.mmc.tmj.entity.result.SliceRespVo;
import com.mmc.tmj.service.file.impl.MinioStoreageServiceImpl;
import com.mmc.tmj.service.route.RouteService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import javax.annotation.Resource;
import javax.validation.Valid;
import javax.validation.constraints.DecimalMin;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

/**
 * @Author lcz
 * @Date 2022/5/12 9:56
 * @Version 1.0
 * @description
 */
@RequestMapping("route")
@RestController
@Api(tags = "航线")
@Validated
public class RouteController {

    @Resource
    private RouteService routeService;


    @PostMapping
    @ApiOperation("规划航线")
    public BaseResponse planRoute(@Valid @RequestBody @ApiParam(value = "航线规划模型") RouteDataDto routeDataDto){
        if(!routeService.planRoute(routeDataDto)){
           return BaseResponse.error("名称重复",400);
        };
        return BaseResponse.ok();
    }

    @PutMapping
    @ApiOperation("编辑航线")
    public BaseResponse updateRoute(@Valid @RequestBody @ApiParam(value = "航线规划模型") RouteDataDto routeDataDto){
        if(!routeService.updateRoute(routeDataDto)){
           return BaseResponse.error("更新失败",400);
        };
        return BaseResponse.ok();
    }

    @PostMapping("/uploadRoutes")
    @ApiOperation("上传多条航线")
    public BaseResponse uploadRoutes(@RequestParam(value = "files") MultipartFile[] files){
        for(MultipartFile file:files){
            uploadRoute(file);
        }
        return BaseResponse.ok();
    }

    @PostMapping("/upload")
    @ApiOperation("上传航线")
    public BaseResponse uploadRoute(@RequestParam(value = "file") MultipartFile file){
        String substring = file.getOriginalFilename()
            .substring(file.getOriginalFilename().lastIndexOf("."));
        if(".json".equals(substring)){
            routeService.saveUploadRoute(file);
            return BaseResponse.ok();
        }
        if(".kml".equals(substring)){
            throw new IllegalArgumentException("暂不支持KML数据格式");
        }
        return BaseResponse.error("文件:" + file.getOriginalFilename() + "格式错误，上传失败");
    }

    @DeleteMapping("/{id}")
    @ApiOperation("依据ID删除航线")
    public BaseResponse deleteById(@PathVariable @ApiParam(value = "航线Id") @DecimalMin("1") int id){
        if(routeService.deleteByid(id)){
            return BaseResponse.ok("删除成功");
        };
        return BaseResponse.error("删除失败");
    }

    @GetMapping("/routes")
    @ApiOperation("分页查询航线")
    public BaseResponse<SliceRespVo<RouteList>> pages(
        @RequestParam(required = false) @ApiParam(value = "航线名称")String name,
        @RequestParam(defaultValue = "1") @ApiParam(value = "第几页") int num,
        @RequestParam(defaultValue = "10") @ApiParam(value = "每页大小") int size,
        @RequestParam(defaultValue = "-id")
        @ApiParam(value = "排序，例子：id,-code,+name。代表按id升序、code降序、name升序依次排列")
            String sort){
        return BaseResponse.ok(routeService.pages(name,num,size,sort));
    }


    @GetMapping("/{name}")
    @ApiOperation("航线详情")
    public BaseResponse<RouteDetail> routeDetail(@PathVariable String name){
        return BaseResponse.ok(routeService.routeDetail(name));
    }
}
