import { useEffect, useState } from 'react';
import SearchBox from '~/components/search-box';
import { Button, message, Modal, Table, Tag } from 'antd';
import { PlusOutlined } from '@ant-design/icons';
import { InterListType, InterReqListType } from '~/api/interface';
import { ColumnsType } from 'antd/es/table';
import { FlyerCenterAPI } from '~/api';
import AddFlyerModal from '~/pages/flyerManage/flyerTeam/comp/addFlyerModal';
import { selectPilotListByTeamType } from '~/api/interface/flyerCenterType';
import UpdateFlyerSkillModel from 'src/pages/flyerManage/flyerTeam/comp/updateFlyerSkillModal';

// 列表的类型
type TableType = InterListType<selectPilotListByTeamType>;
// 搜索表单的类型
type ReqType = InterReqListType<selectPilotListByTeamType>;
// 搜索表单的数据
let query: ReqType = {};
// 邀请状态列表
const pilotStatusList = [
  { label: '同意', value: 1 },
  { label: '拒绝', value: 2 },
  { label: '待飞手确认', value: 0 },
];

// 组件
const FlyerTeamView = () => {
  // 是否展示添加飞手弹窗
  const [addEditShow, setAddEditShow] = useState(false);
  // 表格数据
  const [tableData, setTableData] = useState<TableType>([]);

  const [flyerSkillShow, setFlyerSkillShow] = useState(false);

  const [flyerInfoItem, setFlyerInfoItem] = useState<TableType[0]>();
  // 表格分页配置
  const [pagination, setPagination] = useState({
    total: 0,
    pageSize: 10,
    current: 1,
    totalPage: 0,
  });
  // +++++++++++++++++++++++++++++++++++++++++++++++++++ //
  // 新版通用部分(ES6+ for React) ZhangLK 2022/08/30 Start
  // 加载列表
  const getTableList = async (value = {}) => {
    // 只需要修改这个地方的接口即可
    const res = await FlyerCenterAPI.selectPilotListByTeam({
      pageNo: pagination.current,
      pageSize: pagination.pageSize,
      ...value,
      ...query,
    });
    if (res && res.code === '200') {
      const { list, pageNo, totalCount, pageSize, totalPage } = res.result; // 解构
      setPagination({
        total: totalCount,
        current: pageNo,
        pageSize,
        totalPage,
      });
      setTableData(list || []);
      // console.log('加载列表 --->', list);
    } else {
      message.warning(res.message);
    }
  };
  // 翻页
  const paginationChange = (pageNo: number, pageSize: number) => {
    getTableList({ pageNo, pageSize }).then();
  };
  // 表单提交
  const onFinish = (data: ReqType) => {
    pagination.current = 1;
    query = data;
    getTableList(data).then();
  };
  // +++++++++++++++++++++++++++++++++++++++++++++++++++ //
  // 删除事件
  const handleDelete = (record: TableType[0]) => {
    Modal.confirm({
      title: '提示',
      content: '请确认是否删除?',
      onOk: async () => {
        const res = await FlyerCenterAPI.deletePilotFromTeam({ pilotId: record?.userAccountId });
        if (res && res.code === '200') {
          message.success('删除成功').then();
          paginationChange(pagination.current, pagination.pageSize);
        }
      },
    });
  };

  //打开修改飞手技能弹窗
  const updatePilotSkill = (record: TableType[0]) => {
    setFlyerInfoItem(record);
    setFlyerSkillShow(true);
  };
  // 页面挂载
  useEffect(() => {
    query = {};
    (async () => {
      await getTableList();
    })();
  }, []);
  // 表格结构
  const columns: ColumnsType<TableType[0]> = [
    // {
    //   title: 'UID',
    //   dataIndex: 'userAccountId',
    //   align: 'center',
    // },
    {
      title: '姓名',
      align: 'center',
      dataIndex: 'userName',
    },
    {
      title: '手机号',
      align: 'center',
      dataIndex: 'phoneNum',
    },
    {
      title: '邀请状态',
      align: 'center',
      dataIndex: 'pilotStatus',
      render: (text) => pilotStatusList.find((i) => i.value === text)?.label || text,
    },
    {
      title: '能力认证',
      align: 'center',
      dataIndex: 'pilotAbility',
      width: '20%',
      render: (text) =>
        text ? text.map((v: any) => <Tag key={v.abilityId}>{v.abilityName}</Tag>) : '',
    },
    {
      title: '常驻城市',
      align: 'center',
      dataIndex: 'residentCity',
    },
    {
      title: '操作',
      align: 'center',
      render: (_text, record) => (
        <>
          <Button type='link' danger onClick={() => handleDelete(record)}>
            删除
          </Button>
          <Button type='link' onClick={() => updatePilotSkill(record)}>
            飞手技能
          </Button>
        </>
      ),
    },
  ];
  return (
    <>
      <SearchBox
        search={[
          {
            label: '邀请状态',
            name: 'pilotStatus',
            type: 'Select',
            placeholder: '请选择邀请状态',
            options: pilotStatusList,
          },
        ]}
        searchData={onFinish}
        child={
          <>
            <Button icon={<PlusOutlined />} type={'primary'} onClick={() => setAddEditShow(true)}>
              添加飞手成员
            </Button>
          </>
        }
      />
      <Table
        size='small'
        dataSource={tableData}
        columns={columns}
        rowKey='id'
        // scroll={{ x: 1000 }}
        bordered
        pagination={{
          total: pagination.total,
          pageSize: pagination.pageSize,
          current: pagination.current,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
      <AddFlyerModal
        open={addEditShow}
        title={'添加飞手'}
        onCancel={() => {
          paginationChange(pagination.current, pagination.pageSize);
          setAddEditShow(false);
        }}
      />
      <UpdateFlyerSkillModel
        open={flyerSkillShow}
        onCancel={() => {
          paginationChange(pagination.current, pagination.pageSize);
          setFlyerSkillShow(false);
        }}
        flyerItem={flyerInfoItem}
      />
    </>
  );
};

export default FlyerTeamView;
