import SearchBox, { searchColumns } from '~/components/search-box';
import { Button, Modal, Table } from 'antd';
import { ColumnsType } from 'antd/es/table';
import { PlusOutlined } from '@ant-design/icons';
import AddOrEditLimitModal from './components/addOrEditLimitModal';
import { useEffect, useState } from 'react';
import { InterDataType, InterReqType, PaginationProps } from '~/api/interface';
import { SystemManageAPI } from '~/api';
import { listRoleInfoPageType } from '~/api/interface/systemManageType';
import { useNavigate } from 'react-router-dom';
import './index.scss';

//账号权限列表-返回类型
type accountLimitType = InterDataType<listRoleInfoPageType>['list'];
//账号权限列表-请求参数类型
type accountLimitParametersType = InterReqType<listRoleInfoPageType>;

const AccountLimit = () => {
  const navigate = useNavigate();
  const searchColumns: searchColumns[] = [
    {
      label: '权限角色',
      placeholder: '请输入角色编号或名称',
      name: 'numberOrName',
      type: 'input',
    },
  ];
  const TableColumns: ColumnsType<accountLimitType[0]> = [
    {
      title: '角色编号',
      align: 'center',
      dataIndex: 'roleNo',
    },
    {
      title: '备注',
      align: 'center',
      dataIndex: 'remark',
      width: '20%',
      ellipsis: true,
      render: (text: string) => <div className='limit-remark'>{text}</div>,
    },
    {
      title: '角色名称',
      dataIndex: 'roleName',
      align: 'center',
    },
    {
      title: '创建人',
      align: 'center',
      dataIndex: 'userName',
    },
    {
      title: '操作',
      align: 'center',
      width: '20%',
      render: (_text: string, record) => (
        <>
          <Button type='link' onClick={() => addOrEditRoleClick(record)}>
            变更
          </Button>
          <Button type='link' onClick={() => toLimitInfo(record)}>
            权限
          </Button>
          <Button
            type='link'
            danger
            onClick={() => deleteAccountLimit(record)}
            disabled={!!record.superAdmin}
          >
            删除
          </Button>
        </>
      ),
    },
  ];
  const [query, setQuery] = useState<accountLimitParametersType>();
  const [tableData, setTableData] = useState<accountLimitType>([]);
  const [currentLimit, setCurrentLimit] = useState<accountLimitType[0]>();
  const [pagination, setPagination] = useState<PaginationProps & { totalCount: number }>({
    pageNo: 1,
    pageSize: 10,
    totalCount: 0,
  });
  const [addOrEditLimitModalShow, setAddOrEditLimitModalShow] = useState<boolean>(false);

  //账号权限列表
  const getListRoleInfoPage = (query?: accountLimitParametersType) => {
    SystemManageAPI.getListRoleInfoPage({
      pageNo: pagination.pageNo,
      pageSize: pagination.pageSize,
      ...query,
    }).then(({ result }) => {
      setTableData(result.list || []);
      pagination.totalCount = result.totalCount;
      setPagination(pagination);
    });
  };
  //分页
  const paginationChange = (pageNo: number, pageSize: number) => {
    pagination.pageNo = pageNo;
    pagination.pageSize = pageSize;
    getListRoleInfoPage(query);
  };
  //筛选
  const searchSuccessEvent = (data: accountLimitParametersType) => {
    setQuery(data);
    pagination.pageSize = 10;
    pagination.pageNo = 1;
    getListRoleInfoPage(data);
  };
  //删除账号权限
  const deleteAccountLimit = (record: accountLimitType[0]) => {
    Modal.confirm({
      title: '角色删除',
      content: '确认删除该角色?',
      onOk: () => {
        SystemManageAPI.deleteRoleInfo({ id: record.id }).then(({ code }) => {
          if (code === '200') {
            if (pagination.pageNo !== 1 && tableData.length === 1) {
              pagination.pageNo -= 1;
            }
            getListRoleInfoPage(query);
          }
        });
      },
    });
  };
  //新增、编辑弹窗
  const addOrEditRoleClick = (record?: accountLimitType[0]) => {
    setAddOrEditLimitModalShow(true);
    setCurrentLimit(record ? { ...record } : undefined);
  };
  const addOrEditLimitModalCancel = () => {
    setAddOrEditLimitModalShow(false);
  };
  const addOrEditLimitModalOk = () => {
    getListRoleInfoPage(query);
    setAddOrEditLimitModalShow(false);
  };
  //跳转权限信息
  const toLimitInfo = (record: accountLimitType[0]) => {
    navigate({ pathname: '/systemManage/limitInfo', search: `id=${record.id}` });
  };

  useEffect(() => {
    getListRoleInfoPage();
  }, []);
  return (
    <div className='account-limit'>
      <SearchBox
        search={searchColumns}
        child={
          <Button type='primary' icon={<PlusOutlined />} onClick={() => addOrEditRoleClick()}>
            添加角色
          </Button>
        }
        searchData={searchSuccessEvent}
      />
      <Table
        bordered
        columns={TableColumns}
        rowKey='id'
        dataSource={tableData}
        pagination={{
          total: pagination.totalCount,
          pageSize: pagination.pageSize,
          current: pagination.pageNo,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
      <AddOrEditLimitModal
        open={addOrEditLimitModalShow}
        onCancel={addOrEditLimitModalCancel}
        onOk={addOrEditLimitModalOk}
        currentLimit={currentLimit}
      />
    </div>
  );
};
export default AccountLimit;
