import React, { FC, useEffect, useState } from 'react';
import { Image, message, Modal, ModalProps, Table } from 'antd';
import { ColumnsType } from 'antd/es/table';
import { ProduceManageAPI } from '~/api';
import { InterDataType, PaginationProps } from '~/api/interface';
import { ProductSpecListType } from '~/api/interface/produceManageType';

//产品sku返回类型
type produceSpecType = InterDataType<ProductSpecListType>['list'];

interface PropsType {
  data: produceSpecType;
  produceSpecSelectEvent: (data: produceSpecType) => void;
  productId: number;
  onCancel: () => void;
}

const DesignateSpecModal: FC<ModalProps & PropsType> = ({
  open,
  onCancel,
  data,
  produceSpecSelectEvent,
  productId,
}) => {
  // 表格多选
  const [selectedRowKeys, setSelectedRowKeys] = useState<React.Key[]>([]);
  // 表格多选事件
  const onSelectChange = (newSelectedRowKeys: React.Key[]) => {
    setSelectedRowKeys(newSelectedRowKeys);
  };
  // 产品sku列表
  const [productSpecList, setProductSpecList] = useState<produceSpecType>([]);
  // 表格结构
  const columns: ColumnsType<{ id: number }> = [
    { title: '规格名称', dataIndex: 'specName', align: 'center' },
    {
      title: '规格图片',
      dataIndex: 'specImage',
      align: 'center',
      render: (record) => {
        return <Image width={50} height={50} src={record.specImage} alt='图片' />;
      },
    },
    { title: '版本描述', dataIndex: 'versionDesc', align: 'center' },
    { title: '料号', dataIndex: 'partNo', align: 'center' },
  ];
  // 表格分页配置
  const [pagination, setPagination] = useState<
    PaginationProps & { totalCount: number; totalPage: number }
  >({
    totalCount: 0,
    pageSize: 5,
    pageNo: 1,
    totalPage: 0,
  });
  // 翻页
  const paginationChange = (pageNo: number, pageSize: number) => {
    pagination.pageNo = pageNo;
    pagination.pageSize = pageSize;
    getProduceSpecList(productId);
  };
  // 提交数据到父组件
  const handleSubmit = () => {
    if (selectedRowKeys.length === 0) {
      return message.warning('请选择产品规格');
    }
    const selectProductSpecList = productSpecList.filter((i) => selectedRowKeys.includes(i.id));
    if (selectProductSpecList.length === 0) {
      return message.warning('请选择产品规格');
    }
    produceSpecSelectEvent(selectProductSpecList);
  };
  //取消
  const handleCancel = () => {
    setSelectedRowKeys([]);
    onCancel();
  };
  //获取产品-sku
  const getProduceSpecList = (productSkuId: number) => {
    ProduceManageAPI.listPageProductSpec({
      productSkuId,
      pageSize: pagination.pageSize,
      pageNo: pagination.pageNo,
    }).then(({ result }) => {
      setProductSpecList(result.list || []);
      pagination.totalPage = result.totalPage;
      pagination.totalCount = result.totalCount;
      setPagination(pagination);
      setSelectedRowKeys(data.map((v) => v.id));
    });
  };

  useEffect(() => {
    if (productId) {
      getProduceSpecList(productId);
    }
  }, [productId]);
  return (
    <div>
      <Modal
        title='指定产品规格'
        open={open}
        onCancel={handleCancel}
        width={768}
        zIndex={1010}
        onOk={handleSubmit}
      >
        <Table
          dataSource={productSpecList}
          columns={columns}
          rowKey='id'
          style={{ width: '100%' }}
          size='small'
          bordered
          rowSelection={{
            selectedRowKeys,
            onChange: onSelectChange,
          }}
          pagination={{
            total: pagination.totalCount,
            pageSize: pagination.pageSize,
            current: pagination.pageNo,
            showSizeChanger: true,
            showQuickJumper: true,
            onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
            showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
          }}
        />
      </Modal>
    </div>
  );
};
export default DesignateSpecModal;
