import SearchBox, { searchColumns } from '~/components/search-box';
import { useEffect, useState } from 'react';
import { Button, message, Modal, Table, Tooltip } from 'antd';
import { ColumnsType } from 'antd/es/table';
import { ResourceManageAPI } from '~/api';
import { InterDataType, InterReqType, PaginationProps } from '~/api/interface';
import './index.scss';
import { listPublishPageType } from '~/api/interface/resourceManageType';
import { useNavigate } from 'react-router-dom';

//需求列表返回类型
type requirementsType = InterDataType<listPublishPageType>['list'];
type requirementsParameter = Omit<InterReqType<listPublishPageType>, 'pageSize' | 'pageNo'>;

const RequirementsGathering = () => {
  const navigate = useNavigate();

  const [searchColumnsData, setSearchColumnsData] = useState<searchColumns[]>([
    {
      label: '需求类型',
      type: 'select',
      placeholder: '请选择需求类型',
      name: 'requirementTypeId',
      options: [],
    },
    {
      label: '需求发起人',
      type: 'input',
      placeholder: '输入姓名/手机号搜索',
      name: 'keyword',
    },
  ]);
  const tableColumn: ColumnsType<requirementsType[0]> = [
    {
      title: '需求类型',
      align: 'center',
      dataIndex: 'requirementTypeName',
    },
    {
      title: '姓名',
      align: 'center',
      dataIndex: 'publishName',
    },
    {
      title: '手机号',
      align: 'center',
      dataIndex: 'publishPhone',
    },
    {
      title: '需求描述',
      align: 'center',
      dataIndex: 'requireDescription',
      width: '20%',
      ellipsis: true,
      render: (text: string) => (
        <Tooltip placement='top' title={text}>
          <div className='require-description'>{text}</div>
        </Tooltip>
      ),
    },
    {
      title: '操作',
      align: 'center',
      render: (_text: string, record) => (
        <>
          <Button type='link' danger onClick={() => deleteRequirements(record)}>
            强制删除
          </Button>
          <Button type='link' onClick={() => toGatherDetail(record)}>
            详情
          </Button>
        </>
      ),
    },
  ];
  const [tableData, setTableData] = useState<requirementsType>([]);
  const [loading, setLoading] = useState<boolean>(false);
  const [pagination, setPagination] = useState<PaginationProps & { totalCount: number }>({
    pageNo: 1,
    pageSize: 10,
    totalCount: 0,
  });
  const [query, setQuery] = useState<requirementsParameter>();

  //筛选
  const searchSuccessEvent = (value: any) => {
    pagination.pageSize = 10;
    pagination.pageNo = 1;
    setQuery(value);
    getRequirementsList(value);
  };
  //分页
  const paginationChange = (pageNo: number, pageSize: number) => {
    pagination.pageNo = pageNo;
    pagination.pageSize = pageSize;
    getRequirementsList(query);
  };
  //需求列表
  const getRequirementsList = (query?: requirementsParameter) => {
    setLoading(true);
    ResourceManageAPI.getListPublishPage({
      pageSize: pagination.pageSize,
      pageNo: pagination.pageNo,
      ...query,
    }).then(({ result }) => {
      setLoading(false);
      setTableData(result.list || []);
      pagination.totalCount = result.totalCount;
      setPagination(pagination);
    });
  };
  //需求类型列表
  const getRequirementsTypeList = () => {
    ResourceManageAPI.getRequirementTypeList().then(({ result }) => {
      searchColumnsData[0].options = result.map((v) => ({ id: v.id, name: v.typeName }));
      setSearchColumnsData([...searchColumnsData]);
    });
  };
  //需求删除
  const deleteRequirements = (record: requirementsType[0]) => {
    Modal.confirm({
      title: '需求删除',
      content: '确认删除该需求？',
      onOk: () => {
        ResourceManageAPI.deleteRequirement({ requirementsInfoId: record.id }).then(({ code }) => {
          if (code === '200') {
            message.success('删除成功');
            if (pagination.pageNo !== 1 && tableData.length === 1) {
              pagination.pageNo -= 1;
            }
            getRequirementsList(query);
          }
        });
      },
    });
  };
  //跳转需求详情
  const toGatherDetail = (record: requirementsType[0]) => {
    navigate({ pathname: '/resourceManage/gatherDetail', search: `id=${record.id}` });
  };

  useEffect(() => {
    getRequirementsList();
    getRequirementsTypeList();
  }, []);
  return (
    <div className='requirements-gathering'>
      <SearchBox search={searchColumnsData} searchData={searchSuccessEvent} />
      <Table
        bordered
        columns={tableColumn}
        dataSource={tableData}
        rowKey='id'
        loading={loading}
        pagination={{
          total: pagination.totalCount,
          pageSize: pagination.pageSize,
          current: pagination.pageNo,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
    </div>
  );
};
export default RequirementsGathering;
