import React, { useEffect } from 'react';
import { DatePicker, Form, InputNumber, message, Modal, ModalProps } from 'antd';
import { InterDataType } from '~/api/interface';
import { getOrderAmountDetailsType } from '~/api/interface/orderManageType';
import Big from 'big.js';
import dayjs from 'dayjs';

// 自定义参数类型
interface SelfProps {
  data?: InterDataType<getOrderAmountDetailsType>[0];
  onSubmit?: (data: InterDataType<getOrderAmountDetailsType>[0]) => void;
  list?: InterDataType<getOrderAmountDetailsType>;
}

const OrderSettleModal: React.FC<ModalProps & SelfProps> = ({
  open,
  title,
  onCancel,
  data,
  onSubmit,
  list,
}) => {
  // 表单钩子
  const [editForm] = Form.useForm();
  // 关闭事件
  const handleCancel = () => {
    editForm.resetFields();
    onCancel?.({} as any);
  };
  // 表单值改变事件
  const handleChange = () => {
    // 计算其他金额的总和
    const money = Object.values({
      ...editForm.getFieldsValue(),
      realWages: 0,
      workDate: 0,
      settleAccountsProportion: 0,
    })?.reduce(
      (a: number, b: any) =>
        Big(a || 0)
          .add(Big(b || 0))
          .toNumber(),
      0,
    );
    // 最后乘以结算比例
    const total = Big(money || 0)
      .mul(Big(editForm.getFieldValue('settleAccountsProportion') || 0))
      .toNumber();
    editForm?.setFieldValue('realWages', total);
  };
  // 提交事件
  const handleSubmit = async () => {
    const valid = await editForm.validateFields();
    if (!valid) return;
    // 判断是否有相同的日期
    const has = list?.some((v) => dayjs(v.workDate).isSame(valid.workDate, 'day'));
    if (has && !data?.id) {
      message.warning('已存在相同的日期').then();
      return;
    }
    // 判断当日工资是否为0
    if (valid.realWages === 0) {
      message.warning('当日工资不能为0').then();
      return;
    }
    onSubmit?.({
      ...data,
      ...valid,
      workDate: dayjs(valid.workDate).format('YYYY-MM-DD'),
    });
    editForm.resetFields();
    // console.log('提交事件 --->', valid);
  };
  // 组件挂载
  useEffect(() => {
    if (!data) return;
    editForm.setFieldsValue({
      ...data,
      workDate: data?.workDate ? dayjs(data?.workDate) : undefined,
    });
  }, [data]);
  return (
    <Modal open={open} onCancel={handleCancel} title={title} onOk={handleSubmit}>
      <Form
        labelCol={{ span: 5 }}
        wrapperCol={{ span: 12 }}
        form={editForm}
        autoComplete={'off'}
        onChange={handleChange}
      >
        <Form.Item
          label='结算日期'
          name='workDate'
          rules={[{ required: true, message: '请输入日期' }]}
        >
          <DatePicker allowClear placeholder='请选择结算日期' style={{ width: '100%' }} />
        </Form.Item>
        <Form.Item
          label='结算标准(元/天)'
          name='dailyWage'
          rules={[
            { required: true, message: '请输入结算标准(元/天)' },
            // 最多两位小数
            { pattern: /^(\d+)?(?:\.\d{1,2})?$/, message: '最多两位小数' },
          ]}
          initialValue={0}
        >
          <InputNumber
            placeholder='请输入结算标准(元/天)'
            maxLength={25}
            min={0}
            max={999999999}
            style={{ width: '100%' }}
          />
        </Form.Item>
        <Form.Item
          label='节日补贴(元/天)'
          name='subsidyFestival'
          rules={[
            { required: true, message: '请输入节日补贴(元/天)' },
            // 最多两位小数
            { pattern: /^(\d+)?(?:\.\d{1,2})?$/, message: '最多两位小数' },
          ]}
          initialValue={0}
        >
          <InputNumber
            placeholder='请输入节日补贴(元/天)'
            maxLength={25}
            min={0}
            max={999999999}
            style={{ width: '100%' }}
          />
        </Form.Item>
        <Form.Item
          label='出差租房补贴'
          name='rentalSubsidy'
          rules={[
            { required: true, message: '请输入出差租房补贴' },
            // 最多两位小数
            { pattern: /^(\d+)?(?:\.\d{1,2})?$/, message: '最多两位小数' },
          ]}
          initialValue={0}
        >
          <InputNumber
            placeholder='请输入出差租房补贴'
            maxLength={25}
            min={0}
            max={999999999}
            style={{ width: '100%' }}
          />
        </Form.Item>
        <Form.Item
          label='交通补贴'
          name='trafficSubsidy'
          rules={[
            { required: true, message: '请输入交通补贴' },
            // 最多两位小数
            { pattern: /^(\d+)?(?:\.\d{1,2})?$/, message: '最多两位小数' },
          ]}
          initialValue={0}
        >
          <InputNumber
            placeholder='请输入交通补贴'
            maxLength={25}
            min={0}
            max={999999999}
            style={{ width: '100%' }}
          />
        </Form.Item>
        <Form.Item
          label='高温补贴'
          name='highTemperatureSubsidy'
          rules={[
            { required: true, message: '请输入高温补贴' },
            // 最多两位小数
            { pattern: /^(\d+)?(?:\.\d{1,2})?$/, message: '最多两位小数' },
          ]}
          initialValue={0}
        >
          <InputNumber
            placeholder='请输入高温补贴'
            maxLength={25}
            min={0}
            max={999999999}
            style={{ width: '100%' }}
          />
        </Form.Item>
        <Form.Item
          label='结算比例'
          name='settleAccountsProportion'
          rules={[
            { required: true, message: '请输入结算比例' },
            // 最多两位小数
            { pattern: /^(\d+)?(?:\.\d{1,2})?$/, message: '最多两位小数' },
          ]}
          initialValue={1}
        >
          <InputNumber
            placeholder='请输入结算比例'
            maxLength={25}
            min={0.01}
            max={999999999}
            style={{ width: '100%' }}
          />
        </Form.Item>
        <Form.Item
          label='应结工资(元)'
          name='realWages'
          rules={[
            { required: true, message: '请输入应结工资(元)' },
            // 最多两位小数
            { pattern: /^(\d+)?(?:\.\d{1,2})?$/, message: '最多两位小数' },
          ]}
          initialValue={0}
        >
          <InputNumber
            placeholder='请输入应结工资(元)'
            maxLength={25}
            min={0}
            max={999999999}
            style={{ width: '100%' }}
            disabled={true}
          />
        </Form.Item>
      </Form>
    </Modal>
  );
};

export default OrderSettleModal;
