import React, { useEffect, useState } from 'react';
import SearchBox from '~/components/search-box';
import { Button, message, Table } from 'antd';
import { ActivityManageAPI } from '~/api';
import { mallOrderListType } from '~/api/interface/activityManage';
import { InterListType, InterReqListType } from '~/api/interface';
import { ColumnsType } from 'antd/es/table';
import AddEditModalView from './comp/addEditModal/index';

// 表格列表
type TableType = InterListType<mallOrderListType>;
// 搜索表单的类型
type ReqType = InterReqListType<mallOrderListType>;
// 搜索表单的数据
let query: ReqType = {};

const PointExchangeListPage: React.FC = () => {
  // 新增编辑弹窗是否开启
  const [addEditShow, setAddEditShow] = useState<boolean>(false);
  // 需要编辑的数据
  const [recordData, setRecordData] = useState<TableType[0]>();
  // 表格数据
  const [tableData, setTableData] = useState<TableType>([]);
  // 表格分页配置
  const [pagination, setPagination] = useState({
    total: 0,
    pageSize: 10,
    current: 1,
    totalPage: 0,
  });
  // +++++++++++++++++++++++++++++++++++++++++++++++++++ //
  // 新版通用部分(ES6+ for React) ZhangLK 2022/08/30 Start
  // 加载列表
  const getTableList = async (value = {}) => {
    // 只需要修改这个地方的接口即可
    const res = await ActivityManageAPI.mallOrderList({
      pageNo: pagination.current,
      pageSize: pagination.pageSize,
      ...value,
      ...query,
      isAdmin: 1,
    });
    if (res && res.code === '200') {
      const { list, pageNo, totalCount, pageSize, totalPage } = res.result; // 解构
      setPagination({
        total: totalCount,
        current: pageNo,
        pageSize,
        totalPage,
      });
      setTableData(list || []);
      // console.log('加载列表 --->', list);
    } else {
      message.warning(res.message);
    }
  };
  // 翻页
  const paginationChange = (pageNo: number, pageSize: number) => {
    getTableList({ pageNo, pageSize }).then();
  };
  // 表单提交
  const onFinish = (data: ReqType) => {
    pagination.current = 1;
    query = data;
    getTableList(data).then();
  };
  // +++++++++++++++++++++++++++++++++++++++++++++++++++ //
  // 页面挂载
  useEffect(() => {
    query = {};
    (async () => {
      await getTableList();
    })();
  }, []);
  // 表格列配置
  const columns: ColumnsType<TableType[0]> = [
    {
      title: '兑换商品',
      dataIndex: 'goods',
      align: 'center',
      width: '150px',
      ellipsis: true,
    },
    {
      title: '收货姓名',
      dataIndex: 'userName',
      align: 'center',
    },
    {
      title: '收货电话',
      dataIndex: 'userPhone',
      align: 'center',
    },
    {
      title: '收货地址',
      dataIndex: 'address',
      align: 'center',
      width: '200px',
      ellipsis: true,
    },
    {
      title: '是否发货',
      dataIndex: 'sendStatus',
      align: 'center',
      render: (text) => (text !== 1 ? '未发货' : '已发货'),
    },
    {
      title: '下单时间',
      dataIndex: 'createTime',
      align: 'center',
    },
    {
      title: '发货运营',
      dataIndex: 'sendUserName',
      align: 'center',
    },
    {
      title: '操作',
      dataIndex: 'id',
      align: 'center',
      render: (_text, record) => (
        <>
          <Button
            type='link'
            disabled={record.sendStatus === 1}
            onClick={() => {
              setRecordData(record);
              setAddEditShow(true);
            }}
          >
            发货
          </Button>
        </>
      ),
    },
  ];
  return (
    <>
      <SearchBox
        search={[
          {
            label: '收货姓名',
            name: 'userName',
            type: 'Input',
            placeholder: '请输入收货姓名',
            maxlength: 20,
          },
          {
            label: '收货人电话',
            name: 'userPhone',
            type: 'Input',
            placeholder: '请输入收货人电话',
            maxlength: 20,
          },
        ]}
        isSearch={true}
        isReset={true}
        searchData={onFinish}
      />
      <Table
        size='small'
        dataSource={tableData}
        columns={columns}
        rowKey='id'
        // scroll={{ x: 1000 }}
        bordered
        pagination={{
          total: pagination.total,
          pageSize: pagination.pageSize,
          current: pagination.current,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
      <AddEditModalView
        open={addEditShow}
        title={'发货'}
        data={recordData}
        onCancel={() => {
          setAddEditShow(false);
          setRecordData(undefined);
          paginationChange(pagination?.current, pagination?.pageSize);
        }}
      />
    </>
  );
};

export default PointExchangeListPage;
