import { useSearchParams, useNavigate } from 'react-router-dom';
import { useEffect, useState } from 'react';
import GoodsAPI from '~/api/modules/goodsAPI';
import { InterDataType } from '~/api/interface';
import { mallGoodsDetailsType, skuUnitType } from '~/api/interface/goodsType';
import { Badge, Button, Descriptions, Image, Table, Tag } from 'antd';
import { CategoryManageAPI } from '~/api';
import { categoryListRespType } from '~/api/interface/categoryManage';
import './index.scss';
import { ColumnsType } from 'antd/es/table';

//详情返回类型
type detailType = InterDataType<mallGoodsDetailsType>;
//分类返回类型
type categoryType = InterDataType<categoryListRespType>['list'];
//单位返回类型
type unitType = InterDataType<skuUnitType>;

const GoodsDetails = () => {
  const [searchParams] = useSearchParams();
  const navigate = useNavigate();
  //分类列表
  const [categoryList, setCategoryList] = useState<categoryType>([]);
  //单位列表
  const [skuUnitList, setSkuUnitList] = useState<unitType>([]);
  const [goodsInfoDetails, setGoodsInfoDetails] = useState<detailType>();
  const tableColumns: ColumnsType<detailType['goodsSpecList'][0]> = [
    {
      title: '序号',
      align: 'center',
      render: (_text: string, _record, index: number) => index + 1,
    },
    {
      title: '规格名称',
      align: 'center',
      dataIndex: 'specName',
    },
    {
      title: '选择方式',
      align: 'center',
      dataIndex: 'chooseType',
      render: (text: number) => (text ? '多选' : '单选'),
    },
    {
      title: '是否必选',
      align: 'center',
      dataIndex: 'must',
      render: (text: number) => (text ? '必选' : '非必选'),
    },
    {
      title: '规格单位',
      align: 'center',
      dataIndex: 'skuUnitId',
      render: (text: number) => skuUnitList.find((v) => v.id === text)?.unitName || '',
    },
    {
      title: '规格值',
      align: 'center',
      dataIndex: 'goodsSpecValuesList',
      render: (text: detailType['goodsSpecList'][0]['goodsSpecValuesList']) =>
        text.map((v) => (
          <Tag key={v.id}>
            {v.specValueName}
            {v.partNo ? `(${v.partNo})` : ''}
          </Tag>
        )),
    },
  ];
  const getGoodsDetails = (id: number) => {
    GoodsAPI.getMallGoodsDetails({ id }).then(({ result }) => {
      setGoodsInfoDetails(result);
    });
  };
  //分类列表
  const getCategoryList = () => {
    CategoryManageAPI.getCategoryRespList({ pageNo: 1, pageSize: 99999 }).then(({ result }) => {
      setCategoryList(result.list || []);
    });
  };
  //单位列表
  const getSkuUnit = () => {
    GoodsAPI.getSkuUnit().then(({ result }) => {
      setSkuUnitList(result || []);
    });
  };
  //返回
  const backRoute = () => {
    navigate(-1);
  };

  useEffect(() => {
    getGoodsDetails(Number(searchParams.get('id')));
    getCategoryList();
    getSkuUnit();
  }, []);
  return (
    <div className='goods-detail'>
      <div className='goods-detail-info'>
        <Descriptions title='基本信息' bordered>
          <Descriptions.Item label='商品名称'>{goodsInfoDetails?.tradeName}</Descriptions.Item>
          <Descriptions.Item label='商品副图'>
            {goodsInfoDetails?.resourcesList
              .filter((v) => v.type === 1)
              .map((v) => (
                <Image src={v.url} width={50} height={50} key={v.id} />
              ))}
          </Descriptions.Item>
          <Descriptions.Item label='商品主图'>
            {goodsInfoDetails?.resourcesList
              .filter((v) => v.type === 0)
              .map((v) => (
                <Image src={v.url} width={50} height={50} key={v.id} />
              ))}
          </Descriptions.Item>
          <Descriptions.Item label='商品标签'>
            {goodsInfoDetails?.goodsLabel || '暂无'}
          </Descriptions.Item>
          <Descriptions.Item label='商品分类'>
            {categoryList
              .reduce((pre: string[], cur) => {
                if (cur.id === goodsInfoDetails?.categoryPrimaryId) {
                  pre.push(cur.name);
                  if (goodsInfoDetails?.categorySubId) {
                    pre.push(
                      cur.subDTOList.find((v) => v.id === goodsInfoDetails?.categorySubId)?.name ||
                        '',
                    );
                  }
                }
                return pre;
              }, [])
              .join('/')}
          </Descriptions.Item>
          <Descriptions.Item label='商品状态'>
            <Badge
              status={goodsInfoDetails?.shelfStatus ? 'processing' : 'default'}
              text={goodsInfoDetails?.shelfStatus ? '上架中' : '已下架'}
            />
          </Descriptions.Item>
          <Descriptions.Item label='商品视频' span={1}>
            {goodsInfoDetails?.resourcesList
              .filter((v) => v.type === 2)
              .map((v) => (
                <video
                  src={v.url}
                  key={v.id}
                  style={{ width: '200px', height: '200px' }}
                  controls
                />
              ))}
          </Descriptions.Item>
          <Descriptions.Item label='商品描述'>{goodsInfoDetails?.description}</Descriptions.Item>
        </Descriptions>
      </div>
      <div className='goods-detail-sku'>
        <div className='sku-title'>商品规格</div>
        <Table
          bordered
          columns={tableColumns}
          dataSource={goodsInfoDetails?.goodsSpecList}
          rowKey='id'
          pagination={false}
        ></Table>
      </div>
      <div className='goods-detail-introduce'>
        <div className='introduce-title'>商品详情</div>
        <div
          className='introduce-content'
          dangerouslySetInnerHTML={{ __html: goodsInfoDetails?.goodsDetails || '' }}
        ></div>
      </div>
      <div className='goods-detail-operate'>
        <Button type='primary' onClick={backRoute}>
          返回
        </Button>
      </div>
    </div>
  );
};
export default GoodsDetails;
