import { Button, Popconfirm, Table, Tag } from 'antd';
import { PlusOutlined } from '@ant-design/icons';
import { ColumnsType } from 'antd/es/table';
import { FC, useState } from 'react';
import { InterDataType, InterReqType, PaginationProps } from '~/api/interface';
import { addMallGoodsType, skuUnitType } from '~/api/interface/goodsType';

//商品sku规格类型
type goodsSpecType = Exclude<InterReqType<addMallGoodsType>, undefined>['goodsSpecList'][0];
//单位返回类型
type unitType = InterDataType<skuUnitType>;
interface selfProps {
  addOrEditSkuClick: () => void;
  skuTableData: goodsSpecType[];
  skuUnitList: unitType;
  deleteSkuClick: (record: goodsSpecType) => void;
  editSkuClick: (record: goodsSpecType) => void;
}

const SkuInfo: FC<selfProps> = ({
  addOrEditSkuClick,
  skuTableData,
  skuUnitList,
  deleteSkuClick,
  editSkuClick,
}) => {
  const tableColumns: ColumnsType<goodsSpecType> = [
    {
      title: '序号',
      align: 'center',
      render: (_text: string, _record, index: number) => index + 1,
    },
    {
      title: '规格名称',
      align: 'center',
      dataIndex: 'specName',
    },
    {
      title: '选择方式',
      align: 'center',
      dataIndex: 'chooseType',
      render: (text: number) => (text ? '多选' : '单选'),
    },
    {
      title: '是否必选',
      align: 'center',
      dataIndex: 'must',
      render: (text: number) => (text ? '必选' : '非必选'),
    },
    {
      title: '规格单位',
      align: 'center',
      dataIndex: 'skuUnitId',
      render: (text: number) => skuUnitList.find((v) => v.id === text)?.unitName || '',
    },
    {
      title: '规格值',
      align: 'center',
      dataIndex: 'goodsSpecValuesList',
      render: (text: goodsSpecType['goodsSpecValuesList']) =>
        text.map((v) => (
          <Tag key={v.id}>
            {v.specValueName}
            {v.partNo ? `(${v.partNo})` : ''}
          </Tag>
        )),
    },
    {
      title: '操作',
      align: 'center',
      render: (_text: string, record) => (
        <>
          <Button type='link' onClick={() => editSkuClick(record)}>
            编辑
          </Button>
          <Popconfirm
            placement='topLeft'
            title={'删除规格'}
            description={'确认删除该规格吗？'}
            onConfirm={() => deleteSkuClick(record)}
            okText='确定'
            cancelText='取消'
          >
            <Button type='link' danger>
              删除
            </Button>
          </Popconfirm>
        </>
      ),
    },
  ];
  const [pagination, setPagination] = useState<PaginationProps>({
    pageNo: 1,
    pageSize: 10,
  });
  //分页
  const paginationChange = (pageNo: number, pageSize: number) => {
    pagination.pageNo = pageNo;
    pagination.pageSize = pageSize;
    setPagination({ ...pagination });
  };

  return (
    <div className='sku-info'>
      <div className='sku-info-operate' style={{ margin: ' 20px 0 ' }}>
        <Button
          type='primary'
          icon={<PlusOutlined></PlusOutlined>}
          onClick={() => addOrEditSkuClick()}
        >
          添加规格
        </Button>
      </div>
      <Table
        bordered
        columns={tableColumns}
        dataSource={skuTableData.slice(
          (pagination.pageNo - 1) * pagination.pageSize,
          pagination.pageNo * pagination.pageSize,
        )}
        rowKey='id'
        pagination={{
          total: skuTableData.length,
          pageSize: pagination.pageSize,
          current: pagination.pageNo,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
    </div>
  );
};
export default SkuInfo;
