import { useEffect, useState } from 'react';
import SearchBox from '~/components/search-box';
import { Button, message, Modal, Table } from 'antd';
import { PlusOutlined } from '@ant-design/icons';
import { InterListType, PaginationProps } from '~/api/interface';
import { selectAllSystemMessage } from '~/api/interface/resourceManageType';
import { ResourceManageAPI } from '~/api';
import { ColumnsType } from 'antd/es/table';
import AddEditModalView from './comp/addEditModal';

// 列表类型
type ListType = InterListType<selectAllSystemMessage>;

const NoticeManageView = () => {
  // 新增弹窗是否开启
  const [addEditModalVisible, setAddEditModalVisible] = useState(false);
  // 表格数据
  const [tableData, setTableData] = useState<ListType>([]);
  // 翻页数据
  const [pagination, setPagination] = useState<PaginationProps & { totalCount: number }>({
    pageNo: 1,
    pageSize: 10,
    totalCount: 0,
  });
  // 加载列表
  const getTableList = async (value = {}) => {
    // 只需要修改这个地方的接口即可
    const res = await ResourceManageAPI.selectAllSystemMessage({
      pageNo: pagination.pageNo,
      pageSize: pagination.pageSize,
      ...value,
    });
    if (res && res.code === '200') {
      const { list, pageNo, totalCount, pageSize } = res.result; // 解构
      setPagination({
        totalCount,
        pageNo,
        pageSize,
      });
      setTableData(list);
    }
  };
  // 翻页
  const paginationChange = (pageNo: number, pageSize: number) => {
    getTableList({ pageNo, pageSize }).then();
  };
  // 删除事件
  const handleDelete = (record: ListType[0]) => {
    Modal.confirm({
      title: '提示',
      content: '确定删除该通知吗？',
      onOk: async () => {
        const res = await ResourceManageAPI.removeMessage({
          id: record?.id,
        });
        if (res && res.code === '200') {
          message.success('操作成功');
          paginationChange(pagination.pageNo, pagination.pageSize);
        }
      },
    });
  };
  // 组件挂载
  useEffect(() => {
    getTableList().then();
  }, []);
  // 表格列表
  const columns: ColumnsType<ListType[0]> = [
    {
      title: '序号',
      dataIndex: 'accountNo',
      align: 'center',
      render: (_text, _record, index) => (pagination.pageNo - 1) * pagination.pageSize + index + 1,
    },
    {
      title: '标题',
      dataIndex: 'subject',
      align: 'center',
    },
    {
      title: '内容',
      dataIndex: 'content',
      align: 'center',
      render: (text) => text?.replace(/<[^>]*>/g, ''),
      ellipsis: true,
      // render: (text) => <div dangerouslySetInnerHTML={{ __html: text }} />,
    },
    {
      title: '发布时间',
      dataIndex: 'createTime',
      align: 'center',
      width: '200px',
    },
    {
      title: '操作',
      dataIndex: 'action',
      align: 'center',
      width: '200px',
      render: (_text, record) => (
        <>
          <Button type={'link'} disabled>
            编辑
          </Button>
          <Button type={'link'} danger onClick={() => handleDelete(record)}>
            删除
          </Button>
        </>
      ),
    },
  ];
  return (
    <>
      <SearchBox
        preFixBtn={
          <Button
            icon={<PlusOutlined />}
            type={'primary'}
            onClick={() => setAddEditModalVisible(true)}
          >
            新增通知
          </Button>
        }
      />
      <Table
        size='small'
        dataSource={tableData}
        columns={columns}
        rowKey='id'
        // scroll={{ x: 1500 }}
        bordered
        pagination={{
          total: pagination.totalCount,
          pageSize: pagination.pageSize,
          current: pagination.pageNo,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
      <AddEditModalView
        title={'新增通知'}
        open={addEditModalVisible}
        onCancel={() => {
          setAddEditModalVisible(false);
          paginationChange(pagination.pageNo, pagination.pageSize);
        }}
      />
    </>
  );
};

export default NoticeManageView;
