import { Form, Input, message, Modal, ModalProps, Select } from 'antd';
import { FC, useEffect, useState } from 'react';
import { Uploader } from '~/components/uploader';
import { UploadOutlined } from '@ant-design/icons';
import { InterListType, InterReqType } from '~/api/interface';
import { addGambitType, listGambitType } from '~/api/interface/forumManageType';
import { gambitPropertyList } from '~/utils/dictionary';
import { ForumManageAPI } from '~/api';

// 新增话题参数类型
type addTopicParams = Exclude<InterReqType<addGambitType>, undefined>;
// 话题列表返回类型
type topicListType = InterListType<listGambitType>;
interface selfProps {
  onOk: () => void;
  onCancel: () => void;
  currentTopicItem: topicListType[0] | undefined;
}

const AddTopicModal: FC<ModalProps & selfProps> = ({ open, onCancel, onOk, currentTopicItem }) => {
  const [form] = Form.useForm<addTopicParams>();
  const [gambitIconFileList, setGambitIconFileList] = useState<any>([]);
  const [gambitCoverFileList, setGambitCoverFileList] = useState<any>([]);

  const handleOk = () => {
    form.validateFields().then((values) => {
      ForumManageAPI[currentTopicItem ? 'updateGambit' : 'addGambit']({
        ...values,
        gambitName: '#' + values.gambitName,
        id: currentTopicItem ? currentTopicItem.id : undefined,
      }).then(({ code }) => {
        if (code === '200') {
          message.success(currentTopicItem ? '编辑成功' : '新增成功');
          form.resetFields();
          setGambitIconFileList([]);
          setGambitCoverFileList([]);
          onOk();
        }
      });
    });
  };
  const handleCancel = () => {
    form.resetFields();
    setGambitIconFileList([]);
    setGambitCoverFileList([]);
    onCancel();
  };
  // 图标上传成功
  const gambitIconUploadSuccess = (value: any) => {
    setGambitIconFileList(value);
    form.setFieldValue('gambitIcon', value.length ? value[0].url : undefined);
  };
  // 封面上传成功
  const gambitCoverUploadSuccess = (value: any) => {
    setGambitCoverFileList(value);
    form.setFieldValue('gambitCover', value.length ? value[0].url : undefined);
  };

  useEffect(() => {
    if (currentTopicItem) {
      form.setFieldsValue({
        gambitCover: currentTopicItem.gambitCover || undefined,
        gambitIcon: currentTopicItem.gambitIcon || undefined,
        gambitName: currentTopicItem.gambitName,
        gambitProperty: currentTopicItem.gambitProperty,
      });
      setGambitIconFileList(
        currentTopicItem.gambitIcon
          ? [
              {
                id: Math.random(),
                uid: Math.random(),
                name: 'icon',
                url: currentTopicItem.gambitIcon,
              },
            ]
          : [],
      );
      setGambitCoverFileList(
        currentTopicItem.gambitCover
          ? [
              {
                id: Math.random(),
                uid: Math.random(),
                name: 'icon',
                url: currentTopicItem.gambitCover,
              },
            ]
          : [],
      );
    }
  }, [currentTopicItem]);
  return (
    <Modal title='新增话题' open={open} onOk={handleOk} onCancel={handleCancel}>
      <Form labelCol={{ span: 4 }} wrapperCol={{ span: 15 }} form={form}>
        <Form.Item
          label='名称'
          name='gambitName'
          rules={[{ required: true, message: '请输入名称' }]}
        >
          <Input placeholder='请输入名称' maxLength={30} prefix='#' />
        </Form.Item>
        <Form.Item
          label='图标'
          name='gambitIcon'
          rules={[{ required: true, message: '请上传图标' }]}
        >
          <Uploader
            fileUpload
            listType='picture-card'
            fileLength={1}
            onChange={gambitIconUploadSuccess}
            defaultFileList={gambitIconFileList}
          >
            <UploadOutlined />
          </Uploader>
        </Form.Item>
        <Form.Item
          label='封面'
          name='gambitCover'
          rules={[{ required: true, message: '请上传封面' }]}
        >
          <Uploader
            fileUpload
            listType='picture-card'
            fileLength={1}
            onChange={gambitCoverUploadSuccess}
            defaultFileList={gambitCoverFileList}
          >
            <UploadOutlined />
          </Uploader>
        </Form.Item>
        <Form.Item
          label='属性'
          name='gambitProperty'
          rules={[{ required: true, message: '请选择属性' }]}
        >
          <Select placeholder='请选择属性' options={gambitPropertyList}></Select>
        </Form.Item>
      </Form>
    </Modal>
  );
};
export default AddTopicModal;
