import { Table } from 'antd';
import { forwardRef, useEffect, useImperativeHandle, useRef, useState } from 'react';
import { isEmptyBol, regPriceNumber } from '~/utils/validateUtils';
import { InterDataType, InterReqType } from '~/api/interface';
import { addMallGoodsType, mallGoodsDetailsType } from '~/api/interface/goodsType';
import { filterObjAttr } from '~/utils';
import CommonSkuInfo from '~/components/goods/commonSkuInfo';

type EditableTableProps = Parameters<typeof Table>[0];
type ColumnTypes = Exclude<EditableTableProps['columns'], undefined>;

//规格表单数据类型
type specificationFormListType = {
  optionList: { label: string; value: string }[];
  id: number;
  name: string;
  addSpecificationValueShow: boolean;
  specificationValueList: { name: string; id: number; specificationName: string }[];
};
//规格表格类型
type skuTableType = Exclude<InterReqType<addMallGoodsType>, undefined>['priceStock'][0] & {
  fileList: {
    id: number;
    name: string;
    uid: number;
    url: string;
  }[];
};
//商品详情-返回类型
type goodsDetailType = InterDataType<mallGoodsDetailsType>;

interface selfProps {
  ref: any;
  goodsDetailsInfo: goodsDetailType | undefined;
}

const SkuInfo = forwardRef<any, selfProps>(({ goodsDetailsInfo }, ref) => {
  const commonSkuInfoRef = useRef<any>();

  //表格数据
  const [tableData, setTableData] = useState<(skuTableType & { [key: string]: string })[]>([]);
  //销售价格正则校验
  const salePriceValidator = (_rules: any, value: number) => {
    if (!isEmptyBol(value)) {
      if (regPriceNumber(value.toString())) {
        if (value > 99999999 || value < 0) {
          return Promise.reject(new Error('价格最大为99999999且大于0'));
        }
        return Promise.resolve();
      } else {
        return Promise.reject(new Error('为整数且最多保留两位小数'));
      }
    } else {
      return Promise.reject(new Error('请输入销售价'));
    }
  };
  //渠道正则价格校验
  const channelPriceValidator = (_rules: any, value: number) => {
    if (!isEmptyBol(value)) {
      if (regPriceNumber(value.toString())) {
        if (value > 99999999 || value < 0) {
          return Promise.reject(new Error('价格最大为99999999且大于0'));
        }
        return Promise.resolve();
      } else {
        return Promise.reject(new Error('为整数且最多保留两位小数'));
      }
    } else {
      return Promise.resolve();
    }
  };
  //库存正则校验
  const stockPriceValidator = (_rules: any, value: number) => {
    if (!isEmptyBol(value)) {
      if (/^[+]{0,1}(\d+)$/.test(value.toString())) {
        if (value > 99999999 || value < 0) {
          return Promise.reject(new Error('库存最大为99999999且大于0'));
        }
        return Promise.resolve();
      } else {
        return Promise.reject(new Error('请输入正整数'));
      }
    } else {
      return Promise.resolve();
    }
  };
  //表格-列
  const [defaultColumns, setDefaultColumns] = useState<
    (ColumnTypes[number] & {
      editable?: boolean;
      dataIndex?: string;
      inputType?: string;
      radioOption?: { name: string; id: number }[];
      rules?: any;
      maxLength?: number;
      children?: any;
      placeholder?: string;
    })[]
  >([
    {
      title: '商品规格',
      align: 'center',
      children: [],
    },
    {
      title: '图片',
      align: 'center',
      editable: true,
      dataIndex: 'skuImage',
      inputType: 'uploader',
    },
    {
      title: 'sku编号（料号）',
      align: 'center',
      editable: true,
      dataIndex: 'skuNo',
      maxLength: 30,
      width: '15%',
    },
    {
      title: (
        <div>
          <span style={{ color: 'red' }}>*</span>
          <span>销售价</span>
        </div>
      ),
      align: 'center',
      editable: true,
      dataIndex: 'salePrice',
      rules: [{ required: true, validator: salePriceValidator }],
      inputType: 'number',
      onHeaderCell: () => ({ className: 'custom-header-cell' }),
      placeholder: '销售价',
    },
    {
      title: '渠道价',
      editable: true,
      align: 'center',
      dataIndex: 'channelPrice',
      rules: [{ required: false, validator: channelPriceValidator }],
      inputType: 'number',
    },
    {
      title: '库存',
      editable: true,
      align: 'center',
      dataIndex: 'stock',
      rules: [{ required: false, validator: stockPriceValidator }],
      inputType: 'number',
    },
  ]);

  useImperativeHandle(ref, () => ({
    submitSku,
    getForm: () => commonSkuInfoRef.current.getSpecificationValueForm(),
  }));

  //表头拆分及合并列
  const mergeTableRow = (filterSpecificationFormList: specificationFormListType[]) => {
    const columns = filterSpecificationFormList.map((v, index) => ({
      title: v.optionList[0].value,
      dataIndex: 'name' + (index + 1),
      align: 'center',
      onCell: (_: any, i: number) => {
        //合并列
        if (index < filterSpecificationFormList.length - 1) {
          const count: number = filterSpecificationFormList
            .slice(index + 1, filterSpecificationFormList.length)
            .reduce((pre: number, cur) => {
              return pre * cur.specificationValueList.length;
            }, 1);
          return {
            rowSpan: count !== 1 ? ((i + 1) % count === 1 ? count : 0) : 1,
          };
        } else {
          return {
            rowSpan: 1,
          };
        }
      },
    }));
    defaultColumns[0].children = columns;
    setDefaultColumns([...defaultColumns]);
  };

  //规格值上传图片返回
  const uploadSuccess = (
    record: skuTableType,
    fileList: {
      id: number;
      name: string;
      uid: number;
      url: string;
    }[],
  ) => {
    const tableIndex: number = tableData.findIndex((v) => v.id === record.id);
    if (tableIndex !== -1) {
      tableData[tableIndex].fileList = fileList;
      const obj = Object.create(null);
      obj['skuImage' + record.id] = fileList.length ? fileList[0].url : undefined;
      commonSkuInfoRef.current.getSpecificationValueForm().setFieldsValue(obj);
      setTableData([...tableData]);
    }
  };

  //提交验证
  const submitSku = () => {
    return new Promise((resolve, reject) => {
      commonSkuInfoRef.current
        .getSpecificationForm()
        .validateFields()
        .then(() => {
          const specificationFormList: specificationFormListType[] =
            commonSkuInfoRef.current.getSpecificationFormList();
          const specificationFormItem = specificationFormList.find(
            (v) => !v.specificationValueList.length,
          );
          if (specificationFormItem) {
            reject(`请为规格项${specificationFormItem.optionList[0].value}添加规格值`);
          } else {
            //规格项数据转化
            const specAttrList = specificationFormList.map((v) => ({
              specName: v.optionList[0].value,
              id: goodsDetailsInfo
                ? goodsDetailsInfo.specAttrList.find((i) => i.id === v.id)?.id
                : undefined,
              specValuesList: v.specificationValueList.map((j) => ({
                specName: j.name,
                id: goodsDetailsInfo
                  ? goodsDetailsInfo.specAttrList
                      .find((i) => i.id === v.id)
                      ?.specValuesList.find((i) => i.id === j.id)?.id
                  : undefined,
              })),
            }));
            commonSkuInfoRef.current
              .getSpecificationValueForm()
              .validateFields()
              .then((value: any) => {
                //规格值数据转化
                const priceStock = tableData.reduce((pre: any, cur: any) => {
                  //规格名，规格值组合类型
                  const productSpec = specificationFormList.reduce(
                    (a: { [x: string]: string }, b, currentIndex) => {
                      a[b.optionList[0].value] = cur['name' + (currentIndex + 1)];
                      return a;
                    },
                    {},
                  );
                  pre.push({
                    ...Object.getOwnPropertyNames(value).reduce((a: any, b) => {
                      if (b.includes(cur.id)) {
                        a[b.replace(cur.id, '')] = value[b];
                      }
                      return a;
                    }, {}),
                    productSpec: JSON.stringify(productSpec),
                  });
                  return pre;
                }, []);
                resolve({ priceStock, specAttrList });
              })
              .catch((err: any) => {
                reject(err);
              });
          }
        })
        .catch((err: any) => {
          reject(err);
        });
    });
  };
  //排列组合规格值表单默认数据
  const setTableFormDefault = (tableDataList: (skuTableType & { [key: string]: string })[]) => {
    const tableFormDefault = tableDataList.reduce((pre: any, cur) => {
      return {
        ...pre,
        ...Object.getOwnPropertyNames(filterObjAttr(cur, ['id'])).reduce((a: any, b) => {
          a[b + cur.id] = cur[b];
          return a;
        }, {}),
      };
    }, {});
    commonSkuInfoRef.current.getSpecificationValueForm().setFieldsValue(tableFormDefault);
  };
  //更新表单
  const updateTableData = (tableData: any) => {
    setTableData([...tableData]);
  };
  //更新表单列
  const updateDefaultColumns = (columns: any) => {
    setDefaultColumns([...columns]);
  };

  useEffect(() => {
    if (goodsDetailsInfo) {
      //转化数据
      const covertSpecAttrList = goodsDetailsInfo.specAttrList.map((v, index) => ({
        id: v.id,
        name: 'specName' + index,
        optionList: [{ label: v.specName, value: v.specName }],
        specificationValueList: v.specValuesList.map((i) => ({
          id: i.id,
          name: i.specName,
          specificationName: v.specName,
        })),
        addSpecificationValueShow: false,
      }));
      //规格项表单数据默认数据
      const specFormDefault = goodsDetailsInfo.specAttrList.reduce(
        (pre: any, cur: any, currentIndex) => {
          pre['specName' + currentIndex] = cur.specName;
          return pre;
        },
        {},
      );
      commonSkuInfoRef.current.getSpecificationForm().setFieldsValue(specFormDefault);
      commonSkuInfoRef.current.updateSpecificationFormList([...covertSpecAttrList]);
      mergeTableRow(covertSpecAttrList);
      const tableDataList: (skuTableType & { [key: string]: string })[] =
        goodsDetailsInfo.priceStock.map((v) => ({
          id: v.id,
          salePrice: v.salePrice,
          skuImage: v.skuImage,
          skuNo: v.skuNo,
          stock: v.stock,
          channelPrice: v.channelPrice,
          fileList: v.skuImage
            ? [{ id: Math.random(), uid: Math.random(), url: v.skuImage, name: '规格图片' }]
            : [],
          ...Object.getOwnPropertyNames(JSON.parse(v.productSpec)).reduce(
            (pre: any, cur, currentIndex) => {
              pre['name' + (currentIndex + 1)] = JSON.parse(v.productSpec)[cur];
              pre['specificationName' + (currentIndex + 1)] = cur;
              return pre;
            },
            {},
          ),
        }));
      setTableFormDefault(tableDataList);
      setTableData(tableDataList);
    }
  }, [goodsDetailsInfo]);

  return (
    <div className='sku-info'>
      <CommonSkuInfo
        tableData={tableData}
        updateTableData={updateTableData}
        defaultColumns={defaultColumns}
        updateDefaultColumns={updateDefaultColumns}
        uploadSuccess={uploadSuccess}
        ref={commonSkuInfoRef}
      />
    </div>
  );
});
export default SkuInfo;
