import { InterListType, InterReqType } from '~/api/interface';
import { listPcWechatOrderType, pfConfirmOrderWareType } from '~/api/interface/orderManageType';
import React, { useEffect } from 'react';
import { Col, Form, Input, message, Modal, Radio, Row } from 'antd';
import { Uploader } from '~/components/uploader';
import { PlusOutlined } from '@ant-design/icons';
import { OrderManageAPI } from '~/api';

// 表格数据类型
type TableType = InterListType<listPcWechatOrderType>;
// 请求数据的类型
type ReqType = InterReqType<pfConfirmOrderWareType>;

// 传参类型
interface propType {
  title: string;
  open: boolean;
  closed: any;
  data?: TableType[0];
}

const OrderConfirm: React.FC<propType> = (props) => {
  OrderConfirm.defaultProps = {
    data: undefined,
  };
  // 参数
  const { title, open, closed, data } = props;
  // 物流列表
  const [expressList, setExpressList] = React.useState<{ label: string; value: string }[]>([]);
  // 表单钩子
  const [form] = Form.useForm<ReqType>();
  // 关闭弹窗
  const handleCancel = () => {
    form.resetFields();
    closed();
  };
  // 确认事件
  const handleOk = () => {
    form
      .validateFields()
      .then(async (values) => {
        // console.log('确认事件 --->', values);
        await handleSubmit(values);
      })
      .catch((err) => {
        message
          .warning({
            content: err.errorFields[0].errors[0],
          })
          .then();
      });
  };
  // 提交事件
  const handleSubmit = async (values: ReqType) => {
    const res = await OrderManageAPI.pfConfirmOrderWare({
      ...values,
      orderInfoId: Number(data?.id),
    });
    if (res && res.code === '200') {
      message.success('操作成功');
      handleCancel();
    }
  };
  // 获取物流信息
  const getListExpressInfo = async () => {
    const res = await OrderManageAPI.listExpressInfo({});
    if (res && res.code === '200') {
      setExpressList(res.result.map((item) => ({ label: item.exName, value: item.exCode })));
      // console.log(res);
    }
  };
  // 转换物流信息
  const getExpressInfo = (code: string | undefined | null) => {
    return expressList.find((item) => item.value === code)?.label || code;
  };
  // componentDidMount
  useEffect(() => {
    if (!open) return;
    if (!data) return;
    getListExpressInfo().then();
    console.log('data --->', data);
  }, [open]);
  return (
    <Modal
      open={open}
      title={title}
      onCancel={handleCancel}
      onOk={handleOk}
      destroyOnClose
      width={600}
    >
      <Form
        name='addForm'
        form={form}
        labelAlign='right'
        labelCol={{ span: 8 }}
        wrapperCol={{ span: 16 }}
        autoComplete='new-password'
      >
        <Form.Item label='收货方式' labelCol={{ span: 4 }}>
          物流归还
        </Form.Item>
        <Form.Item label='物流单号' labelCol={{ span: 4 }}>
          {data?.receipt?.sendExNo}
        </Form.Item>
        <Form.Item label='物流公司' labelCol={{ span: 4 }}>
          {getExpressInfo(data?.receipt?.sendExCode)}
        </Form.Item>
        <Row gutter={{ xs: 8, sm: 16, md: 24 }}>
          <Col span={11}>
            <Form.Item
              label='质检照片'
              name='imgs'
              rules={[{ required: true, message: '请上传质检照片' }]}
            >
              <Uploader
                listType={'picture-card'}
                fileUpload
                fileLength={3}
                onChange={(e) => {
                  form.setFieldValue(
                    'imgs',
                    e.map((item) => item.url),
                  );
                }}
              >
                <PlusOutlined />
              </Uploader>
            </Form.Item>
          </Col>
          <Col span={11}>
            <Form.Item
              label='质检视频'
              name='videoUrl'
              rules={[{ required: true, message: '请上传质检视频' }]}
            >
              <Uploader
                listType={'picture-card'}
                fileUpload
                fileLength={1}
                fileType={['video/mp4', 'video/avi', 'video/wmv', 'video/rmvb']}
                fileSize={10}
                onChange={(e) => {
                  form.setFieldValue('videoUrl', e[0].url);
                }}
              >
                <PlusOutlined />
              </Uploader>
            </Form.Item>
          </Col>
        </Row>
        <Row gutter={{ xs: 8, sm: 16, md: 24 }}>
          <Col span={11}>
            <Form.Item
              label='设备状态'
              name='vcuSatus'
              rules={[{ required: true, message: '请选择设备状态' }]}
              initialValue={0}
            >
              <Radio.Group
                options={[
                  { label: '正常', value: 0 },
                  { label: '故障', value: 1 },
                ]}
                onChange={(e) => {
                  form.setFieldValue('vcuSatus', e.target.value);
                }}
              />
            </Form.Item>
          </Col>
          <Col span={11}>
            <Form.Item
              label='操作密码'
              name='authPwd'
              rules={[{ required: true, message: '请输入操作密码' }]}
            >
              <Input.Password
                placeholder={'请输入操作密码'}
                maxLength={20}
                allowClear
                autoComplete='new-password'
              />
            </Form.Item>
          </Col>
        </Row>
      </Form>
    </Modal>
  );
};

export default OrderConfirm;
