import SearchBox from '~/components/search-box';
import { Button, Form, Image, message, Modal, Radio, Table, Tooltip } from 'antd';
import {
  ArrowDownOutlined,
  ArrowUpOutlined,
  DownOutlined,
  PlusOutlined,
  RightOutlined,
} from '@ant-design/icons';
import { categoryListRespType } from '~/api/interface/categoryManage';
import { useEffect, useState } from 'react';
import { InterDataType, PaginationProps } from '~/api/interface';
import { CategoryManageAPI } from '~/api';
import EditableCell from '~/components/EditableCell';
import AddOrEditCategoryModal from './components/addOrEditCategoryModal';
import './index.scss';
import _ from 'lodash';

type EditableTableProps = Parameters<typeof Table>[0];
type ColumnTypes = Exclude<EditableTableProps['columns'], undefined>;
//分类列表返回类型
type categoryType = InterDataType<categoryListRespType>['list'];
const CategoryList = () => {
  const [categoryTableForm] = Form.useForm<any>();
  const [selectedRowKeys, setSelectedRowKeys] = useState<React.Key[]>([]);
  //新增，编辑分类弹窗
  const [addOrEditCategoryModalShow, setAddOrEditCategoryModalShow] = useState<boolean>(false);
  const [currentCategory, setCurrentCategory] = useState<categoryType[0]>();
  const [loading, setLoading] = useState<boolean>(false);
  // 编辑的行
  const [editingKey, setEditingKey] = useState<number[]>([]);
  //展开的行
  const [expandedRowKeys, setExpandedRowKeys] = useState<number[]>([]);

  const tableColumns: (ColumnTypes[number] & {
    editable?: boolean;
    dataIndex?: string;
    rules?: any;
    maxLength?: number;
    inputType?: string;
  })[] = [
    {
      title: '分类名称',
      dataIndex: 'name',
      editable: true,
      width: '20%',
      ellipsis: true,
      onHeaderCell: () => ({
        style: {
          textAlign: 'center',
        },
      }),
      rules: [{ required: true, message: '请输入分类名称' }],
      maxLength: 15,
      render: (text: string, record: any) => (
        <span
          style={{
            marginLeft: record.subDTOList && record.subDTOList.length != 0 ? '10px' : '24px',
          }}
        >
          {text}
        </span>
      ),
    },
    {
      title: '图片',
      align: 'center',
      dataIndex: 'icon',
      render: (url: string) => (url ? <Image src={url} width={40} height={40} /> : ''),
    },
    {
      title: '描述',
      align: 'center',
      dataIndex: 'description',
      editable: true,
      width: '20%',
      ellipsis: true,
      maxLength: 70,
      inputType: 'textArea',
      render: (text: string) => (
        <Tooltip placement='topLeft' title={<span>{text}</span>}>
          <span>{text}</span>
        </Tooltip>
      ),
    },
    {
      title: '创建时间',
      align: 'center',
      dataIndex: 'createTime',
    },
    {
      title: '操作',
      onHeaderCell: () => ({
        style: {
          textAlign: 'center',
        },
      }),
      onCell: () => ({
        style: {
          textAlign: 'right',
        },
      }),
      render: (_text: string, record: any) =>
        editingKey.some((id: number) => id === record.id) ? (
          <>
            <Button type='link' onClick={() => addChildCategorySubmit(record)}>
              确定
            </Button>
            <Button type='link' onClick={() => addChildCategoryCancel(record)}>
              取消
            </Button>
          </>
        ) : (
          <>
            {record.subDTOList ? (
              <Button type='link' onClick={() => addChildrenCategory(record)}>
                新增子分类
              </Button>
            ) : (
              ''
            )}
            <Button type='link' onClick={() => addOrEditCategoryModalShowClick(record)}>
              编辑
            </Button>
            <Button type='link' danger onClick={() => deleteCategory(record)}>
              删除
            </Button>
          </>
        ),
    },
  ];
  const mergedColumns = tableColumns.map((col: any) => {
    if (!col.editable) {
      return col;
    }
    return {
      ...col,
      onCell: (record: categoryType[0], index: number) => ({
        record,
        index,
        dataIndex: col.dataIndex,
        title: col.title,
        editing: editingKey.some((id: number) => id === record.id),
        rules: col.rules,
        maxLength: col.maxLength,
        inputType: col.inputType,
      }),
    };
  });
  const [tableData, setTableData] = useState<categoryType>([]);
  const [allTableData, setAllTableData] = useState<categoryType>([]);
  //分页
  const [pagination, setPagination] = useState<PaginationProps & { totalCount: number }>({
    pageNo: 1,
    pageSize: 10,
    totalCount: 0,
  });

  //获取分类列表
  const getCategoryList = (query?: PaginationProps, isAll?: boolean) => {
    setLoading(true);
    CategoryManageAPI.getCategoryRespList({
      pageNo: pagination.pageNo,
      pageSize: pagination.pageSize,
      ...query,
    }).then(({ result }) => {
      setLoading(false);
      if (isAll) {
        setAllTableData(result.list ? [...result.list] : []);
      } else {
        setTableData(result.list ? [...result.list] : []);
        pagination.totalCount = result.totalCount;
        setPagination(pagination);
      }
    });
  };
  const paginationChange = (pageNo: number, pageSize: number) => {
    pagination.pageNo = pageNo;
    pagination.pageSize = pageSize;
    getCategoryList();
  };
  // 排序选择
  const onSelectChange = (record: categoryType[0]) => {
    setSelectedRowKeys([record.id]);
  };
  //新增、编辑分类弹窗
  const addOrEditCategoryModalShowClick = (record?: categoryType[0] | any) => {
    if (record && !record.subDTOList) {
      setEditingKey([...editingKey, record.id]);
      const defaultFormVal = Object.getOwnPropertyNames(record).reduce((pre: any, cur: string) => {
        if (['name', 'description'].includes(cur)) {
          pre[cur + record.id] = record[cur];
        }
        return pre;
      }, {});
      categoryTableForm.setFieldsValue(defaultFormVal);
    } else {
      setAddOrEditCategoryModalShow(true);
      setCurrentCategory(record ? { ...record } : undefined);
    }
  };
  const addOrEditCategoryModalCancel = () => {
    setAddOrEditCategoryModalShow(false);
  };
  const addOrEditCategoryModalOk = () => {
    setAddOrEditCategoryModalShow(false);
    getCategoryList();
  };
  //删除分类
  const deleteCategory = (record: categoryType[0]) => {
    Modal.confirm({
      title: '分类删除',
      content: '确认删除这条分类?',
      onOk: () => {
        console.log('当前数据-->', record);
        CategoryManageAPI[record.subDTOList ? 'deletePrimaryCategory' : 'deleteSubCategory']({
          id: record.id,
        }).then(({ code }) => {
          if (code === '200') {
            message.success('删除成功');
            if (pagination.pageNo !== 1 && tableData.length === 1) {
              pagination.pageNo -= 1;
            }
            getCategoryList();
            getCategoryList({ pageNo: 1, pageSize: 99999 }, true);
          }
        });
      },
    });
  };
  //新增子分类
  const addChildrenCategory = (record: categoryType[0]) => {
    const categoryItemIndex = tableData.findIndex((v) => v.id === record.id);
    if (categoryItemIndex !== -1) {
      const selfChildId: number = Math.random();
      tableData[categoryItemIndex].subDTOList = [
        ...tableData[categoryItemIndex].subDTOList,
        { id: selfChildId, name: '', description: '', categoryPrimaryId: record.id },
      ];
      tableData.splice(categoryItemIndex, 1, tableData[categoryItemIndex]);
      setTableData([...tableData]);
      setEditingKey([...editingKey, selfChildId]);
    }
    const isExpend: boolean = expandedRowKeys.some((key: number) => key === record.id);
    if (!isExpend) {
      setExpandedRowKeys([...expandedRowKeys, record.id]);
    }
  };
  //   展开监听
  const onExpand = (_expanded: boolean, record: categoryType[0]) => {
    const index = expandedRowKeys.findIndex((item: any) => item === record.id);
    if (index !== -1) {
      expandedRowKeys.splice(index, 1);
    } else {
      expandedRowKeys.push(record.id);
    }
    setExpandedRowKeys([...expandedRowKeys]);
  };
  //提交新增子分类
  const addChildCategorySubmit = (record: categoryType[0]['subDTOList'][0]) => {
    categoryTableForm
      .validateFields(['description' + record.id, 'name' + record.id])
      .then((values) => {
        CategoryManageAPI[record.name ? 'updateSubCategory' : 'addSubCategory']({
          categoryPrimaryId: record.categoryPrimaryId,
          name: values['name' + record.id],
          description: values['description' + record.id],
          id: record.name ? record.id : undefined,
        }).then(({ code }) => {
          if (code === '200') {
            message.success(record.name ? '编辑子分类成功' : '新增子分类成功');
            const editingKeyIndex = editingKey.findIndex((id) => id === record.id);
            if (editingKeyIndex !== -1) {
              editingKey.splice(editingKeyIndex, 1);
              setEditingKey([...editingKey]);
              getCategoryList();
            }
          }
        });
      });
  };
  //取消新增子分类
  const addChildCategoryCancel = (record: categoryType[0]['subDTOList'][0]) => {
    const tableIndex: number = tableData.findIndex((v) => v.id === record.categoryPrimaryId);
    if (tableIndex !== -1) {
      const subDTOIndex: number = tableData[tableIndex].subDTOList.findIndex(
        (v) => v.id === record.id,
      );
      if (subDTOIndex !== -1) {
        //新增下的取消
        if (!record.name) {
          tableData[tableIndex].subDTOList.splice(subDTOIndex, 1);
          tableData.splice(tableIndex, 1, tableData[tableIndex]);
          setTableData([...tableData]);
        }
        //新增、编辑下的取消
        const editKeyIndex = editingKey.findIndex((id) => id === record.id);
        editingKey.splice(editKeyIndex, 1);
        setEditingKey([...editingKey]);
        //新增下的取消
        if (tableData[tableIndex].subDTOList.length === 0 && !record.name) {
          const expandedRowKeysIndex = expandedRowKeys.findIndex(
            (id) => id === record.categoryPrimaryId,
          );
          expandedRowKeys.splice(expandedRowKeysIndex, 1);
          setExpandedRowKeys([...expandedRowKeys]);
        }
      }
    }
  };
  // 切换排序
  const handleSort = _.debounce(async (from: 'up' | 'down') => {
    if (selectedRowKeys.length === 0) {
      message.warning('请选择需要排序的数据').then();
      return;
    }
    // 当前选项
    const id = selectedRowKeys.at(-1);
    // 当前索引
    const index = allTableData.findIndex((i) => i.id === id);
    // 当前表格中的索引
    const tableIndex = tableData.findIndex((i) => i.id === id);
    // 第一条数据不能上移
    if (index === 0 && from === 'up') {
      message.warning('已是第一条数据').then();
      return;
    }
    // 最后一条数据不能下移
    if (index === allTableData.length - 1 && from === 'down') {
      message.warning('已是最后一条数据').then();
      return;
    }
    // 转换位置
    const res = await CategoryManageAPI.exchangeCategory(
      from === 'up'
        ? [
            { id: allTableData[index - 1].id, sort: allTableData[index - 1].sort },
            { id: allTableData[index].id, sort: allTableData[index].sort },
          ]
        : [
            { id: allTableData[index + 1].id, sort: allTableData[index + 1].sort },
            { id: allTableData[index].id, sort: allTableData[index].sort },
          ],
    );
    if (res && res.code === '200') {
      message.success('操作成功').then();
      // 如果是当前页的第一条数据
      if (tableIndex === 0 && from === 'up') {
        paginationChange(pagination.pageNo - 1, pagination.pageSize);
        getCategoryList({ pageNo: 1, pageSize: 9999 }, true);
        return;
      }
      // 如果是当前页的最后一条数据
      if (tableIndex === tableData.length - 1 && from === 'down') {
        paginationChange(pagination.pageNo + 1, pagination.pageSize);
        getCategoryList({ pageNo: 1, pageSize: 9999 }, true);
        return;
      }
      paginationChange(pagination.pageNo, pagination.pageSize);
      getCategoryList({ pageNo: 1, pageSize: 9999 }, true);
    }
  }, 500);
  //防抖
  // const  debounce:<T extends ()=>T>= (func: T, delay: number)=> {
  //   let timeout: NodeJS.Timeout;
  //
  //   return function (this: ThisParameterType<T>, ...args: Parameters<T>): ReturnType<T> {
  //     clearTimeout(timeout);
  //
  //     timeout = setTimeout(() => {
  //       func.apply(this, args);
  //     }, delay);
  //   } as T;
  // }

  useEffect(() => {
    getCategoryList();
    getCategoryList({ pageNo: 1, pageSize: 99999 }, true);
  }, []);

  return (
    <div className='category-list'>
      <SearchBox
        child={
          <>
            <Button
              icon={<PlusOutlined />}
              type='primary'
              onClick={() => addOrEditCategoryModalShowClick()}
            >
              新增分类
            </Button>
            <Button
              icon={<ArrowUpOutlined />}
              type='primary'
              onClick={() => handleSort('up')}
            ></Button>
            <Button
              icon={<ArrowDownOutlined />}
              type='primary'
              onClick={() => handleSort('down')}
            ></Button>
          </>
        }
      />
      <Form form={categoryTableForm} component={false}>
        <Table
          bordered
          rowSelection={{
            hideSelectAll: true,
            selectedRowKeys,
            renderCell: (checked: boolean, record: categoryType[0]) => {
              return (
                <>
                  {record.subDTOList ? (
                    <Radio onChange={() => onSelectChange(record)} checked={checked} />
                  ) : (
                    ''
                  )}
                </>
              );
            },
          }}
          loading={loading}
          rowKey='id'
          columns={mergedColumns}
          dataSource={tableData}
          rowClassName={(record: categoryType[0]) => {
            if (record.subDTOList) {
              return '';
            }
            return 'category-list-row-bg';
          }}
          pagination={{
            total: pagination.totalCount,
            pageSize: pagination.pageSize,
            current: pagination.pageNo,
            showSizeChanger: true,
            showQuickJumper: true,
            onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
            showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
          }}
          components={{
            body: {
              cell: EditableCell,
            },
          }}
          expandedRowKeys={expandedRowKeys}
          onExpand={(expanded: boolean, record: categoryType[0]) => onExpand(expanded, record)}
          expandable={{
            rowExpandable: (record) => {
              if (record.subDTOList && record.subDTOList.length != 0) {
                return true;
              }
              return false;
            },
            expandIcon: ({ expanded, onExpand, record }) => {
              if (expanded && record.subDTOList && record.subDTOList.length != 0) {
                return <DownOutlined onClick={(e) => onExpand(record, e)} />;
              }
              if (record.subDTOList && record.subDTOList.length != 0) {
                return <RightOutlined onClick={(e) => onExpand(record, e)} />;
              }
              return <></>;
            },
          }}
          childrenColumnName='subDTOList'
        />
      </Form>

      {/*新增，编辑分类*/}
      <AddOrEditCategoryModal
        open={addOrEditCategoryModalShow}
        onCancel={addOrEditCategoryModalCancel}
        onOK={addOrEditCategoryModalOk}
        currentCategory={currentCategory}
      />
    </div>
  );
};
export default CategoryList;
