import { useEffect, useState } from 'react';
import SearchBox from '~/components/search-box';
import { ResourceManageAPI } from '~/api';
import qs from 'query-string';
import { useLocation, useNavigate } from 'react-router-dom';
import { Button, Table } from 'antd';
import { ArrowLeftOutlined } from '@ant-design/icons';
import {
  releaseTenderNewsApplyList,
  releaseTenderNewsInfo,
} from '~/api/interface/resourceManageType';
import { InterListType, InterReqListType } from '~/api/interface';
import { ColumnsType } from 'antd/es/table';

// 列表类型
type ListType = InterListType<releaseTenderNewsInfo>;
// 表格类型
type TableType = InterListType<releaseTenderNewsApplyList>;
// 请求的参数
type ReqType = InterReqListType<releaseTenderNewsApplyList>;
// 搜索表单的数据
let query: ReqType = {};

const TenderManageFeedback = () => {
  // 路由钩子
  const location = useLocation();
  // 导航钩子
  const navigate = useNavigate();
  // 返回上一页
  const handleBack = () => {
    navigate(-1);
  };
  // 表格数据
  const [tableData, setTableData] = useState<TableType>([]);
  // 表格分页配置
  const [pagination, setPagination] = useState({
    total: 0,
    pageSize: 10,
    current: 1,
    totalPage: 0,
  });
  // 快讯详情列表
  const [tenderInfoList, setTenderInfoId] = useState<
    { value: number; label: string; data: ListType[0] }[]
  >([]);
  // 获取快讯详情列表
  const getNewsInfoList = async () => {
    const res = await ResourceManageAPI.releaseTenderNewsInfo({
      pageNo: 1,
      pageSize: 9999,
      tenderNewsId: Number(qs.parse(location.search)?.id),
    });
    if (res && res.code === '200') {
      const list = res.result?.list || [];
      setTenderInfoId(
        list?.map((i, j) => ({
          value: i.id,
          label: `${j + 1}、${i.tenderTitle}`,
          data: i,
        })),
      );
    }
  };
  // 加载列表
  const getTableList = async (value = {}) => {
    // 只需要修改这个地方的接口即可
    const res = await ResourceManageAPI.releaseTenderNewsApplyList({
      pageNo: pagination.current,
      pageSize: pagination.pageSize,
      ...value,
      ...query,
      tenderNewsId: Number(qs.parse(location.search)?.id),
    });
    if (res && res.code === '200') {
      const { list, pageNo, totalCount, pageSize, totalPage } = res.result; // 解构
      setPagination({
        total: totalCount,
        current: pageNo,
        pageSize,
        totalPage,
      });
      setTableData(list);
    }
  };
  // 翻页
  const paginationChange = (pageNo: number, pageSize: number) => {
    getTableList({ pageNo, pageSize }).then();
  };
  // 表单提交
  const onFinish = (data: ReqType) => {
    pagination.current = 1;
    query = data;
    getTableList(data).then();
  };
  // componentDidMount
  useEffect(() => {
    query = {};
    getNewsInfoList().then();
    getTableList().then();
  }, []);
  // 表格结构
  const columns: ColumnsType<TableType[0]> = [
    {
      title: '编号',
      dataIndex: 'tenderInfoNo',
      align: 'center',
    },
    {
      title: '标题',
      dataIndex: 'tenderTitle',
      align: 'center',
      ellipsis: true,
    },
    {
      title: '用户昵称',
      dataIndex: 'nickName',
      align: 'center',
    },
    {
      title: '用户姓名',
      dataIndex: 'userName',
      align: 'center',
    },
    {
      title: '电话号码',
      dataIndex: 'phoneNum',
      align: 'center',
    },
  ];
  return (
    <>
      <SearchBox
        search={[
          {
            label: '项目编号',
            name: 'tenderInfoId',
            type: 'Select',
            placeholder: '请选择编号',
            options: tenderInfoList,
          },
          { label: '用户姓名', name: 'userName', type: 'input', placeholder: '请输入用户姓名' },
        ]}
        searchData={onFinish}
        child={
          <>
            <Button type={'primary'} icon={<ArrowLeftOutlined />} onClick={handleBack}>
              返回
            </Button>
          </>
        }
      />
      <Table
        size='small'
        dataSource={tableData}
        columns={columns}
        rowKey='id'
        // scroll={{ x: 1500 }}
        bordered
        pagination={{
          total: pagination.total,
          pageSize: pagination.pageSize,
          current: pagination.current,
          showSizeChanger: true,
          showQuickJumper: true,
          onChange: (page: number, pageSize: number) => paginationChange(page, pageSize),
          showTotal: (total, range) => `当前 ${range[0]}-${range[1]} 条记录 / 共 ${total} 条数据`,
        }}
      />
    </>
  );
};

export default TenderManageFeedback;
