import { FC, useEffect, useState } from 'react';
import { Cascader, Form, Input, message, Modal, ModalProps, Select } from 'antd';
import { OrderManageAPI } from '~/api';
import { InterReqType } from '~/api/interface';
import { rentOrderSendType } from '~/api/interface/orderManageType';
import regionData from '~/assets/json/district.json';

// 发货请求类型
type sendMallType = Omit<Exclude<InterReqType<rentOrderSendType>, undefined>, 'takeRegion'> & {
  takeRegion: string[];
};

interface selfProps {
  currentOrderItem: any;
  onCancel: () => void;
  onOk: () => void;
}
const ShipmentsOrder: FC<ModalProps & selfProps> = ({ onOk, onCancel, open, currentOrderItem }) => {
  const [form] = Form.useForm<sendMallType>();
  const [listExpressInfo, setListExpressInfo] = useState<{ label: string; value: string }[]>([]);

  //获取物流公司列表
  const getExpressList = () => {
    OrderManageAPI.listExpressInfo().then(({ result }) => {
      setListExpressInfo(result?.map((v) => ({ label: v.exName, value: v.exCode })) || []);
    });
  };

  const handleOk = () => {
    form.validateFields().then((value) => {
      OrderManageAPI.rentOrderSend({
        ...value,
        uavOrderId: currentOrderItem?.id,
        id: currentOrderItem?.uavOrderExpressDTO.id,
        takeRegion: value?.takeRegion?.join('/'),
      }).then(({ code }) => {
        if (code === '200') {
          message.success('发货成功');
          form.resetFields();
          onOk();
        }
      });
    });
  };
  const handleCancel = () => {
    form.resetFields();
    onCancel();
  };
  useEffect(() => {
    if (open) {
      getExpressList();
    }
  }, [open]);

  useEffect(() => {
    if (currentOrderItem && open) {
      form.setFieldsValue({
        takeName: currentOrderItem.uavOrderExpressDTO.takeName,
        takePhone: currentOrderItem.uavOrderExpressDTO.takePhone,
        takeRegion: currentOrderItem.uavOrderExpressDTO.takeRegion.split('/'),
        takeAddress: currentOrderItem.uavOrderExpressDTO.takeAddress,
      });
    }
  }, [currentOrderItem]);

  return (
    <Modal open={open} title='发货' onOk={handleOk} onCancel={handleCancel}>
      <Form
        name='basic'
        labelCol={{ span: 4 }}
        wrapperCol={{ span: 16 }}
        initialValues={{ remember: true }}
        autoComplete='off'
        form={form}
      >
        <Form.Item
          label='物流公司'
          name='sendExpCode'
          rules={[
            {
              required: true,
              message: '请选择物流公司',
            },
          ]}
        >
          <Select placeholder='请选择物流公司' allowClear options={listExpressInfo}></Select>
        </Form.Item>
        <Form.Item
          label='物流编号'
          name='sendExpNo'
          rules={[
            {
              required: true,
              message: '请输入物流编号',
            },
          ]}
        >
          <Input placeholder='请输入物流编号' allowClear maxLength={50} />
        </Form.Item>
        <Form.Item
          label='用户姓名'
          name='takeName'
          rules={[
            {
              required: true,
              message: '请输入用户姓名',
            },
          ]}
        >
          <Input placeholder='请输入用户姓名' allowClear maxLength={30} />
        </Form.Item>
        <Form.Item
          label='手机号'
          name='takePhone'
          rules={[
            {
              required: true,
              message: '请输入手机号',
            },
            {
              pattern: /^1[3-9]\d{9}$/,
              message: '手机号格式不正确',
            },
          ]}
        >
          <Input placeholder='请输入手机号' allowClear maxLength={30} />
        </Form.Item>
        <Form.Item
          label='收货地址'
          name='takeRegion'
          rules={[
            {
              required: false,
              message: '请选择收货地址',
            },
          ]}
        >
          <Cascader
            options={regionData}
            placeholder='请选择收货地址'
            allowClear
            fieldNames={{
              label: 'name',
              value: 'name',
              children: 'childInfo',
            }}
          />
        </Form.Item>
        <Form.Item
          label='详细地址'
          name='takeAddress'
          rules={[
            {
              required: false,
              message: '请输入详细地址',
            },
          ]}
        >
          <Input.TextArea maxLength={70} showCount placeholder='请输入详细地址' allowClear />
        </Form.Item>
      </Form>
    </Modal>
  );
};
export default ShipmentsOrder;
