import React, { useEffect } from 'react';
import { Form, Input, message, Modal } from 'antd';
import { maxLength } from '~/utils/validateUtils';
import { CouponManageAPI } from '~/api';

// 传参类型
interface propType {
  title: string;
  open: boolean;
  closed: any;
  data?: any;
}

export const IncreaseModal: React.FC<propType> = (props: propType) => {
  // 组件默认值
  IncreaseModal.defaultProps = {
    data: null,
  };
  // 参数
  const { title, open, closed, data } = props;
  // 表单钩子
  const [form] = Form.useForm();
  // 关闭弹窗
  const handleCancel = () => {
    form.resetFields();
    closed();
  };
  // 提交弹窗
  const handleOk = () => {
    form
      .validateFields()
      .then(async (values) => {
        await handleSubmit(values);
        // console.log("handleOk -->", values);
      })
      .catch((err) => {
        message
          .warning({
            content: err.errorFields[0].errors[0],
          })
          .then();
      });
  };
  // 新增发行量监听
  const handleIncrease = ({ target: { value } }: any) => {
    if (value) {
      form.setFieldsValue({
        lastTotalBefore: Number(data.lastTotal) + Number(value),
      });
    } else {
      form.setFieldsValue({
        lastTotalBefore: data.lastTotal,
      });
    }
  };
  // 提交数据
  const handleSubmit = async (values: { count: number }) => {
    const res = await CouponManageAPI.CouponIncrease({
      id: data.id,
      count: Number(values.count),
    });
    if (res && res.code === '200') {
      message.success('操作成功').then();
      handleCancel();
    }
  };
  useEffect(() => {
    if (!data) return;
    // console.log("IncreaseModal --->", data);
    form.setFieldsValue({
      ...data,
      lastTotalBefore: data.lastTotal,
    });
  }, [data]);
  return (
    <Modal
      open={open}
      title={title}
      onCancel={handleCancel}
      onOk={handleOk}
      destroyOnClose
      width={400}
    >
      <Form
        name='addForm'
        form={form}
        labelAlign='right'
        labelCol={{ span: 8 }}
        // layout="inline"
      >
        <Form.Item label='当前发行总量' name='lastTotal'>
          <Input placeholder='请输入当前发行总量' maxLength={20} allowClear disabled suffix='张' />
        </Form.Item>
        <Form.Item
          label='新增发行量'
          name='count'
          rules={[
            { required: true, message: '请输入新增发行量' },
            // 增发合计不能超过100000
            {
              validator: (rule, value) => {
                if (Number(data.lastTotal) + Number(value) > 100000) {
                  return Promise.reject('增发合计不能超过100000');
                }
                return Promise.resolve();
              },
            },
          ]}
        >
          <Input
            placeholder='请输入新增发行量'
            maxLength={20}
            allowClear
            prefix='+'
            suffix='张'
            type='number'
            onChange={handleIncrease}
            onInput={maxLength}
          />
        </Form.Item>
        <Form.Item label='更新后总发行量' name='lastTotalBefore'>
          <Input
            placeholder='请输入更新后总发行量'
            maxLength={20}
            allowClear
            disabled
            suffix='张'
          />
        </Form.Item>
      </Form>
    </Modal>
  );
};
