import React, { useEffect, useState } from 'react';
import { Form, Input, message, Modal, ModalProps } from 'antd';
import { InterDataType, InterReqType } from '~/api/interface';
import { getCompanyInfoByIdType } from '~/api/interface/systemManageType';
import { UploadOutlined } from '@ant-design/icons';
import { Uploader } from '~/components/uploader';
import { getSuggestionType, updateSuggestionType } from '~/api/interface/customManageType';
import { CustomManageAPI } from '~/api';

// 店铺类型
type DetailType = InterDataType<getCompanyInfoByIdType>;
// 请求类型
type ReqType = InterReqType<updateSuggestionType>;
// 列表类型
type DataType = InterDataType<getSuggestionType>[0];

const AddSolutionModal: React.FC<
  ModalProps & { detail: DetailType; onCancel?: () => void; data?: DataType }
> = ({ open, title, onCancel, data }) => {
  // 表单数据
  const [form] = Form.useForm<ReqType>();
  // 封面图列表
  const [coverUrlList, setCoverUrlList] = useState<{ url: string }[]>();
  // 文件列表
  const [suggestFileList, setSuggestFileList] = useState<{ url: string }[]>();
  // 封面上传回调
  const coverUrlChange = (e: { url: string }[]) => {
    setCoverUrlList(e);
    form.setFieldValue('coverUrl', e[0].url);
    // console.log('封面上传回调 --->', e);
  };
  // 方案文件回调
  const suggestFileChange = (e: { url: string }[]) => {
    setSuggestFileList(e);
    form.setFieldValue('suggestFile', e[0].url);
    // console.log('方案文件回调 --->', e);
  };
  // 取消事件
  const handleCancel = () => {
    form.resetFields();
    setCoverUrlList([]);
    setSuggestFileList([]);
    onCancel?.();
  };
  // 确定事件
  const handleOk = () => {
    form
      .validateFields()
      .then(async (values) => {
        await handleSubmit(values);
      })
      .catch((err) => {
        // console.log('确定事件 --->', err);
        message
          .warning({
            content: err.errorFields[0].errors[0],
          })
          .then();
      });
  };
  // 提交事件
  const handleSubmit = async (values: ReqType) => {
    const res = await CustomManageAPI.updateSuggestion({ ...data, ...values });
    if (res && res.code === '200') {
      message.success({ content: '操作成功' });
      handleCancel();
      // console.log('提交事件 --->', values);
    }
  };
  // 组件挂载
  useEffect(() => {
    if (!data?.id) return;
    form.setFieldsValue(data);
    setCoverUrlList([{ url: data.coverUrl }]);
    setSuggestFileList([{ url: data.suggestFile }]);
  }, [data]);
  return (
    <Modal open={open} title={title} width={450} onCancel={handleCancel} onOk={handleOk}>
      <Form form={form} labelCol={{ span: 4 }} wrapperCol={{ span: 16 }}>
        <Form.Item
          label='方案标题'
          name='title'
          rules={[{ required: true, message: '请输入方案标题' }]}
        >
          <Input placeholder={'请输入方案标题'} maxLength={10} allowClear />
        </Form.Item>
        <Form.Item
          label='方案封面'
          name='coverUrl'
          rules={[{ required: true, message: '请上传方案封面' }]}
        >
          <Uploader
            fileUpload
            listType='picture-card'
            defaultFileList={coverUrlList}
            onChange={coverUrlChange}
            fileLength={1}
            fileSize={50}
          >
            <UploadOutlined />
          </Uploader>
        </Form.Item>
        <Form.Item
          label='方案文件'
          name='suggestFile'
          rules={[{ required: true, message: '请输入方案文件' }]}
        >
          <Uploader
            fileUpload
            listType='picture-card'
            defaultFileList={suggestFileList}
            onChange={suggestFileChange}
            fileType={['application/pdf']}
            fileLength={1}
            fileSize={100}
          >
            <UploadOutlined />
          </Uploader>
        </Form.Item>
      </Form>
    </Modal>
  );
};

export default AddSolutionModal;
