import { FC, useEffect, useState } from 'react';
import { Cascader, Form, Input, message, Modal, ModalProps, Select } from 'antd';
import { CommonAPI, OrderManageAPI } from '~/api';
import { InterDataType, InterListType, InterReqType } from '~/api/interface';
import { getSecondDistrictInfoType } from '~/api/interface/commonType';
import { mallOrderListType, mallOrderSendType } from '~/api/interface/orderManageType';

// 地址返回类型
type locationType = InterDataType<getSecondDistrictInfoType>;
type mallOrderType = InterListType<mallOrderListType>[0];
// 发货请求类型
type sendMallType = InterReqType<mallOrderSendType>;

interface selfProps {
  currentOrderItem: mallOrderType | undefined;
  onCancel: () => void;
  onOk: () => void;
}
const ShipmentsOrder: FC<ModalProps & selfProps> = ({ onOk, onCancel, open, currentOrderItem }) => {
  const [form] = Form.useForm<sendMallType>();
  const [listExpressInfo, setListExpressInfo] = useState<{ label: string; value: string }[]>([]);
  const [location, setLocation] = useState<locationType>([]);

  //获取物流公司列表
  const getExpressList = () => {
    OrderManageAPI.listExpressInfo().then(({ result }) => {
      setListExpressInfo(result?.map((v) => ({ label: v.exName, value: v.exCode })) || []);
    });
  };
  //   获取地址信息
  const getLocation = () => {
    CommonAPI.getSecondDistrictInfo().then(({ result }) => {
      setLocation(result || []);
    });
  };

  const handleOk = () => {
    form.validateFields().then((value) => {
      OrderManageAPI.mallOrderSend({
        ...value,
        uavOrderId: currentOrderItem?.id,
        id: currentOrderItem?.uavOrderExpressDTO.id,
      }).then(({ code }) => {
        if (code === '200') {
          message.success('发货成功');
          form.resetFields();
          onOk();
        }
      });
    });
  };
  const handleCancel = () => {
    form.resetFields();
    onCancel();
  };
  useEffect(() => {
    if (open) {
      getLocation();
      getExpressList();
    }
  }, [open]);

  useEffect(() => {
    if (currentOrderItem && open) {
      form.setFieldsValue({
        takeName: currentOrderItem.uavOrderExpressDTO.takeName,
        takePhone: currentOrderItem.uavOrderExpressDTO.takePhone,
        takeRegion: currentOrderItem.uavOrderExpressDTO.takeRegion,
        takeAddress: currentOrderItem.uavOrderExpressDTO.takeAddress,
      });
    }
  }, [currentOrderItem]);

  return (
    <Modal open={open} title='发货' onOk={handleOk} onCancel={handleCancel}>
      <Form
        name='basic'
        labelCol={{ span: 4 }}
        wrapperCol={{ span: 16 }}
        initialValues={{ remember: true }}
        autoComplete='off'
        form={form}
      >
        <Form.Item
          label='物流公司'
          name='sendExpCode'
          rules={[
            {
              required: true,
              message: '请选择物流公司',
            },
          ]}
        >
          <Select placeholder='请选择物流公司' allowClear options={listExpressInfo}></Select>
        </Form.Item>
        <Form.Item
          label='物流编号'
          name='sendExpNo'
          rules={[
            {
              required: true,
              message: '请输入物流编号',
            },
          ]}
        >
          <Input placeholder='请输入物流编号' allowClear maxLength={50} />
        </Form.Item>
        <Form.Item
          label='用户姓名'
          name='takeName'
          rules={[
            {
              required: true,
              message: '请输入用户姓名',
            },
          ]}
        >
          <Input placeholder='请输入用户姓名' allowClear maxLength={30} />
        </Form.Item>
        <Form.Item
          label='手机号'
          name='takePhone'
          rules={[
            {
              required: true,
              message: '请输入手机号',
            },
            {
              pattern: /^1[3-9]\d{9}$/,
              message: '手机号格式不正确',
            },
          ]}
        >
          <Input placeholder='请输入手机号' allowClear maxLength={30} />
        </Form.Item>
        <Form.Item
          label='收货地址'
          name='takeRegion'
          rules={[
            {
              required: false,
              message: '请选择收货地址',
            },
          ]}
        >
          <Cascader
            options={location}
            placeholder='请选择收货地址'
            allowClear
            fieldNames={{
              label: 'name',
              value: 'name',
              children: 'childInfo',
            }}
          />
        </Form.Item>
        <Form.Item
          label='详细地址'
          name='takeAddress'
          rules={[
            {
              required: false,
              message: '请输入详细地址',
            },
          ]}
        >
          <Input.TextArea maxLength={70} showCount placeholder='请输入详细地址' allowClear />
        </Form.Item>
      </Form>
    </Modal>
  );
};
export default ShipmentsOrder;
