import React from 'react';
import { Image, Table } from 'antd';
import { ColumnsType } from 'antd/es/table';
import { InterDataType } from '~/api/interface';
import { rentOrderDetailType, rentOrderDictType } from '~/api/interface/orderManageType';
import dayjs from 'dayjs';
import './index.scss';

// 表格数据类型
type TableType = any;
// 接口返回的类型
type DataType = InterDataType<rentOrderDetailType>;
// 订单状态返回类型
type OrderStatusType = InterDataType<rentOrderDictType>;

// 参数类型
type PropsType = {
  detail: DataType;
  orderStatusList: OrderStatusType;
};

const DetailInformation: React.FC<PropsType> = ({ detail, orderStatusList }) => {
  // 表格结构
  const columns: ColumnsType<TableType[0]> = [
    {
      title: '商品',
      align: 'center',
      width: '20%',
      render: (_text: any, record) => (
        <div className='mall-sku-item'>
          <Image src={record?.skuImg} className='sku-img' />
          <div className='sku-info'>
            <div className='info-name'>{record?.orderName}</div>
            <div className='info-spec'>
              {Object.entries(JSON.parse(record?.sku || '[]')).map((v, index) => (
                <span key={index}>{v.join('：')}</span>
              ))}
            </div>
          </div>
        </div>
      ),
    },
    {
      title: '单价(元)',
      dataIndex: 'unitPrice',
      align: 'center',
      render: (text) => `￥${text?.toLocaleString() || 0}`,
    },
    {
      title: '数量',
      dataIndex: 'wareNum',
      align: 'center',
    },
    {
      title: '订单状态',
      dataIndex: 'statusCode',
      align: 'center',
      render: (text: number) =>
        orderStatusList.find((item) => item.status === text?.toString())?.waiting,
    },
    {
      title: '应收款',
      dataIndex: 'orderTotalAmount',
      align: 'center',
      render: (text) => `￥${text?.toLocaleString() || 0}`,
    },
  ];

  return (
    <div className='detail-information'>
      <div className='detail-title'>订单明细</div>
      <div className='detail-text'>
        <span className='item'>订单编号：{detail?.orderNo}</span>
        <span className='item'>
          订单租期：
          {dayjs(detail?.startDate).format('YYYY-MM-DD')}
          {' ~ '}
          {dayjs(detail?.endDate).format('YYYY-MM-DD')}
        </span>
        <span className='item'>{`创建时间：${detail?.createTime}`}</span>
      </div>
      <Table
        size='small'
        dataSource={[detail]}
        columns={columns}
        rowKey='id'
        bordered
        pagination={false}
      />
    </div>
  );
};

export default DetailInformation;
