import { Button, Cascader, Form, Input, Radio, Select } from 'antd';
import { Uploader } from '~/components/uploader';
import type { RadioChangeEvent } from 'antd';
import { UploadOutlined } from '@ant-design/icons';
import React, { forwardRef, useEffect, useImperativeHandle, useState } from 'react';
import { InterDataType } from '~/api/interface';
import { categoryListRespType } from '~/api/interface/categoryManage';
import { CategoryManageAPI } from '~/api';
import deletePng from '~/assets/image/delete.png';
import './index.scss';

//分类返回类型
type categoryType = InterDataType<categoryListRespType>['list'];

interface selfProps {
  ref: any;
}
//基本信息表单类型
export type baseInfoType = {
  tradeName: string;
  description: string;
  mainImgList: {
    id: number;
    name: string;
    uid: number;
    url: string;
  }[];
  subImgList: {
    id: number;
    name: string;
    uid: number;
    url: string;
  }[];
  videoList: {
    id: number;
    name: string;
    uid: number;
    url: string;
  }[];
  categoryId: number[];
  shelfStatus: number;
  labelShow: number;
  goodsLabel: string;
};

const BaseInfo = forwardRef<any, selfProps>((_props, ref) => {
  const [baseInfoForm] = Form.useForm<baseInfoType>();
  const [labelShow, setLabelShow] = useState<boolean>(false);
  //分类列表
  const [categoryList, setCategoryList] = useState<categoryType>([]);
  //主图
  const [mainFileList, setMainFileList] = useState<
    {
      id: number;
      name: string;
      uid: number;
      url: string;
    }[]
  >([]);
  //副图
  const [subFileList, setSubFileList] = useState<
    {
      id: number;
      name: string;
      uid: number;
      url: string;
    }[]
  >([]);
  //视频
  const [videoFileList, setVideoFileList] = useState<
    {
      id: number;
      name: string;
      uid: number;
      url: string;
    }[]
  >([]);

  useImperativeHandle(ref, () => ({
    getForm: () => baseInfoForm,
    setLabelShow: (value: boolean) => setLabelShow(value),
    mediaData: {
      setMainFileList,
      setSubFileList,
      setVideoFileList,
    },
  }));

  //标签选中监听
  const labelRadioChange = (e: RadioChangeEvent) => {
    setLabelShow(!!e.target.value);
  };
  //上传结果
  const uploadSuccess = (
    fileList: {
      id: number;
      name: string;
      uid: number;
      url: string;
    }[],
    type: string,
  ) => {
    switch (type) {
      case 'mainImgList':
        setMainFileList(fileList);
        baseInfoForm.setFieldValue('mainImgList', fileList);
        break;
      case 'subImgList':
        setSubFileList(fileList);
        baseInfoForm.setFieldValue('subImgList', fileList);
        break;
      case 'videoList':
        setVideoFileList(fileList);
        baseInfoForm.setFieldValue('videoList', fileList);
        break;
      default:
        break;
    }
  };
  //分类列表
  const getCategoryList = () => {
    CategoryManageAPI.getCategoryRespList({ pageNo: 1, pageSize: 99999 }).then(({ result }) => {
      setCategoryList(result.list || []);
    });
  };
  //视频删除
  const deleteVideo = () => {
    setVideoFileList([]);
    baseInfoForm.setFieldValue('videoList', undefined);
  };

  useEffect(() => {
    getCategoryList();
  }, []);

  return (
    <div className='base-info'>
      <Form
        labelCol={{ span: 2 }}
        wrapperCol={{ span: 8 }}
        form={baseInfoForm}
        initialValues={{ labelShow: 0, shelfStatus: 1, priceShow: 1 }}
      >
        <Form.Item
          label='商品名称'
          name='tradeName'
          rules={[{ required: true, message: '请输入商品名称' }]}
        >
          <Input placeholder='请输入商品名称' maxLength={50} />
        </Form.Item>
        <Form.Item
          label='商品描述'
          name='description'
          rules={[{ required: true, message: '请输入商品描述' }]}
        >
          <Input.TextArea rows={4} placeholder='请输入商品描述' maxLength={70} showCount />
        </Form.Item>
        <Form.Item
          label='商品主图'
          name='mainImgList'
          rules={[{ required: true, message: '请上传商品主图' }]}
        >
          <Uploader
            fileUpload
            listType='picture-card'
            onChange={(fileList) => uploadSuccess(fileList, 'mainImgList')}
            defaultFileList={mainFileList}
          >
            <UploadOutlined />
          </Uploader>
        </Form.Item>
        <Form.Item label='商品副图' name='subImgList'>
          <Uploader
            fileUpload
            listType='picture-card'
            onChange={(fileList) => uploadSuccess(fileList, 'subImgList')}
            defaultFileList={subFileList}
            fileLength={4}
          >
            <UploadOutlined />
          </Uploader>
        </Form.Item>
        <Form.Item label='商品视频' name='videoList'>
          {videoFileList.length ? (
            <div className='goods-video-wrap'>
              <video
                src={videoFileList[0].url}
                style={{ width: '200px', height: '200px' }}
                controls
              />
              <img src={deletePng} alt='删除' onClick={deleteVideo} />
            </div>
          ) : (
            <Uploader
              fileUpload
              listType='text'
              onChange={(fileList) => uploadSuccess(fileList, 'videoList')}
              defaultFileList={videoFileList}
              fileType={['video/mp4', 'video/avi', 'video/wmv', 'video/rmvb']}
              fileSize={50}
            >
              <Button icon={<UploadOutlined />}>上传视频</Button>
            </Uploader>
          )}
        </Form.Item>
        <Form.Item
          label='商品分类'
          name='categoryId'
          rules={[{ required: true, message: '请选择商品分类' }]}
        >
          <Cascader
            placeholder='请选择商品分类'
            options={categoryList}
            fieldNames={{ label: 'name', value: 'id', children: 'subDTOList' }}
          />
        </Form.Item>
        <Form.Item
          label='商品状态'
          name='shelfStatus'
          rules={[{ required: true, message: '请选择商品状态' }]}
        >
          <Select placeholder='请选择商品状态'>
            <Select.Option value={1}>上架</Select.Option>
            <Select.Option value={0}>下架</Select.Option>
          </Select>
        </Form.Item>
        <Form.Item label='显示价格' name='priceShow'>
          <Radio.Group>
            <Radio value={1}>显示</Radio>
            <Radio value={0}>不显示</Radio>
          </Radio.Group>
        </Form.Item>
        <Form.Item label='显示标签' name='labelShow'>
          <Radio.Group onChange={labelRadioChange}>
            <Radio value={1}>显示</Radio>
            <Radio value={0}>不显示</Radio>
          </Radio.Group>
        </Form.Item>
        {labelShow ? (
          <Form.Item
            label='商品标签'
            name='goodsLabel'
            rules={[{ required: true, message: '请输入商品标签' }]}
          >
            <Input placeholder='请输入商品标签' maxLength={10} />
          </Form.Item>
        ) : (
          ''
        )}
      </Form>
    </div>
  );
});
export default BaseInfo;
