import React, { useEffect, useState } from 'react';
import { InterDataType, InterListType } from '~/api/interface';
import { selectUserListType, systemMessageListByUserType } from '~/api/interface/systemManageType';
import { transDateStr } from '~/utils';
import { isArray } from 'lodash';
import { SystemManageAPI } from '~/api';

// 消息类型
type MessageType = InterDataType<selectUserListType>;

const MessageChatListView: React.FC<{
  messageList: MessageType;
  current: number;
  isSystem: boolean;
  onSelect: (j: number) => void;
  onSystem: () => void;
}> = ({ messageList, current, isSystem, onSelect, onSystem }) => {
  // 获取消息展示信息
  const getMessageShow = (data: MessageType[0]) => {
    // console.log('展示信息 --->', props.detail)
    const { companyInfoVO, userAccountId, userImg, nickName, userName } = data;
    // 用户对商家聊天
    if (companyInfoVO?.id && companyInfoVO?.companyBindUserId === userAccountId) {
      return {
        icon: `${companyInfoVO?.brandLogo}?x-oss-process=image/quality,q_25`,
        name: companyInfoVO?.brandName || companyInfoVO?.companyName || '与商家的聊天',
        label: true,
        labelText: '商家客服',
      };
    }
    // 商家对用户聊天
    if (companyInfoVO?.id && companyInfoVO?.companyBindUserId !== userAccountId) {
      return {
        icon: `${userImg}?x-oss-process=image/quality,q_25`,
        name: nickName || userName || '与用户的聊天',
        label: true,
        labelText: '客服消息',
      };
    }
    // 用户对用户聊天
    return {
      icon: `${userImg}?x-oss-process=image/quality,q_25`,
      name: nickName || userName || '与用户的聊天',
      label: false,
      labelText: undefined,
    };
  };
  // 获取最后一条消息
  const getEndMessage = (data: MessageType[0]) => {
    try {
      return isArray(JSON.parse(data?.endMessage)) ? '[图片]' : data?.endMessage;
    } catch (error) {
      return data?.endMessage || '[其他消息]';
    }
  };
  // 系统消息列表
  const [systemList, setSystemList] = useState<InterListType<systemMessageListByUserType>>();
  // 获取系统消息列表
  const getSystemList = async () => {
    const res = await SystemManageAPI.systemMessageListByUser({
      pageNo: 1,
      pageSize: 1,
    });
    if (res && res.code === '200') {
      setSystemList(res.result?.list || []);
    }
  };
  // 组件挂载
  useEffect(() => {
    getSystemList().then();
  }, []);
  return (
    <div className='message-chat-list'>
      <div
        className={`list-item flex-start ${isSystem && 'item-active'}`}
        onClick={() => onSystem?.()}
      >
        <img
          className='item-image'
          src={`https://file.iuav.com/file/sharefly-message-sys-icon.png`}
          alt='用户头像'
        />
        <div className='item-content'>
          <div className='title text-ellipsis'>系统消息</div>
          <div className='text text-ellipsis'>{systemList?.[0]?.subject}</div>
        </div>
        <div className='item-time'>{transDateStr(systemList?.[0]?.createTime)}</div>
      </div>
      {messageList?.map((i, j) => (
        <div
          className={`list-item flex-start ${current === j && !isSystem && 'item-active'}`}
          key={j}
          onClick={() => onSelect?.(j)}
        >
          <img className='item-image' src={getMessageShow(i)?.icon} alt='用户头像' />
          <div className='item-content'>
            <div className='title text-ellipsis flex-start'>
              <view>{getMessageShow(i)?.name}</view>
              {getMessageShow(i)?.label && (
                <div className='label'>{getMessageShow(i)?.labelText}</div>
              )}
            </div>
            <div className='text flex-between'>
              <div className='text text-ellipsis'>
                {!i?.unreadCount ? getEndMessage(i) : `你有${i?.unreadCount}条消息未读`}
              </div>
              {i?.unreadCount ? <div className='tag'>{i?.unreadCount}</div> : null}
            </div>
          </div>
          <div className='item-time'>{transDateStr(i?.createTime)}</div>
        </div>
      ))}
    </div>
  );
};

export default MessageChatListView;
