import React, { useEffect, useState } from 'react';
import { InterDataType } from '~/api/interface';
import { getCompanyInfoByIdType } from '~/api/interface/systemManageType';
import { CustomManageAPI } from '~/api';
import { Button, Table, Image, Modal, message } from 'antd';
import { PlusOutlined } from '@ant-design/icons';
import AddSolutionModal from '../addSolutionModal/index';
import { getSuggestionType } from '~/api/interface/customManageType';
import { ColumnsType } from 'antd/es/table';

// 店铺类型
type DetailType = InterDataType<getCompanyInfoByIdType>;
// 列表类型
type ListType = InterDataType<getSuggestionType>;

const StoreSolutionView: React.FC<{
  detail: DetailType;
}> = ({ detail }) => {
  // 添加解决方案是否显示
  const [isAddSolution, setIsAddSolution] = useState<boolean>(false);
  // 当前编辑的数据
  const [editData, setEditData] = useState<ListType[0]>();
  // 解决方案列表
  const [solutionList, setSolutionList] = useState<ListType>([]);
  // 获取解决方案列表
  const getSolutionList = async () => {
    const res = await CustomManageAPI.getSuggestion({
      backUserId: detail?.backUserId,
    });
    if (res && res.code === '200') {
      setSolutionList(res.result);
      // console.log('获取解决方案列表 --->', res.result);
    }
  };
  // 删除事件
  const handleDelete = (record: ListType[0]) => {
    Modal.confirm({
      title: '提示',
      content: '删除后将无法恢复，是否确认删除？',
      onOk: async () => {
        const res = await CustomManageAPI.deleteSuggestion({
          id: record?.id,
        });
        if (res && res.code === '200') {
          await getSolutionList();
          message.success('删除成功');
        }
      },
    });
  };
  // 组件挂载
  useEffect(() => {
    if (detail?.backUserId) getSolutionList().then();
    // console.log('storeSolutionView --->', detail);
  }, []);
  const columns: ColumnsType<ListType[0]> = [
    {
      title: '序号',
      dataIndex: 'accountNo',
      align: 'center',
      width: '50px',
      render: (_text, _record, index) => index + 1,
    },
    {
      title: '方案名称',
      dataIndex: 'title',
      align: 'center',
    },
    {
      title: '方案封面',
      dataIndex: 'coverUrl',
      align: 'center',
      render: (text) => <Image src={text} width={35} height={35} />,
    },
    {
      title: '方案文件',
      dataIndex: 'suggestFile',
      align: 'center',
      render: (text) => (
        <a href={text} target={'_blank'} rel='noreferrer'>
          查看
        </a>
      ),
    },
    {
      title: '操作',
      dataIndex: 'action',
      align: 'center',
      render: (_text, record) => (
        <>
          <Button
            type={'link'}
            onClick={() => {
              setEditData(record);
              setIsAddSolution(true);
            }}
          >
            编辑
          </Button>
          <Button type={'link'} onClick={() => handleDelete(record)} danger>
            删除
          </Button>
        </>
      ),
    },
  ];
  return (
    <>
      <div className={'store-title flex-between'}>
        <div className='title'>行业解决方案</div>
        <div className='action'>
          <Button icon={<PlusOutlined />} type={'primary'} onClick={() => setIsAddSolution(true)}>
            添加解决方案
          </Button>
        </div>
      </div>
      <Table
        dataSource={solutionList}
        rowKey={'id'}
        columns={columns}
        size={'small'}
        bordered={true}
      ></Table>
      <AddSolutionModal
        open={isAddSolution}
        detail={detail}
        data={editData}
        title={editData?.id ? '编辑方案' : '添加方案'}
        onCancel={() => {
          setIsAddSolution(false);
          setEditData(undefined);
          getSolutionList().then();
        }}
      />
    </>
  );
};

export default StoreSolutionView;
