import React, { FC, useEffect } from 'react';
import {
  DatePicker,
  Form,
  Input,
  InputNumber,
  message,
  Modal,
  ModalProps,
  Radio,
  Select,
} from 'antd';
import { InterListType } from '~/api/interface';
import { listActivityPagesType } from '~/api/interface/activityManage';
import dayjs from 'dayjs';
import { ActivityManageAPI } from '~/api';

// 参数
interface selfProps {
  data?: InterListType<listActivityPagesType>[0];
}
const AddOrEditModal: FC<ModalProps & selfProps> = ({ open, onCancel, title, data }) => {
  // 表单钩子
  const [editForm] = Form.useForm();
  // 提交数据
  const handleSubmit = async () => {
    const valid = await editForm.validateFields();
    if (!valid) return;
    const res = await ActivityManageAPI[data?.id ? 'activityUpdate' : 'activityInsert']({
      ...data,
      ...valid,
      startTime: dayjs(valid.rangeTime[0]).format('YYYY-MM-DD HH:mm:ss'),
      endTime: dayjs(valid.rangeTime[1]).format('YYYY-MM-DD HH:mm:ss'),
    });
    if (res && res.code === '200') {
      message.success('操作成功');
      handleCancel();
    }
  };
  // 关闭弹窗
  const handleCancel = () => {
    editForm.resetFields();
    onCancel?.({} as any);
  };
  // 组件挂载
  useEffect(() => {
    if (data) {
      editForm.setFieldsValue({
        ...data,
        rangeTime: [dayjs(data.startTime), dayjs(data.endTime)],
      });
    } else {
      editForm.resetFields();
    }
    console.log('组件挂载 --->', data);
  }, [open]);
  return (
    <Modal open={open} onCancel={handleCancel} title={title} onOk={handleSubmit}>
      <Form labelCol={{ span: 4 }} wrapperCol={{ span: 16 }} form={editForm}>
        <Form.Item
          label='活动名称'
          name='activityName'
          rules={[{ required: true, message: '请输入活动名称' }]}
        >
          <Input placeholder='请输入连续签到天数' maxLength={25} allowClear />
        </Form.Item>
        <Form.Item
          label='活动类型'
          name='activityType'
          rules={[{ required: true, message: '请选择活动类型' }]}
        >
          <Select
            placeholder='请选择活动类型'
            options={[{ label: '新用户邀请', value: 0 }]}
            allowClear
          />
        </Form.Item>
        <Form.Item
          label='活动状态'
          name='activityStatus'
          rules={[{ required: true, message: '请选择活动状态' }]}
          initialValue={1}
        >
          <Select
            placeholder='请选择活动状态'
            options={[
              { label: '进行中', value: 1 },
              { label: '已结束', value: 0 },
            ]}
            allowClear
          />
        </Form.Item>
        <Form.Item
          label='活动说明'
          name='description'
          rules={[{ required: true, message: '请输入活动说明' }]}
        >
          <Input.TextArea placeholder='请输入活动说明' maxLength={200} allowClear showCount />
        </Form.Item>
        <Form.Item
          label='活动时间'
          name='rangeTime'
          rules={[{ required: true, message: '请选择活动时间' }]}
        >
          <DatePicker.RangePicker placeholder={['活动开始时间', '活动结束时间']} showTime />
        </Form.Item>
        <Form.Item
          label='奖励类型'
          name='rewardType'
          rules={[{ required: true, message: '请选择奖励类型' }]}
          initialValue={0}
        >
          <Radio.Group
            options={[
              { label: '积分', value: 0 },
              { label: '余额', value: 1 },
              { label: '优惠券', value: 2, disabled: true },
            ]}
          />
        </Form.Item>
        <Form.Item
          label='奖励额度'
          name='rewardValue'
          rules={[
            { required: true, message: '请输入奖励额度' },
            { pattern: /^[0-9]*$/g, message: '请输入整数' },
          ]}
        >
          <InputNumber
            min={1}
            max={100000}
            placeholder='请输入奖励额度'
            style={{ width: '200px' }}
          />
        </Form.Item>
        <Form.Item
          label='人数限制'
          name='requireNum'
          rules={[
            { required: true, message: '请输入人数限制' },
            { pattern: /^[0-9]*$/g, message: '请输入整数' },
          ]}
          initialValue={0}
          extra={'默认0，为不限制'}
        >
          <InputNumber
            min={0}
            max={100000}
            placeholder='请输入人数限制'
            style={{ width: '200px' }}
          />
        </Form.Item>
      </Form>
    </Modal>
  );
};
export default AddOrEditModal;
