package com.mmc.pms.service.mall.impl;

import com.mmc.pms.auth.dto.LoginSuccessDTO;
import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.mall.MallGoodsDao;
import com.mmc.pms.entity.SkuUnitDO;
import com.mmc.pms.entity.mall.GoodsSpecDO;
import com.mmc.pms.entity.mall.GoodsSpecValuesDO;
import com.mmc.pms.entity.mall.MallGoodsDO;
import com.mmc.pms.entity.mall.MallGoodsResourcesDO;
import com.mmc.pms.model.mall.GoodsResourcesVO;
import com.mmc.pms.model.mall.GoodsSpecVO;
import com.mmc.pms.model.mall.GoodsSpecValuesVO;
import com.mmc.pms.model.mall.MallGoodsVO;
import com.mmc.pms.model.sale.dto.SkuUnitDTO;
import com.mmc.pms.model.sale.qo.MallGoodsQO;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.mall.MallGoodsService;
import com.mmc.pms.util.CodeUtil;
import com.mmc.pms.util.SnowFlake;
import com.mmc.pms.util.TDateUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author LW
 * @date 2023/7/24 17:37 概要：
 */
@Service
public class MallGoodsServiceImpl implements MallGoodsService {
    @Resource
    private MallGoodsDao mallGoodsDao;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody addMallGoods(MallGoodsVO mallGoodsVO, Integer userAccountId) {
        ResultBody resultError = checkInformation(mallGoodsVO, userAccountId);
        if (resultError != null) return resultError;
        int count = mallGoodsDao.countMallGoods(userAccountId);
        // 使用雪花算法生成雪花id，用作商品id,及其他id
        SnowFlake snowFlake = new SnowFlake(2, 3);
        long id = snowFlake.nextId();
        mallGoodsVO.setId(id);
        MallGoodsDO mallGoodsDO = new MallGoodsDO(mallGoodsVO);
        mallGoodsDO.setGoodsNo("MG" + TDateUtil.getDateStr(new Date(), "yyyyMMddHHmmss") + CodeUtil.getRandomNum(4));
        mallGoodsDO.setUserAccountId(userAccountId);
        mallGoodsDO.setSort(count + 1);
        // 将基础信息存储入库
        mallGoodsDao.insertMallGoodsBaseInfo(mallGoodsDO);
        // 将商品图片等资源存入数据库中
        insertMallGoodsResources(mallGoodsVO, id);
        // 将商品规格存入数据库
        insertMallGoodsSpec(mallGoodsVO.getGoodsSpecList(), id);
        return ResultBody.success();
    }

    @Transactional(rollbackFor = Exception.class)
    public void insertMallGoodsSpec(List<GoodsSpecVO> goodsSpecList, long id) {
        // 获取输入的规格信息
        for (GoodsSpecVO goodsSpecVO : goodsSpecList) {
            goodsSpecVO.setMallGoodsId(id);
            GoodsSpecDO goodsSpecDO = new GoodsSpecDO(goodsSpecVO);
            // 插入规格信息
            mallGoodsDao.insertGoodsSpec(goodsSpecDO);
            List<GoodsSpecValuesDO> goodsSpecValuesList = goodsSpecVO.getGoodsSpecValuesList().stream().map(d -> {
                d.setGoodsSpecId(goodsSpecDO.getId());
                return new GoodsSpecValuesDO(d);
            }).collect(Collectors.toList());
            // 批量插入规格值的信息
            mallGoodsDao.batchInsertSpecValues(goodsSpecValuesList);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void insertMallGoodsResources(MallGoodsVO mallGoodsVO, long id) {
        List<MallGoodsResourcesDO> mallGoodsResourcesList = mallGoodsVO.getResourcesList().stream().map(d -> {
            MallGoodsResourcesDO mallGoodsResourcesDO = new MallGoodsResourcesDO(d);
            mallGoodsResourcesDO.setMallGoodsId(id);
            return mallGoodsResourcesDO;
        }).collect(Collectors.toList());
        mallGoodsDao.batchInsertMallGoodsResources(mallGoodsResourcesList);
    }

    private ResultBody checkInformation(MallGoodsVO mallGoodsVO, Integer userAccountId) {
        // 查询该账账号下是否有相同的商品名称存在
        if (mallGoodsDao.countMallGoodsByName(mallGoodsVO, userAccountId) > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        // 判断主图是否为空
        List<GoodsResourcesVO> resourcesList = mallGoodsVO.getResourcesList()
                .stream().filter(d -> d.getType().equals(0))
                .collect(Collectors.toList());
        if (resourcesList.size() == 0) {
            return ResultBody.error(ResultEnum.GOODS_PIC_IS_NOT_NULL);
        }
        return null;
    }

    @Override
    public ResultBody<MallGoodsVO> mallGoodsDetails(Long id) {
        MallGoodsDO mallGoodsDO = mallGoodsDao.getMallGoodsBaseInfo(id);
        if (mallGoodsDO == null) {
            return ResultBody.error("商品不存在或已删除！");
        }
        MallGoodsVO mallGoodsVO = mallGoodsDO.buildMallGoodsVO();
        // 获取图片及其他资源信息
        List<MallGoodsResourcesDO> mallGoodsResourcesList = mallGoodsDao.getMallGoodsResources(id);
        mallGoodsVO.setResourcesList(mallGoodsResourcesList.stream()
                .map(MallGoodsResourcesDO::buildGoodsResourcesVO).collect(Collectors.toList()));
        // 获取规格信息
        List<GoodsSpecDO> goodsSpecList = mallGoodsDao.getMallGoodsSpec(id);
        List<GoodsSpecVO> goodsSpec = goodsSpecList.stream().map(GoodsSpecDO::buildGoodsSpecVO)
                .collect(Collectors.toList());
        // 获取规格值信息
        List<GoodsSpecValuesDO> goodsSpecValuesList = mallGoodsDao.getMallGoodsSpecValues(goodsSpec.stream().map(GoodsSpecVO::getId).collect(Collectors.toList()));
        List<GoodsSpecValuesVO> goodsSpecValues = goodsSpecValuesList.stream().map(GoodsSpecValuesDO::buildGoodsSpecValuesVO).collect(Collectors.toList());
        Map<Integer, List<GoodsSpecValuesVO>> goodsSpecValuesVO = goodsSpecValues.stream().collect(Collectors.groupingBy(GoodsSpecValuesVO::getGoodsSpecId));
        for (GoodsSpecVO goodsSpecVO : goodsSpec) {
            goodsSpecVO.setGoodsSpecValuesList(goodsSpecValuesVO.get(goodsSpecVO.getId()));
        }
        mallGoodsVO.setGoodsSpecList(goodsSpec);
        return ResultBody.success(mallGoodsVO);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody editMallGoods(MallGoodsVO mallGoodsVO, Integer userAccountId) {
        ResultBody resultError = checkInformation(mallGoodsVO, userAccountId);
        if (resultError != null) return resultError;
        MallGoodsDO mallGoodsDO = new MallGoodsDO(mallGoodsVO);
        // 修改商城商品的基本信息
        mallGoodsDao.updateMallGoods(mallGoodsDO);
        // 修改商城商品的图片等资源信息,先删除后新增
        mallGoodsDao.deleteMallGoodsResources(mallGoodsVO.getId());
        this.insertMallGoodsResources(mallGoodsVO, mallGoodsVO.getId());
        // 从数据库获取商品规格信息
        List<GoodsSpecDO> dbGoodsSpec = mallGoodsDao.getMallGoodsSpec(mallGoodsVO.getId());
        List<Integer> specIds = mallGoodsVO.getGoodsSpecList().stream().map(GoodsSpecVO::getId).filter(Objects::nonNull).collect(Collectors.toList());
        // 对比要删除的规格
        List<Integer> deleteSpec = dbGoodsSpec.stream().map(GoodsSpecDO::getId).filter(id -> !specIds.contains(id)).collect(Collectors.toList());
        if (deleteSpec.size() != 0) {
            mallGoodsDao.deleteGoodsSpec(deleteSpec);
        }
        // 获取规格值信息
        Map<Integer, List<GoodsSpecValuesDO>> specValuesMap = mallGoodsDao.getMallGoodsSpecValues(specIds).stream()
                .collect(Collectors.groupingBy(GoodsSpecValuesDO::getGoodsSpecId));
        // 修改规格信息
        for (GoodsSpecVO goodsSpecVO : mallGoodsVO.getGoodsSpecList()) {
            if (goodsSpecVO.getId() != null) {
                GoodsSpecDO goodsSpecDO = new GoodsSpecDO(goodsSpecVO);
                mallGoodsDao.updateGoodsSpec(goodsSpecDO);
                // 数据库获取到的规格值信息
                List<GoodsSpecValuesDO> dbSpecValues = specValuesMap.get(goodsSpecVO.getId());
                List<Integer> specValuesId = goodsSpecVO.getGoodsSpecValuesList().stream().map(GoodsSpecValuesVO::getId).filter(Objects::nonNull).collect(Collectors.toList());
                List<Integer> deleteSpecValueId = dbSpecValues.stream().map(GoodsSpecValuesDO::getId).filter(id -> !specValuesId.contains(id)).collect(Collectors.toList());
                if (deleteSpecValueId.size() != 0) {
                    mallGoodsDao.deleteGoodsSpecValues(deleteSpecValueId);
                }
                // 修改规格值的信息
                List<GoodsSpecValuesVO> updateSpecValue = goodsSpecVO.getGoodsSpecValuesList().stream().filter(d -> d.getId() != null).collect(Collectors.toList());
                for (GoodsSpecValuesVO goodsSpecValuesVO : updateSpecValue) {
                    GoodsSpecValuesDO goodsSpecValuesDO = new GoodsSpecValuesDO(goodsSpecValuesVO);
                    mallGoodsDao.updateGoodsSpecValue(goodsSpecValuesDO);
                }
                // 新增id为空的规格值
                List<GoodsSpecValuesVO> newGoodsSpecValues = goodsSpecVO.getGoodsSpecValuesList().stream().filter(d -> d.getId() == null).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(newGoodsSpecValues)) {
                    List<GoodsSpecValuesDO> goodsSpecValuesList = newGoodsSpecValues.stream().map(d -> {
                        d.setGoodsSpecId(goodsSpecVO.getId());
                        return new GoodsSpecValuesDO(d);
                    }).collect(Collectors.toList());
                    // 批量插入规格值的信息
                    mallGoodsDao.batchInsertSpecValues(goodsSpecValuesList);
                }
            }
        }
        // 新增id为空的规格
        List<GoodsSpecVO> newGoodsSpec = mallGoodsVO.getGoodsSpecList().stream().filter(d -> d.getId() == null).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(newGoodsSpec)) {
            this.insertMallGoodsSpec(newGoodsSpec, mallGoodsVO.getId());
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody getSkuUnit() {
        List<SkuUnitDO> skuUnitList = mallGoodsDao.getSkuUnit();
        List<SkuUnitDTO> list =
                skuUnitList.stream().map(SkuUnitDO::buildSkuUnitDTO).collect(Collectors.toList());
        return ResultBody.success(list);
    }


    @Override
    public PageResult listPageGoodsInfo(MallGoodsQO param, LoginSuccessDTO loginSuccessDTO) {
        if (loginSuccessDTO.getRoleInfo().getSuperAdmin().equals(1)) {
            // 超级管理员获取所有商品信息
            return getMallGoodsInfo(param);
        } else {
            // 非超级管理员获取自家的商品信息
            param.setUserAccountId(loginSuccessDTO.getUserAccountId());
            return getMallGoodsInfo(param);
        }
    }

    private PageResult getMallGoodsInfo(MallGoodsQO param) {
        int count;
        List<MallGoodsDO> mallGoodsList;
        count = mallGoodsDao.countListMallGoods(param);
        if (count == 0) {
            return PageResult.buildPage(param.getPageNo(), param.getPageSize(), count);
        }
        Integer pageNo = param.getPageNo();
        param.buildCurrentPage();
        mallGoodsList = mallGoodsDao.listMallGoods(param);
        List<MallGoodsVO> pageList = mallGoodsList.stream().map(MallGoodsDO::buildListMallGoodsVO).collect(Collectors.toList());
        return PageResult.buildPage(pageNo, param.getPageSize(), count, pageList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody exchange(Long firstId, Long secondId) {
        MallGoodsDO firstMallGoodsBaseInfo = mallGoodsDao.getMallGoodsBaseInfo(firstId);
        MallGoodsDO secondMallGoodsBaseInfo = mallGoodsDao.getMallGoodsBaseInfo(secondId);
        int updateCount1 = mallGoodsDao.updateMallGoodsSort(firstId, firstMallGoodsBaseInfo.getSort());
        int updateCount2 = mallGoodsDao.updateMallGoodsSort(secondId, secondMallGoodsBaseInfo.getSort());
        if (updateCount1 == updateCount2) {
            return ResultBody.success();
        } else {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            return ResultBody.error("排序失败");
        }
    }

    @Override
    public ResultBody upOrDownShelf(Long id, Integer status) {
        mallGoodsDao.updateMallGoodsByShelf(id, status);
        return ResultBody.success();
    }

    @Override
    public ResultBody removeMallGoods(Long id) {
        mallGoodsDao.removeMallGoods(id);
        return ResultBody.success();
    }
}
