package com.mmc.pms.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.BackstageTaskServiceDao;
import com.mmc.pms.dao.CategoriesDao;
import com.mmc.pms.dao.DirectoryDao;
import com.mmc.pms.dao.GoodsInfoDao;
import com.mmc.pms.entity.Categories;
import com.mmc.pms.entity.DirectoryDO;
import com.mmc.pms.entity.GoodsInfo;
import com.mmc.pms.entity.ServiceDO;
import com.mmc.pms.feign.ImsAppApi;
import com.mmc.pms.model.categories.dto.*;
import com.mmc.pms.model.categories.vo.ClassifyInfoVO;
import com.mmc.pms.model.categories.vo.CurriculumInfoVO;
import com.mmc.pms.model.categories.vo.DirectoryInfoVO;
import com.mmc.pms.model.categories.vo.RelevantBusinessVO;
import com.mmc.pms.model.sale.vo.QueryClassifyVO;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.BackstageTaskService;
import com.mmc.pms.service.CategoriesService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author lw
 * @description 针对表【categories(通用分类表)】的数据库操作Service实现
 * @createDate 2023-05-24 10:29:28
 */
@Service
@Slf4j
public class CategoriesServiceImpl implements CategoriesService {
  @Autowired private CategoriesDao categoriesDao;
  @Resource private GoodsInfoDao goodsInfoDao;
  @Autowired private DirectoryDao directoryDao;
  @Autowired private BackstageTaskService backstageTaskService;
  @Autowired private BackstageTaskServiceDao backstageTaskServiceDao;
  @Autowired private ImsAppApi imsAppApi;

  @Override
  public ResultBody addOrEditDirectory(DirectoryInfoVO param) {
    int type = categoriesDao.countUpdateDirectoryName(param);
    if (type > 0) {
      return ResultBody.error(ResultEnum.DIRECTORY_NAME_HAS_BEEN_EXIST);
    }
    DirectoryDO directory = new DirectoryDO(param);
    if (param.getId() == null) {
      categoriesDao.insertDirectory(directory);
    } else {
      // 获取默认目录信息
      DirectoryDO directoryDO = categoriesDao.selectDirectoryById(param.getId());
      if (directoryDO != null && !directoryDO.getDirectoryName().equals(param.getDirectoryName())) {
        return ResultBody.error(ResultEnum.DEFAULT_DIRECTORY);
      }
      categoriesDao.updateDirectory(directory);
    }
    return ResultBody.success();
  }

  @Override
  public PageResult directoryList(Integer pageNo, Integer pageSize, Integer type) {
    int count = categoriesDao.countDirectoryList();
    if (count == 0) {
      return PageResult.buildPage(pageNo, pageSize, count);
    }
    List<DirectoryDO> directoryList =
        categoriesDao.directoryList((pageNo - 1) * pageSize, pageSize, type);
    List<DirectoryInfoVO> list =
        directoryList.stream().map(DirectoryDO::buildDirectoryInfoVO).collect(Collectors.toList());
    return PageResult.buildPage(pageNo, pageSize, count, list);
  }

  @Override
  public ResultBody removeDirectory(Integer id) {
    // 查询该目录下是否有分类有的话就不给删除
    int count = categoriesDao.countDirectory(id);
    if (count > 0) {
      return ResultBody.error(ResultEnum.THERE_ARE_CATEGORIES_IN_THE_DIRECTORY);
    }
    DirectoryDO defaultCount = categoriesDao.selectDirectoryById(id);
    if (defaultCount != null) {
      return ResultBody.error(ResultEnum.DEFAULT_DIRECTORY);
    }
    categoriesDao.removeDirectory(id);
    return ResultBody.success();
  }

  @Override
  public ResultBody addClassification(ClassifyInfoVO classifyInfoVO) {
    int count = categoriesDao.countClassificationByName(classifyInfoVO);
    if (count > 0) {
      return ResultBody.error(ResultEnum.WARE_TYPE_NAME_HAS_BEEN_EXIST);
    }
    Categories categories = new Categories(classifyInfoVO);
    if (classifyInfoVO.getPid() == 0) {
      if (classifyInfoVO.getIcon() != null) {
        int typeCount = categoriesDao.getCountCategoriesByPid(0, classifyInfoVO.getType());
        categories.setSort(typeCount + 1);
        categoriesDao.insertClassification(categories);
      } else {
        return ResultBody.error(ResultEnum.WARE_TYPE_ICON_NOT_NULL);
      }
    } else {
      categoriesDao.insertClassification(categories);
    }
    return ResultBody.success();
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public ResultBody updateClassification(ClassifyInfoVO classifyInfoVO) {
    int count = categoriesDao.countClassificationByName(classifyInfoVO);
    if (count > 0) {
      return ResultBody.error(ResultEnum.WARE_TYPE_NAME_HAS_BEEN_EXIST);
    }
    if (classifyInfoVO.getPid() == 0) {
      if (classifyInfoVO.getIcon() != null) {
        categoriesDao.updateClassification(classifyInfoVO);
      } else {
        return ResultBody.error(ResultEnum.WARE_TYPE_ICON_NOT_NULL);
      }
    } else {
      categoriesDao.updateClassification(classifyInfoVO);
    }
    return ResultBody.success();
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public ResultBody exchangeSortType(Integer firstId, Integer secondId) {
    Categories firstCategories = categoriesDao.getGoodsGroupById(firstId);
    Categories secondCategories = categoriesDao.getGoodsGroupById(secondId);
    int updateCount1 = categoriesDao.updateTypeSort(firstId, secondCategories.getSort());
    int updateCount2 = categoriesDao.updateTypeSort(secondId, firstCategories.getSort());
    if (updateCount1 == updateCount2) {
      return ResultBody.success();
    } else {
      TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
      return ResultBody.error("排序失败");
    }
  }

  @Override
  public PageResult getClassificationList(QueryClassifyVO queryClassifyVO) {
    int count = categoriesDao.countListClassification(queryClassifyVO);
    if (count == 0) {
      return PageResult.buildPage(
          queryClassifyVO.getPageNo(), queryClassifyVO.getPageSize(), count);
    }

    int pageNo = queryClassifyVO.getPageNo();
    queryClassifyVO.buildCurrentPage();
    List<Categories> categories = categoriesDao.selectAllClassification(queryClassifyVO);
    List<ClassifyInfoDTO> categoriesList =
        categories.stream().map(Categories::buildClassifyInfoDTO).collect(Collectors.toList());

    List<ClassifyInfoDTO> topLevelCategories = new ArrayList<>();
    Map<Integer, ClassifyInfoDTO> categoriesMap = new HashMap<>();

    // 将每个数据模型对象添加到Map中，以便在递归过程中查找它们的父母
    for (ClassifyInfoDTO category : categoriesList) {
      category.setChildren(new ArrayList<>());
      categoriesMap.put(category.getId(), category);
    }
    // 构建树结构
    for (ClassifyInfoDTO category : categoriesList) {
      if (category.getPid() == 0) {
        topLevelCategories.add(category);
      } else {
        ClassifyInfoDTO parent = categoriesMap.get(category.getPid());
        parent.getChildren().add(category);
      }
    }
    return PageResult.buildPage(pageNo, queryClassifyVO.getPageSize(), count, topLevelCategories);
  }

  @Override
  public ResultBody getClassifyDetails(Integer id) {
    Categories goodsGroup = categoriesDao.getGoodsGroupById(id);
    return ResultBody.success(goodsGroup == null ? null : goodsGroup.buildClassifyDetailsDTO());
  }

  @Override
  public PageResult queryRelevantBusiness(Integer id, Integer type,Integer pageNo,Integer pageSize) {
    RelevantBusinessVO relevantBusinessVO = new RelevantBusinessVO();
    switch (type) {
      case 0:
        int count = goodsInfoDao.countGoodsInfoByCategoryId(id);
        if (count == 0) {
          return PageResult.buildPage(pageNo, pageSize, count);
        }
        List<GoodsInfo> mallGoodsInfo = goodsInfoDao.ListGoodsInfoByCategoryId((pageNo - 1) * pageSize, pageSize,id,0);
        if (CollectionUtils.isNotEmpty(mallGoodsInfo)) {
          relevantBusinessVO.setRelevanceGoodsInfoVOs(
                  mallGoodsInfo.stream()
                  .map(GoodsInfo::buildRelevanceGoodsInfoVO)
                  .collect(Collectors.toList()));
          return PageResult.buildPage(pageNo,pageSize,count,relevantBusinessVO);
        }
        break;
      case 1:
        int countOne = goodsInfoDao.countGoodsInfoByCategoryId(id);
        if (countOne == 0) {
          return PageResult.buildPage(pageNo, pageSize, countOne);
        }
        List<GoodsInfo> leaseTermGoodsInfo = goodsInfoDao.ListGoodsInfoByCategoryId((pageNo - 1) * pageSize, pageSize,id,1);
        if (CollectionUtils.isNotEmpty(leaseTermGoodsInfo)) {
          relevantBusinessVO.setRelevanceGoodsInfoVOs(
                  leaseTermGoodsInfo.stream()
                          .map(GoodsInfo::buildRelevanceGoodsInfoVO)
                          .collect(Collectors.toList()));
          return PageResult.buildPage(pageNo,pageSize,countOne,relevantBusinessVO);
        }
        break;
        case 2:
          int countTwo = goodsInfoDao.countServiceByCategoryId(id);
          if (countTwo == 0) {
            return PageResult.buildPage(pageNo, pageSize, countTwo);
          }
          List<ServiceDO> serviceList = backstageTaskServiceDao.queryWorkInfoByCategoryId((pageNo - 1) * pageSize,pageSize,id);
          if (CollectionUtils.isNotEmpty(serviceList)) {
            relevantBusinessVO.setRelevanceServiceInfoVOs(
                    serviceList.stream().map(ServiceDO :: buildRelevanceServiceInfoVO)
                            .collect(Collectors.toList()));
            return PageResult.buildPage(pageNo,pageSize,countTwo,relevantBusinessVO);
          }
          break;
        case 3 :
          int countThree = imsAppApi.feignCountCurriculumInfo(id);
          if (countThree == 0){
            return PageResult.buildPage(pageNo, pageSize, countThree);
          }
          List<CurriculumInfoVO> curriculumList = imsAppApi.feignCurriculumInfoList(id);
          if(CollectionUtils.isNotEmpty(curriculumList)){
            relevantBusinessVO.setRelevanceCurriculumVOs(curriculumList
                    .stream().skip((pageNo - 1) * pageSize).limit(pageSize)
                    .map(CurriculumInfoVO::buildRelevanceCurriculumVO)
                    .collect(Collectors.toList()));
            return PageResult.buildPage(pageNo, pageSize, countThree,relevantBusinessVO);
          }
          break;
      default:
        throw new RuntimeException("输入类型有误！");
    }
    return PageResult.buildPage(pageNo,pageSize,0);
  }

  @Override
  public ResultBody getDirectoryList(Integer type) {
    List<DirectoryDO> directoryList = categoriesDao.getDirectoryList(type);
    List<DirectoryInfoVO> list =
        directoryList.stream().map(DirectoryDO::buildDirectoryInfoVO).collect(Collectors.toList());
    return ResultBody.success(list);
  }

  @Override
  public List<Categories> getCategoriesListByDirectoryName(String directoryName) {
    DirectoryDO directoryDO = directoryDao.getDirectoryByName(directoryName);
    List<Categories> categories = categoriesDao.getCategoriesByDirectoryId(directoryDO.getId());
    return categories;
  }

  @Override
  public ResultBody getApplicationList(String directoryName) {
    List<Categories> categories = getCategoriesListByDirectoryName(directoryName);
    if (org.springframework.util.CollectionUtils.isEmpty(categories)) {
      return ResultBody.success();
    }
    List<CategoriesDTO> collect =
        categories.stream().map(CategoriesDTO::new).collect(Collectors.toList());
    return ResultBody.success(collect);
  }

  @Override
  public List<Categories> getCategoriesListByIds(Set<Integer> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return null;
    }
    return categoriesDao.getCategoriesListByIds(ids);
  }

  @Override
  public List<AllCategoryDTO> feigQqueryCategoryInfoByType(Integer type) {
    List<DirectoryDO> categoryDirectoryList = getCategoryDirectoryList(type);
    if (CollectionUtils.isNotEmpty(categoryDirectoryList)) {
      List<AllCategoryDTO> allCategoryDTOList =
          categoryDirectoryList.stream()
              .map(DirectoryDO::buildAllCategoryDTO)
              .collect(Collectors.toList());
      Map<Integer, List<CategoriesInfoListDTO>> categoryMap = getCategoryMap(allCategoryDTOList);
      addSubCategories(allCategoryDTOList, categoryMap);
      log.info("Res: " + JSONObject.toJSON(allCategoryDTOList));
      return allCategoryDTOList;
    }
    return null;
  }

  @Override
  public ResultBody deleteRelevantBusiness(Integer id) {
    int count = goodsInfoDao.countGoodsInfoByCategoryId(id);
    if (count > 0) {
      return ResultBody.error(ResultEnum.GROUP_DONT_DELETE);
    }
    Integer taskCount =
        backstageTaskService.getWorkServiceCountByCategoriesId(Collections.singletonList(id));
    if (taskCount > 0) {
      return ResultBody.error(ResultEnum.GROUP_DONT_DELETE);
    }
    List<CurriculumInfoVO> curriculumInfo = imsAppApi.feignCurriculumInfoList(id);
    if (CollectionUtils.isNotEmpty(curriculumInfo)) {
      return ResultBody.error(ResultEnum.GROUP_DONT_DELETE);
    }
    // 获取分类信息
    Categories categories = categoriesDao.getGoodsGroupById(id);
    if (categories != null && categories.getParentId().equals(0)) {
      int childCount = categoriesDao.countChildById(id);
      if (childCount > 0) {
        return ResultBody.error(ResultEnum.GROUP_DONT_DELETE_BY_CHILD);
      }
    }
    categoriesDao.deleteById(id);
    return ResultBody.success();
  }

  @Override
  public ResultBody queryCategoryInfoByType(Integer type) {
    List<DirectoryDO> categoryDirectoryList = getCategoryDirectoryList(type);
    if (CollectionUtils.isNotEmpty(categoryDirectoryList)) {
      List<AllCategoryDTO> allCategoryDTOList =
          categoryDirectoryList.stream()
              .map(DirectoryDO::buildAllCategoryDTO)
              .collect(Collectors.toList());
      Map<Integer, List<CategoriesInfoListDTO>> categoryMap = getCategoryMap(allCategoryDTOList);
      addSubCategories(allCategoryDTOList, categoryMap);
      return ResultBody.success(allCategoryDTOList);
    }
    return ResultBody.success();
  }

  private List<DirectoryDO> getCategoryDirectoryList(Integer type) {
    return categoriesDao.getDirectoryList(type);
  }

  private Map<Integer, List<CategoriesInfoListDTO>> getCategoryMap(
      List<AllCategoryDTO> allCategoryDTOList) {
    List<Integer> directoryIdIds =
        allCategoryDTOList.stream()
            .map(AllCategoryDTO::getDirectoryId)
            .collect(Collectors.toList());
    List<Categories> categoriesList = categoriesDao.selectCategoryByDirectoryId(directoryIdIds);
    List<CategoriesInfoListDTO> list = categoriesList.stream()
            .map(Categories::buildCategoriesInfoListDTO)
            .collect(Collectors.toList());
    List<Integer> categoryId = list.stream().map(CategoriesInfoListDTO::getId).collect(Collectors.toList());
    for (Integer id : categoryId) {
      List<Categories> subClassifyList = categoriesDao.getSubClassifyList(id);
      if (CollectionUtils.isNotEmpty(subClassifyList)){
        List<CategoriesInfoListDTO> childList =
            subClassifyList.stream()
                .map(Categories::buildCategoriesInfoListDTO)
                .collect(Collectors.toList());
        for (CategoriesInfoListDTO categories : list) {
          if (id.equals(categories.getId())){
            categories.setChildren(childList);
            break;
          }
        }
      }
    }
    return list.stream().collect(Collectors.groupingBy(CategoriesInfoListDTO::getDirectoryId));
  }

  private void addSubCategories(
      List<AllCategoryDTO> allCategoryDTOList,
      Map<Integer, List<CategoriesInfoListDTO>> categoryMap) {
    for (AllCategoryDTO allCategoryDTO : allCategoryDTOList) {
      Optional.ofNullable(categoryMap.get(allCategoryDTO.getDirectoryId()))
          .ifPresent(allCategoryDTO::setCategoriesInfoListDTO);
    }
  }

  @Override
  public ResultBody<List<ClassifyDetailsDTO>> getSubClassifyList(Integer id) {
    List<Categories> categoriesList = categoriesDao.getSubClassifyList(id);
    if (CollectionUtils.isNotEmpty(categoriesList)){
      List<ClassifyDetailsDTO> list = categoriesList.stream().map(Categories::buildClassifyDetailsDTO).collect(Collectors.toList());
      return ResultBody.success(list);
    }
    return ResultBody.success();
  }
}
