package com.mmc.pms.service.lease.impl;

import com.mmc.pms.auth.dto.LoginSuccessDTO;
import com.mmc.pms.common.ResultBody;
import com.mmc.pms.dao.lease.LeaseGoodsDao;
import com.mmc.pms.entity.lease.*;
import com.mmc.pms.model.lease.qo.LeaseGoodsQO;
import com.mmc.pms.model.lease.vo.LeaseGoodsVO;
import com.mmc.pms.model.lease.vo.LeasePartsListVO;
import com.mmc.pms.model.lease.vo.LeasePriceStockVO;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.lease.LeaseGoodsService;
import com.mmc.pms.service.mall.MallGoodsService;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @Author LW
 * @date 2023/8/4 10:37
 * 概要：
 */
@Service
public class LeaseGoodsServiceImpl implements LeaseGoodsService {
    @Resource
    MallGoodsService mallGoodsService;
    @Resource
    LeaseGoodsDao leaseGoodsDao;

    @Override
    public ResultBody addLeaseGoods(LeaseGoodsVO leaseGoodsVO, Integer userAccountId) {
        ResultBody resultError = mallGoodsService.checkInformation(leaseGoodsVO, null, userAccountId);
        if (resultError != null) return resultError;
        int count = leaseGoodsDao.countLeaseGoodsByUserAccountId(userAccountId);
        LeaseGoodsDO leaseGoodsDO = new LeaseGoodsDO(leaseGoodsVO);
        leaseGoodsDO.setUserAccountId(userAccountId);
        leaseGoodsDO.setSort(count + 1);
        // 将租赁商品基本信息存储入数据库
        leaseGoodsDao.insertLeaseGoodsBaseInfo(leaseGoodsDO);
        // 将商品图片等资源存入数据库中
        mallGoodsService.insertMallGoodsResources(leaseGoodsVO, null, leaseGoodsDO.getId());
        // 将商品规格存入数据库
        mallGoodsService.addMallGoodsSpec(leaseGoodsVO.getSpecAttrList(), leaseGoodsDO.getId(), null, leaseGoodsVO.getPriceStock(), 1);
        // 将商品清单入库
        insertLeasePartsList(leaseGoodsVO.getLeasePartsList(), leaseGoodsDO.getId());
        return ResultBody.success();
    }

    @Transactional(rollbackFor = Exception.class)
    public void insertLeasePartsList(List<LeasePartsListVO> leasePartsList, Integer id) {
        if (CollectionUtils.isNotEmpty(leasePartsList)) {
            List<LeasePartsListDO> list = leasePartsList.stream().map(d -> {
                LeasePartsListDO leasePartsListDO = new LeasePartsListDO(d);
                leasePartsListDO.setLeaseGoodsId(id);
                return leasePartsListDO;
            }).collect(Collectors.toList());
            leaseGoodsDao.batchInsertLeaseParts(list);
        }
    }

    @Override
    public ResultBody getLeaseTermInfo() {
        return ResultBody.success(leaseGoodsDao.getLeaseTermInfo());
    }

    @Override
    public ResultBody leaseGoodsDetails(Integer id) {
        LeaseGoodsDO leaseGoodsDO = leaseGoodsDao.getLeaseGoodsBaseInfo(id);
        if (leaseGoodsDO == null) {
            return ResultBody.error("商品不存在或已删除！");
        }
        LeaseGoodsVO leaseGoodsVO = leaseGoodsDO.buildLeaseGoodsVO();
        // 获取图片及其他资源信息
        List<LeaseGoodsResourcesDO> leaseGoodsResourcesList = leaseGoodsDao.getLeaseGoodsResources(id);
        leaseGoodsVO.setResourcesList(leaseGoodsResourcesList.stream()
                .map(LeaseGoodsResourcesDO::buildGoodsResourcesVO).collect(Collectors.toList()));
        // 获取规格及规格值
        List<LeaseSpecAttrDO> specAttrDOList = leaseGoodsDao.listSpecAttr(id);
        leaseGoodsVO.setSpecAttrList(specAttrDOList.stream().map(LeaseSpecAttrDO::buildSpecAttrVO).collect(Collectors.toList()));
        // 获取价格库存排列组合
        List<LeasePriceStockDO> priceStockDOList = leaseGoodsDao.listPriceStock(id);
        leaseGoodsVO.setPriceStock(priceStockDOList.stream().map(LeasePriceStockDO::buildPriceStockVO).collect(Collectors.toList()));
        // 获取商品清单信息
        List<LeasePartsListDO> leasePartsDOList = leaseGoodsDao.listLeasePartsListDO(id);
        leaseGoodsVO.setLeasePartsList(CollectionUtils.isEmpty(leasePartsDOList) ? null :
                leasePartsDOList.stream().map(LeasePartsListDO::buildLeasePartsListVO).collect(Collectors.toList()));
        return ResultBody.success(leaseGoodsVO);
    }

    @Override
    public ResultBody editLeaseGoods(LeaseGoodsVO leaseGoodsVO, Integer userAccountId) {
        ResultBody resultError = mallGoodsService.checkInformation(leaseGoodsVO, null, userAccountId);
        if (resultError != null) return resultError;
        LeaseGoodsDO leaseGoodsDO = new LeaseGoodsDO(leaseGoodsVO);
        // 修改商品基本信息
        leaseGoodsDao.updateLeaseGoodsBaseInfo(leaseGoodsDO);
        // 修改商城商品的图片等资源信息,先删除后新增
        leaseGoodsDao.deleteLeaseGoodsResources(leaseGoodsVO.getId());
        mallGoodsService.insertMallGoodsResources(leaseGoodsVO, null, leaseGoodsVO.getId());
        // 修改商品清单
        leaseGoodsDao.deleteLeasePartsList(leaseGoodsVO.getId());
        if (CollectionUtils.isNotEmpty(leaseGoodsVO.getLeasePartsList())) {
            this.insertLeasePartsList(leaseGoodsVO.getLeasePartsList(), leaseGoodsVO.getId());
        }
        // 修改规格
        mallGoodsService.updateSpecInfo(null, leaseGoodsVO);
        return ResultBody.success();
    }

    @Override
    public ResultBody leaseGoodsList(LeaseGoodsQO param, LoginSuccessDTO loginSuccessDTO) {
        if (loginSuccessDTO == null || loginSuccessDTO.getRoleInfo().getSuperAdmin().equals(1)) {
            // 小程序和超级管理员获取所有商品信息
            return ResultBody.success(getLeaseGoodsInfo(param));
        } else {
            // 非超级管理员获取自家的商品信息
            param.setUserAccountId(loginSuccessDTO.getUserAccountId());
            return ResultBody.success(getLeaseGoodsInfo(param));
        }
    }

    private PageResult getLeaseGoodsInfo(LeaseGoodsQO param) {
        int count = leaseGoodsDao.countLeaseGoods(param);
        if (count == 0) {
            return PageResult.buildPage(param.getPageNo(), param.getPageSize(), count);
        }
        Integer pageNo = param.getPageNo();
        param.buildCurrentPage();
        List<LeaseGoodsDO> leaseGoodsDOList = leaseGoodsDao.listLeaseGoods(param);
        // 转成vo
        List<LeaseGoodsVO> leaseGoodsVOList = leaseGoodsDOList.stream().map(LeaseGoodsDO::buildLeaseGoodsVO)
                .skip(param.getPageNo()).limit(param.getPageSize()).collect(Collectors.toList());
        leaseGoodsVOList = leaseGoodsVOList.stream().peek(d -> {
            // 获取最低押金及最高押金
            List<LeasePriceStockVO> list = d.getPriceStock().stream().sorted(Comparator.comparing(LeasePriceStockVO::getCashPledge)).collect(Collectors.toList());
            LeasePriceStockVO lastElement = list.get(list.size() - 1);
            LeasePriceStockVO firstElement = list.get(0);
            d.setCashPledgeRange("￥" + firstElement.getCashPledge() + "~" + "￥" + lastElement.getCashPledge());
            // 获取最低租金
            BigDecimal minPrice = list.stream()
                    .map(o -> Stream.of(o.getThreeDaysRental(), o.getSevenDaysRental(), o.getThirtyDaysRental(), o.getNinetyDaysRental(), o.getMaxDaysRental())
                            .filter(Objects::nonNull)
                            .min(BigDecimal::compareTo)
                            .orElse(BigDecimal.ZERO))
                    .min(Comparator.naturalOrder()).get();
            // 获取最高租金
            BigDecimal maxPrice = list.stream()
                    .map(o -> Stream.of(o.getThreeDaysRental(), o.getSevenDaysRental(), o.getThirtyDaysRental(), o.getNinetyDaysRental(), o.getMaxDaysRental())
                            .filter(Objects::nonNull)
                            .max(BigDecimal::compareTo)
                            .orElse(BigDecimal.ZERO))
                    .max(Comparator.naturalOrder()).get();
            d.setRentalRange("￥" + minPrice + "~" + "￥" + maxPrice);
            d.setStock(list.stream().mapToInt(o -> o.getStock() == null ? 0 : o.getStock()).sum());
        }).collect(Collectors.toList());
        return PageResult.buildPage(pageNo, param.getPageSize(), count, leaseGoodsVOList);
    }

    @Override
    public ResultBody batchOnShelfOrTakeDown(List<Integer> goodsIds, Integer status) {
        leaseGoodsDao.batchOnShelfOrTakeDown(goodsIds, status);
        return ResultBody.success();
    }

    @Override
    public ResultBody batchRemoveWareInfo(List<Integer> list) {
        leaseGoodsDao.removeWareInfo(list);
        return ResultBody.success();
    }
}
