package com.mmc.pms.service.Impl;

import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.ProductSkuDao;
import com.mmc.pms.entity.ProductSkuDO;
import com.mmc.pms.entity.ProductSpecDO;
import com.mmc.pms.entity.ProductSpecPriceDO;
import com.mmc.pms.model.dto.*;
import com.mmc.pms.model.qo.ProductSkuQO;
import com.mmc.pms.model.vo.ProductSpecCPQVO;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.ProductSkuService;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author 23214
 * @description 针对表【product_sku(产品sku表)】的数据库操作Service实现
 * @createDate 2023-05-25 14:55:56
 */
@Service
public class ProductSkuServiceImpl implements ProductSkuService {
    @Autowired
    private ProductSkuDao productSkuDao;

    @Override
    public ResultBody addProductSku(ProductSkuVO param) {
        // 获取名称判断此前是否已经存在
        int count = productSkuDao.countSkuName(param);
        if (count > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        ProductSkuDO productSkuDO = new ProductSkuDO(param);
        // 新增产品sku
        int status = productSkuDao.insertProductSku(productSkuDO);
        if (status <= 0) {
            return ResultBody.error(ResultEnum.FAILED_TO_ADD_DATA);
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody getProductSkuDetail(Integer id) {
        // 校验此sku是否还存在或已删除
        int count = productSkuDao.countSkuIsExist(id);
        if (count <= 0) {
            return ResultBody.error(ResultEnum.SKU_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
        }
        ProductSkuDO productSkuDO = productSkuDao.getProductSkuDetail(id);
        return ResultBody.success(productSkuDO.buildProductSkuDTO());
    }

    @Override
    public ResultBody editProductSku(ProductSkuVO param) {
        // 获取名称判断此前是否已经存在
        int count = productSkuDao.countSkuName(param);
        if (count > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        ProductSkuDO productSkuDO = new ProductSkuDO(param);
        int status = productSkuDao.updateProductSku(productSkuDO);
        if (status <= 0) {
            return ResultBody.error(ResultEnum.FAILED_TO_EDIT_DATA);
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody listPageProductSku(ProductSkuQO productSkuQO) {
        int count = productSkuDao.countListPageProductSku(productSkuQO);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(productSkuQO.getPageNo(), productSkuQO.getPageSize(), count));
        }
        Integer pageNo = productSkuQO.getPageNo();
        productSkuQO.buildCurrentPage();
        List<ProductSkuDO> productSkuList = productSkuDao.listPageProductSku(productSkuQO);
        List<ProductSkuDTO> list = productSkuList.stream().map(ProductSkuDO::buildProductSkuDTO).collect(Collectors.toList());
        return ResultBody.success(PageResult.buildPage(pageNo, productSkuQO.getPageSize(), count, list));
    }

    @Override
    public ResultBody addOrEditProductSpec(ProductSpecVO param) {
        // 获取名称判断此前是否已经存在
        int count = productSkuDao.countSpecName(param);
        if (count > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        if (param.getId() == null) {
            ProductSpecDO productSpecDO = new ProductSpecDO(param);
            // 新增产品sku
            int status = productSkuDao.insertProductSpec(productSpecDO);
            if (status <= 0) {
                return ResultBody.error(ResultEnum.FAILED_TO_ADD_DATA);
            }
        } else {
            ProductSpecDO productSpecDO = new ProductSpecDO(param);
            int status = productSkuDao.updateProductSpec(productSpecDO);
            if (status <= 0) {
                return ResultBody.error(ResultEnum.FAILED_TO_EDIT_DATA);
            }
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody getProductSpecDetail(Integer id) {
        // 校验此sku是否还存在或已删除
        int count = productSkuDao.countSpecIsExist(id);
        if (count <= 0) {
            return ResultBody.error(ResultEnum.SPEC_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
        }
        ProductSpecDO productSpecDO = productSkuDao.getProductSpecDetail(id);
        return ResultBody.success(productSpecDO.buildProductSpecDTO());
    }

    @Override
    public ResultBody listPageProductSpec(Integer pageNo, Integer pageSize, Integer productSkuId) {
        int count = productSkuDao.countListPageProductSpec(productSkuId);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(pageNo, pageSize, count));
        }
        List<ProductSpecDO> productSpecList = productSkuDao.listPageProductSpec((pageNo - 1) * pageSize, pageSize, productSkuId);
        List<ProductSpecDTO> list = productSpecList.stream().map(ProductSpecDO::buildProductSpecDTO).collect(Collectors.toList());
        return ResultBody.success(PageResult.buildPage(pageNo, pageSize, count, list));
    }

    @Override
    public ResultBody productSpecCPQ(ProductSpecCPQVO productSpecCPQVO) {
        // 判断该规格是否存在
        int count = productSkuDao.countSpecIsExist(productSpecCPQVO.getProductSpecId());
        if (count <= 0) {
            return ResultBody.error(ResultEnum.SPEC_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
        }
        // 批量插入规格销售或租赁价格
        return insertSpecPrice(productSpecCPQVO);
    }

    @Override
    public List<ProductSpecPriceDO> getProductSpecPriceDOS(ProductSpecCPQVO productSpecCPQVO) {
        // 批量插入规格价格
        return productSpecCPQVO.getSpecPrice().stream().map(d -> {
            ProductSpecPriceDO productSpecPriceDO = new ProductSpecPriceDO();
            productSpecPriceDO.setCooperationTag(d.getCooperationTag());
            productSpecPriceDO.setPrice(d.getPrice());
            productSpecPriceDO.setType(productSpecCPQVO.getType());
            productSpecPriceDO.setProductSpecId(productSpecCPQVO.getProductSpecId());
            if (productSpecCPQVO.getType().equals(1)) {
                productSpecPriceDO.setLeaseTerm(productSpecCPQVO.getLeaseTerm());
            }
            return productSpecPriceDO;
        }).collect(Collectors.toList());
    }

    @Override
    public ResultBody updateProductSpecCPQ(ProductSpecCPQVO productSpecCPQVO) {
        // 先删除原来该规格下的所有价格配置信息
        productSkuDao.removeProductSpecCPQ(productSpecCPQVO);
        // 批量插入规格销售或租赁价格
        return insertSpecPrice(productSpecCPQVO);
    }

    @NotNull
    public ResultBody insertSpecPrice(ProductSpecCPQVO productSpecCPQVO) {
        List<ProductSpecPriceDO> list = getProductSpecPriceDOS(productSpecCPQVO);
        // 批量插入规格销售价格
        if (productSpecCPQVO.getType().equals(0)) {
            int status = productSkuDao.batchInsertSpecPrice(list);
            if (status <= 0) {
                return ResultBody.error(ResultEnum.FAILED_TO_ADD_DATA);
            }
        } else {
            // 批量插入租赁价格
            productSkuDao.batchInsertLeaseSpecPrice(list);
        }
        return ResultBody.success();
    }


    @Override
    public ResultBody getProductSpecCPQ(ProductSpecCPQVO productSpecCPQVO) {
        if (productSpecCPQVO.getType().equals(1) && productSpecCPQVO.getLeaseTerm() == null) {
            return ResultBody.error("租赁期限不能为空！");
        }
        List<ProductSpecPriceDO> productSpecPriceList = productSkuDao.getProductSpecPrice(productSpecCPQVO);
        List<ProductSpecPriceDTO> list = productSpecPriceList.stream()
                .map(ProductSpecPriceDO::buildProductSpecPriceDTO).collect(Collectors.toList());
        return ResultBody.success(list);
    }
}




