package com.mmc.pms.service.Impl;

import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.BrandManageDao;
import com.mmc.pms.dao.IndustrySpecDao;
import com.mmc.pms.dao.ProductDao;
import com.mmc.pms.entity.*;
import com.mmc.pms.model.dto.*;
import com.mmc.pms.model.qo.IndustrySkuQO;
import com.mmc.pms.model.vo.GoodsProductSkuVO;
import com.mmc.pms.model.vo.IndustrySkuVO;
import com.mmc.pms.model.vo.IndustrySpecVO;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.IndustrySpecService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @Author LW
 * @date 2022/10/8 10:58
 * 概要：
 */
@Service
public class IndustrySpecServiceImpl implements IndustrySpecService {
    @Autowired
    IndustrySpecDao industrySpecDao;
    @Autowired
    ProductDao productDao;

    @Autowired
    private BrandManageDao brandManageDao;

    @Override
    public ResultBody addIndustrySku(IndustrySkuVO param) {
        // 获取名称判断此前是否已经存在
        int count = industrySpecDao.countSkuName(param);
        if (count > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        IndustrySku industrySku = new IndustrySku(param);
        // 新增产品sku
        int status = industrySpecDao.insertIndustrySku(industrySku);
        if (status <= 0) {
            return ResultBody.error(ResultEnum.FAILED_TO_ADD_DATA);
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody getIndustrySkuDetail(Integer id) {
        // 判断此sku是否还存在
        int count = industrySpecDao.countIndustrySkuById(id);
        if (count <= 0) {
            return ResultBody.error(ResultEnum.SKU_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
        }
        IndustrySku IndustrySku = industrySpecDao.getIndustrySkuById(id);
        return ResultBody.success(IndustrySku.buildIndustrySku());
    }

    @Override
    public ResultBody editIndustrySku(IndustrySkuVO param) {
        // 判断此sku是否还存在
        int count = industrySpecDao.countIndustrySkuById(param.getId());
        if (count <= 0) {
            return ResultBody.error(ResultEnum.SKU_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
        }
        // 获取名称判断此前是否已经存在
        int skuNameCount = industrySpecDao.countSkuName(param);
        if (skuNameCount > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        IndustrySku IndustrySku = new IndustrySku(param);
        // 新增产品sku
        int status = industrySpecDao.updateIndustrySku(IndustrySku);
        if (status <= 0) {
            return ResultBody.error(ResultEnum.FAILED_TO_EDIT_DATA);
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody listPageIndustrySku(IndustrySkuQO param) {
        int count = industrySpecDao.countListPageIndustrySku(param);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(param.getPageNo(), param.getPageSize(), count));
        }
        Integer pageNo = param.getPageNo();
        param.buildCurrentPage();
        List<IndustrySku> industrySkuList = industrySpecDao.listPageIndustrySku(param);
        return ResultBody.success(PageResult.buildPage(pageNo, param.getPageSize(), count,
                industrySkuList.stream()
                        .map(IndustrySku::buildIndustrySkuDTO)
                        .collect(Collectors.toList())));
    }

    /**
     * 新增方案规格
     *
     * @param param
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody addIndustrySpec(IndustrySpecVO param) {
        // 获取名称判断此前是否已经存在
        int count = industrySpecDao.countSpecName(param);
        if (count > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        // 新增规格表信息
        IndustrySpecDO industrySpecDO = new IndustrySpecDO(param);
        int status = industrySpecDao.insertIndustrySpec(industrySpecDO);
        if (status <= 0) {
            return ResultBody.error(ResultEnum.FAILED_TO_ADD_DATA);
        }
        // 新增规格绑定产品sku表信息
        addInventorySpec(param, industrySpecDO);
        return ResultBody.success();
    }

    private void addInventorySpec(IndustrySpecVO param, IndustrySpecDO industrySpecDO) {
        for (ProductInventoryVO d : param.getProductInventoryList()) {
            IndustryProductInventoryDO industryProductInventoryDO = new IndustryProductInventoryDO(d);
            industryProductInventoryDO.setIndustrySpecId(industrySpecDO.getId());
            industrySpecDao.insertIndustryProductInventory(industryProductInventoryDO);
            // 新增清单绑定的产品规格信息
            // 判断该sku是否指定产品规格 0:未指定 1：指定
            if (d.getSelect().equals(0)) {
                // 如果是表示不指定则要添加所有对sku下的所有产品规格
                // 先查出该sku下的所有规格信息
                List<ProductSpecDO> productSpecList = productDao.listProductSpec(d.getProductSku().getId());
                // 批量插入
                for (ProductSpecDO productSpecDO : productSpecList) {
                    InventorySpecDO inventorySpecDO = new InventorySpecDO();
                    inventorySpecDO.setIndustryProductInventoryId(industryProductInventoryDO.getId());
                    inventorySpecDO.setProductSpecId(productSpecDO.getId());
                    inventorySpecDO.setProductSkuId(productSpecDO.getProductSkuId());
                    industrySpecDao.insertInventorySpec(inventorySpecDO);
                }
            } else {
                // 指定的直接拿指定的数据
                for (ProductSpecVO productSpecVO : d.getProductSpecList()) {
                    InventorySpecDO inventorySpecDO = new InventorySpecDO();
                    inventorySpecDO.setIndustryProductInventoryId(industryProductInventoryDO.getId());
                    inventorySpecDO.setProductSpecId(productSpecVO.getId());
                    inventorySpecDO.setProductSkuId(productSpecVO.getProductSkuId());
                    industrySpecDao.insertInventorySpec(inventorySpecDO);
                }
            }
        }
    }

    /**
     * 获取行业规格信息
     *
     * @param industrySpecId
     * @return
     */
    @Override
    public ResultBody getIndustrySpecDetail(Integer industrySpecId) {
        // 查询当前行业规格是否存在
        int count = industrySpecDao.countIndustrySpec(industrySpecId);
        if (count <= 0) {
            return ResultBody.error(ResultEnum.SPEC_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
        }
        // 初始化行业规格DTO
        IndustrySpecDTO industrySpecDTO;
        // 获取行业规格的基本信息
        IndustrySpecDO industrySpec = industrySpecDao.getIndustrySpecById(industrySpecId);
        industrySpecDTO = industrySpec.buildIndustrySpecDTO();
        // 获取这个规格下的产品sku信息
        List<IndustryProductInventoryDO> industryProductInventory = industrySpecDao.getIndustryProductInventory(industrySpecId);
        // 获取产品skuId
        List<Integer> productSkuId = industryProductInventory.stream().map(IndustryProductInventoryDO::getProductSkuId).collect(Collectors.toList());
        // 分别查出产品sku的详细信息
        List<ProductSkuDO> productSkuList = productDao.listProductSkuDO(productSkuId);
        // 查出产品sku对应的品牌的信息
        Set<Integer> brandInfoIds = productSkuList.stream().map(ProductSkuDO::getBrandInfoId).collect(Collectors.toSet());
        List<BrandInfoDO> brandInfoList = brandManageDao.listBrandInfoByIds(brandInfoIds);
        // 拿到行业产品清单id的集合
        List<Integer> industryProductInventoryIds = industryProductInventory.stream().map(IndustryProductInventoryDO::getId).collect(Collectors.toList());
        // 再分别查出产品规格id
        List<InventorySpecDO> inventorySpecList = productDao.listInventorySpecInfo(industryProductInventoryIds);
        if (inventorySpecList.size() != 0) {
            // 获取所有产品规格id
            List<Integer> productSpecIds = inventorySpecList.stream().map(InventorySpecDO::getProductSpecId).collect(Collectors.toList());
            // 根据这些规格id查询规格信息
            List<ProductSpecDO> productSpecInfoList = productDao.listProductSpecInfo(productSpecIds);
            // 构建产品规格信息
            List<ProductSpecVO> productSpecList = productSpecInfoList.stream().map(d -> {
                ProductSpecVO productSpecVO = new ProductSpecVO();
                productSpecVO.setSpecImage(d.getSpecImage());
                productSpecVO.setProductSkuId(d.getProductSkuId());
                productSpecVO.setSpecName(d.getSpecName());
                productSpecVO.setId(d.getId());
                productSpecVO.setPartNo(d.getPartNo());
                productSpecVO.setVersionDesc(d.getVersionDesc());
                return productSpecVO;
            }).collect(Collectors.toList());
            // 根据产品skuId分组
            Map<Integer, List<ProductSpecVO>> maps = productSpecList.stream().collect(Collectors.groupingBy(ProductSpecVO::getProductSkuId));
            // 构建其余产品清单的信息
            List<ProductInventoryVO> list = new ArrayList<>();
            for (IndustryProductInventoryDO d : industryProductInventory) {
                for (ProductSkuDO e : productSkuList) {
                    GoodsProductSkuVO goodsProductSkuVO = new GoodsProductSkuVO();
                    goodsProductSkuVO.setId(d.getId());
                    goodsProductSkuVO.setProductName(e.getProductName());
                    for (BrandInfoDO b : brandInfoList) {
                        if (e.getBrandInfoId().equals(b.getId())) {
                            goodsProductSkuVO.setProductBrand(b.getBrandName());
                            break;
                        }
                    }
                    goodsProductSkuVO.setGoodsTypeId(e.getCategoriesId());
                    goodsProductSkuVO.setModel(e.getModel());
                    ProductInventoryVO productInventoryVO = new ProductInventoryVO();
                    productInventoryVO.setProductSku(goodsProductSkuVO);
                    if (d.getProductSkuId().equals(e.getId())) {
                        // 匹配分组之后id相等的数据构建到ProductInventoryVO对象中
                        List<ProductSpecVO> productSpecVOS = maps.get(e.getId());
                        productInventoryVO.setProductSpecList(productSpecVOS);
                        productInventoryVO.setSelect(d.getSelected());
                        productInventoryVO.setProductSkuId(d.getProductSkuId());
                        list.add(productInventoryVO);
                    }
                }
            }
            // 组装产品清单
            industrySpecDTO.setProductInventoryList(list);
        }
        return ResultBody.success(industrySpecDTO);
    }

    /**
     * 编辑行业规格信息
     *
     * @param param
     * @return
     */
    @Override
    public ResultBody editIndustrySpec(IndustrySpecVO param) {
        int count = industrySpecDao.countIndustrySpec(param.getId());
        if (count <= 0) {
            return ResultBody.error(ResultEnum.SPEC_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
        }
        // 获取名称判断此前是否已经存在
        int countName = industrySpecDao.countSpecName(param);
        if (countName > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        // 修改行业规格基本信息
        IndustrySpecDO industrySpecDO = new IndustrySpecDO(param);
        int status = industrySpecDao.updateIndustrySpec(industrySpecDO);
        if (status <= 0) {
            return ResultBody.error(ResultEnum.FAILED_TO_EDIT_DATA);
        }
        // 先获取绑定的的产品skuId
        List<IndustryProductInventoryDO> industryProductInventory = industrySpecDao.getIndustryProductInventory(param.getId());
        List<Integer> industryProductInventoryIds = industryProductInventory.stream().map(IndustryProductInventoryDO::getId).collect(Collectors.toList());
        // 批量删除原来绑定的产品规格信息
        if (industryProductInventoryIds.size() != 0) {
            industrySpecDao.batchDeleteInventorySpec(industryProductInventoryIds);
        }
        // 批量删除绑定的产品sku信息
        industrySpecDao.deleteIndustryProductInventory(param.getId());
        // 新增规格绑定产品sku表信息
        addInventorySpec(param, industrySpecDO);
        return ResultBody.success();
    }


    /**
     * 列表页面行业规范
     *
     * @param pageNo        页面没有
     * @param pageSize      页面大小
     * @param industrySkuId 行业sku id
     * @param keyword       关键字
     * @return {@link ResultBody}
     */
    @Override
    public ResultBody listPageIndustrySpec(Integer pageNo, Integer pageSize, Integer industrySkuId, String keyword) {
        int count = industrySpecDao.countListPageIndustrySpec(industrySkuId, keyword);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(pageNo, pageSize, count));
        }
        List<IndustrySpecDO> industrySpecList = industrySpecDao.listPageIndustrySpec((pageNo - 1) * pageSize, pageSize, industrySkuId, keyword);
        List<IndustrySpecDTO> list = industrySpecList.stream().map(IndustrySpecDO::buildIndustrySpecDTO).collect(Collectors.toList());
        return ResultBody.success(PageResult.buildPage(pageNo, pageSize, count, list));
    }

    /**
     * 行业规格价格配置
     *
     * @param industrySpecCPQQ
     * @return
     */
    @Override
    public ResultBody industrySpecCPQ(IndustrySpecCPQVO industrySpecCPQQ) {
        // 判断该规格是否存在
        int count = industrySpecDao.countIndustrySpec(industrySpecCPQQ.getIndustrySpecId());
        if (count <= 0) {
            return ResultBody.error(ResultEnum.SPEC_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
        }
        // 批量插入规格价格
        insertSpecPrice(industrySpecCPQQ);
        return ResultBody.success();
    }

    private List<IndustrySpecPriceDO> getIndustrySpecPriceDOS(IndustrySpecCPQVO industrySpecCPQQ) {
        // 批量插入规格价格
        return industrySpecCPQQ.getSpecPrice().stream().map(d -> {
            IndustrySpecPriceDO industrySpecPriceDO = new IndustrySpecPriceDO();
            industrySpecPriceDO.setCooperationTag(d.getCooperationTag());
            industrySpecPriceDO.setType(industrySpecCPQQ.getType());
            industrySpecPriceDO.setPrice(d.getPrice());
            if (industrySpecCPQQ.getType().equals(1)) {
                industrySpecPriceDO.setLeaseTerm(industrySpecCPQQ.getLeaseTerm());
            }
            industrySpecPriceDO.setIndustrySpecId(industrySpecCPQQ.getIndustrySpecId());
            return industrySpecPriceDO;
        }).collect(Collectors.toList());
    }

    /**
     * 行业规格价格配置修改
     *
     * @param industrySpecCPQQ
     * @return
     */
    @Override
    public ResultBody updateIndustrySpecCPQ(IndustrySpecCPQVO industrySpecCPQQ) {
        // 先删除原来该规格下的所有价格配置信息
        industrySpecDao.removeIndustrySpecCPQ(industrySpecCPQQ);
        // 批量插入规格价格
        insertSpecPrice(industrySpecCPQQ);
        return ResultBody.success();
    }

    public ResultBody insertSpecPrice(IndustrySpecCPQVO industrySpecCPQQ) {
        List<IndustrySpecPriceDO> list = getIndustrySpecPriceDOS(industrySpecCPQQ);
        // 批量插入规格价格
        if (industrySpecCPQQ.getType().equals(0)) {
            int status = industrySpecDao.batchInsertSpecPrice(list);
            if (status <= 0) {
                return ResultBody.error(ResultEnum.FAILED_TO_ADD_DATA);
            }
        } else {
            // 批量插入租赁价格
            industrySpecDao.batchInsertLeaseSpecPrice(list);
        }
        return ResultBody.success();
    }

    /**
     * 获取配置价格信息
     *
     * @param industrySpecCPQQ 行业规范
     * @return {@link ResultBody}
     */
    @Override
    public ResultBody getIndustrySpecCPQ(IndustrySpecCPQVO industrySpecCPQQ) {
        if (industrySpecCPQQ.getType().equals(1) && industrySpecCPQQ.getLeaseTerm() == null) {
            return ResultBody.error("租赁期限不能为空！");
        }
        List<IndustrySpecPriceDO> industrySpecPriceList = industrySpecDao.getIndustrySpecCPQ(industrySpecCPQQ);
        List<IndustrySpecPriceDTO> list = industrySpecPriceList.stream()
                .map(IndustrySpecPriceDO::buildIndustrySpecPriceDTO).collect(Collectors.toList());
        return ResultBody.success(list);
    }


    @Override
    public ResultBody removeIndustrySku(Integer id) {
        // 查询该sku下是否有行业规格存在
        int specCount = industrySpecDao.countIndustrySpecBySkuId(id);
        if (specCount <= 0) {
            // 删除
            industrySpecDao.removeIndustrySku(id);
            return ResultBody.success();
        } else {
            return ResultBody.error(ResultEnum.SPEC_EXIST_UNDER_THE_SKU);
        }
    }

    /**
     * 删除行业规格
     *
     * @param id id
     * @return {@link ResultBody}
     */
    @Override
    public ResultBody removeIndustrySpec(Integer id) {
        // 判断该规格是否存在
        int count = industrySpecDao.countIndustrySpec(id);
        if (count <= 0) {
            return ResultBody.error(ResultEnum.SPEC_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
        }
        // 获取该规格下绑定的商品的信息
        RemoveSkuDTO removeSkuDTO = new RemoveSkuDTO();
        List<MallIndustrySkuInfoSpecDO> mallIndustrySkuInfoSpecList = industrySpecDao.listMallIndustrySpec(id);
        if (mallIndustrySkuInfoSpecList.size() != 0) {
            List<String> goodsName = new ArrayList<>();
            mallIndustrySkuInfoSpecList.forEach(d -> goodsName.add(d.getGoodsName()));
            removeSkuDTO.setGoodsName(goodsName);
            return ResultBody.success(removeSkuDTO);
        } else {
            // 根据规格id获取规格清单绑定的信息
            List<IndustryProductInventoryDO> industryProductInventory = industrySpecDao.getIndustryProductInventory(id);
            if (industryProductInventory.size() != 0) {
                // 删除行业绑定的清单
                industrySpecDao.removeIndustryProductInventory(industryProductInventory.stream().map(IndustryProductInventoryDO::getId).collect(Collectors.toList()));
                List<Integer> ids = industryProductInventory.stream().map(IndustryProductInventoryDO::getId).collect(Collectors.toList());
                List<InventorySpecDO> inventorySpecList = industrySpecDao.listInventorySpec(ids);
                // 删除
                industrySpecDao.removeInventorySpec(inventorySpecList.stream().map(InventorySpecDO::getId).collect(Collectors.toList()));
            }
            industrySpecDao.removeIndustrySpec(id);
            return ResultBody.success();
        }
    }
//
//    @Override
//    public List<IndustrySpecPriceDTO> listAppGoodsPriceByTagId(Integer goodsInfoId, Integer tagInfoId) {
//        List<IndustrySpecPriceDO> industrySpecPriceDOS = industrySpecDao.listAppGoodsPriceByTagId(goodsInfoId, tagInfoId);
//        if (CollectionUtils.isEmpty(industrySpecPriceDOS)) {
//            return null;
//        }
//        List<IndustrySpecPriceDTO> collect = industrySpecPriceDOS.stream().map(IndustrySpecPriceDO::buildIndustrySpecPriceDTO).collect(Collectors.toList());
//        return collect;
//    }
}
