package com.mmc.pms.service.Impl;

import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.GoodsInfoDao;
import com.mmc.pms.dao.IndustrySpecDao;
import com.mmc.pms.dao.ProductDao;
import com.mmc.pms.entity.*;
import com.mmc.pms.model.dto.*;
import com.mmc.pms.model.vo.*;
import com.mmc.pms.service.GoodsInfoService;
import com.mmc.pms.util.CodeUtil;
import com.mmc.pms.util.TDateUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author 23214
 * @description 针对表【goods_info(商品基本信息)】的数据库操作Service实现
 * @createDate 2023-05-27 14:08:45
 */
@Service
public class GoodsInfoServiceImpl implements GoodsInfoService {

    @Resource
    private GoodsInfoDao goodsInfoDao;

    @Resource
    private ProductDao productDao;

    @Resource
    private ProductServiceImpl productSkuService;

    @Resource
    private IndustrySpecDao industrySpecDao;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody addGoods(GoodsAddVO goodsAddVO) {
        // 判断商品名称是否存在
        if (goodsInfoDao.countGoodsInfoByName(goodsAddVO) > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        // 判断商品详情中描述是否为空
        if (goodsAddVO.getGoodsDetailVO().getGoodsDesc() == null) {
            return ResultBody.error(ResultEnum.GOODS_DESC_IS_NOT_NULL);
        }
        String YYYY_MM_DD_HH_MM_SS = "yyyyMMddHHmmss";
        // 初始化商品对象,并构建对象
        GoodsInfo goodsInfo = new GoodsInfo(goodsAddVO);
        goodsInfo.setGoodsNo("IUAV" + TDateUtil.getDateStr(new Date(), YYYY_MM_DD_HH_MM_SS) + CodeUtil.getRandomNum(4));
        // 暂未接入用户默认传1
        goodsInfo.setAddGoodsUserId(1);
        goodsInfo.setSort(goodsInfoDao.countGoodsInfo() + 1);
        // 插入商品基本信息
        goodsInfoDao.insertGoodsInfo(goodsInfo);
        // 调用图片视频插入信息方法
        addGoodsImageInfo(goodsInfo.getId(), goodsAddVO);
        // 调用商品详情信息方法
        addGoodsDetail(goodsInfo.getId(), goodsAddVO.getGoodsDetailVO());
        // 判断其他服务是否为空,不为空则插入
        if (!CollectionUtils.isEmpty(goodsAddVO.getOtherService())) {
            addOtherService(goodsAddVO.getOtherService(), goodsInfo.getId());
        }
        // 调用产品规格的新增方法
        if (!goodsAddVO.getDirectoryId().equals(2)) {
            // 添加产品规格信息
            productSkuSpecOperation(goodsInfo, goodsAddVO.getProductSpec());
        } else {
            industrySkuSpecOperation(goodsInfo, goodsAddVO.getProductSpec());
        }
        return ResultBody.success();
    }

    /**
     * 插入行业sku及规格信息
     */
    @Transactional(rollbackFor = Exception.class)
    public void industrySkuSpecOperation(GoodsInfo goodsInfo, List<GoodsProdSpecVO> productSpec) {
        for (GoodsProdSpecVO goodsSpecVO : productSpec) {
            MallIndustrySkuInfoDO mallIndustrySkuInfoDO = new MallIndustrySkuInfoDO(goodsSpecVO).setGoodsInfoId(goodsInfo.getId());
            // 插入数据库商品对应的sku信息
            goodsInfoDao.insertMallIndustrySkuInfo(mallIndustrySkuInfoDO);
            for (Integer industrySpecId : goodsSpecVO.getSpecIds()) {
                // 构建商品对应的规格信息
                MallIndustrySkuInfoSpecDO mallIndustrySkuInfoSpecDO = new MallIndustrySkuInfoSpecDO()
                        .setIndustrySpecId(industrySpecId)
                        .setMallIndustrySkuInfoId(mallIndustrySkuInfoDO.getId())
                        .setGoodsInfoId(goodsInfo.getId());
                // 插入数据库商品对应的规格信息
                goodsInfoDao.insertMallIndustrySkuInfoSpec(mallIndustrySkuInfoSpecDO);
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void productSkuSpecOperation(GoodsInfo goodsInfo, List<GoodsProdSpecVO> productSpec) {
        // 遍历规格信息，获取其中自定义的规格信息
        List<GoodsProdSpecVO> customGoodsSpecList = productSpec.stream().filter(spec -> spec.getFlag().equals(1)).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(customGoodsSpecList)) {
            addCustomization(goodsInfo, customGoodsSpecList);
        }
        // 遍历规格信息，获取其中非自定义的规格信息
        List<GoodsProdSpecVO> goodsSpecList = productSpec.stream().filter(spec -> spec.getFlag().equals(0)).collect(Collectors.toList());
        for (GoodsProdSpecVO goodsSpec : goodsSpecList) {
            MallProdInfoDO mallProdInfoDO = buildMallProdSkuInfo(goodsInfo, goodsSpec, goodsSpec.getSkuId());
            mallProdInfoDO.setProductSpecIdList(goodsSpec.getSpecIds().stream().map(String::valueOf).collect(Collectors.joining(",")));
            // 插入数据库商品对应的sku信息
            productDao.insertMallProdSkuInfo(mallProdInfoDO);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public MallProdInfoDO buildMallProdSkuInfo(GoodsInfo goodsInfo, GoodsProdSpecVO goodsSpec, Integer id) {
        MallProdInfoDO mallProdInfoDO = new MallProdInfoDO();
        mallProdInfoDO.setGoodsInfoId(goodsInfo.getId());
        mallProdInfoDO.setProdSkuId(id);
        mallProdInfoDO.setProdSkuSpecName(goodsSpec.getGoodsSpecName());
        mallProdInfoDO.setCategoriesId(goodsSpec.getCategoryId());
        mallProdInfoDO.setChooseType(goodsSpec.getChooseType());
        mallProdInfoDO.setMust(goodsSpec.getMust());
        mallProdInfoDO.setFlag(goodsSpec.getFlag());
        mallProdInfoDO.setSkuUnitId(goodsSpec.getSkuUnitId());
        return mallProdInfoDO;
    }

    @Transactional(rollbackFor = Exception.class)
    public void addCustomization(GoodsInfo goodsInfo, List<GoodsProdSpecVO> customGoodsSpecList) {
        // 遍历自定义规格信息
        for (GoodsProdSpecVO productSpec : customGoodsSpecList) {
            // 构建商品对应的sku信息
            ProductSkuDO productSkuDO = new ProductSkuDO().setCategoriesId(productSpec.getCategoryId())
                    .setProductName(productSpec.getProductName()).setCustomize(1)
                    .setDirectoryId(goodsInfo.getDirectoryId());
            // 插入产品sku信息
            productDao.insertProductSku(productSkuDO);
            // 先将自定的信息存储到数据库中
            List<ProductSpecDO> customizeSpec = productSpec.getCustomizeInfo().stream().map(param -> {
                ProductSpecDO productSpecDO = new ProductSpecDO().setProductSkuId(productSkuDO.getId())
                        .setVersionDesc("自定义").setSpecImage(param.getSpecImage())
                        .setSpecName(param.getSpecName()).setPartNo(param.getPartNo());
                // 新增产品sku
                productDao.insertProductSpec(productSpecDO);
                // 批量配置价格信息
                ProductSpecCPQVO productSpecCPQVO = param.getProductSpecCPQVO();
                if (productSpecCPQVO != null) {
                    productSpecCPQVO.setProductSpecId(productSpecDO.getId());
                    productSkuService.insertSpecPrice(productSpecCPQVO);
                }
                return productSpecDO;
            }).collect(Collectors.toList());
            List<Integer> ids = customizeSpec.stream().map(ProductSpecDO::getId).collect(Collectors.toList());
            MallProdInfoDO mallProdInfoDO = buildMallProdSkuInfo(goodsInfo, productSpec, productSkuDO.getId());
            mallProdInfoDO.setProductSpecIdList(ids.stream().map(String::valueOf).collect(Collectors.joining(",")));
            // 插入数据库商品对应的sku信息
            productDao.insertMallProdSkuInfo(mallProdInfoDO);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void addOtherService(List<Integer> otherService, Integer goodsId) {
        // 插入其他服务
        List<GoodsServiceDO> otherList = otherService.stream().map(d -> {
            GoodsServiceDO goodsServiceDO = new GoodsServiceDO();
            goodsServiceDO.setGoodsInfoId(goodsId);
            goodsServiceDO.setSaleServiceId(d);
            return goodsServiceDO;
        }).collect(Collectors.toList());
        goodsInfoDao.insertGoodsService(otherList);
    }

    @Transactional(rollbackFor = Exception.class)
    public void addGoodsDetail(Integer goodsId, GoodsDetailVO goodsDetailVO) {
        GoodsDetailDO goodsDetailDO = new GoodsDetailDO(goodsDetailVO);
        goodsDetailDO.setGoodsInfoId(goodsId);
        goodsInfoDao.insertGoodsDetail(goodsDetailDO);
    }

    @Transactional(rollbackFor = Exception.class)
    public void addGoodsImageInfo(Integer goodsId, GoodsAddVO goodsAddVO) {
        List<GoodsImgDO> list = goodsAddVO.getImages().stream().map(d -> {
            GoodsImgDO goodsImgDO = new GoodsImgDO(d);
            goodsImgDO.setGoodsInfoId(goodsId);
            return goodsImgDO;
        }).collect(Collectors.toList());
        // 插入图片信息
        goodsInfoDao.insertGoodsImgInfo(list);
        // 插入商品视频信息
        if (goodsAddVO.getGoodsVideo() != null) {
            GoodsVideoDO goodsVideoDO = new GoodsVideoDO().setGoodsInfoId(goodsId).setVideoUrl(goodsAddVO.getGoodsVideo());
            goodsInfoDao.insertVideoInfo(goodsVideoDO);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody editGoodsInfo(GoodsAddVO goodsAddVO) {
        // 判断商品是否还存在
        int count = goodsInfoDao.countGoodsInfoById(goodsAddVO.getId());
        if (count <= 0) {
            return ResultBody.error(ResultEnum.GOODS_NOT_EXIST_OR_ALREADY_DOWN_SHELF);
        }
        // 判断商品名称是否存在
        if (goodsInfoDao.countGoodsInfoByName(goodsAddVO) > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        // 判断商品详情中描述是否为空
        if (goodsAddVO.getGoodsDetailVO().getGoodsDesc() == null) {
            return ResultBody.error(ResultEnum.GOODS_DESC_IS_NOT_NULL);
        }
        // 初始化商品对象,并构建对象
        GoodsInfo goodsInfo = new GoodsInfo(goodsAddVO);
        // 插入商品基本信息
        goodsInfoDao.updateGoodsInfo(goodsInfo);
        // 修改商品详情信息
        updateGoodsDetail(goodsAddVO);
        // 修改商品图片及视频信息
        updateImageInfo(goodsAddVO);
        // 修改其他服务信息
        updateOtherService(goodsAddVO);
        // 修改规格信息
        if (!goodsAddVO.getDirectoryId().equals(2)) {
            updateMallProductSkuSpec(goodsAddVO);
        } else {
            updateMallIndustrySkuSpec(goodsAddVO);
        }
        return null;
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateMallIndustrySkuSpec(GoodsAddVO goodsAddVO) {
        // 从数据库获取该商品对应绑定的skuId的集合
        List<MallIndustrySkuInfoDO> mallIndusSkuInfoList = goodsInfoDao.getMallIndustrySkuInfo(goodsAddVO.getId());
        List<Integer> skuIds = mallIndusSkuInfoList.stream().map(MallIndustrySkuInfoDO::getId).collect(Collectors.toList());
        // 先删除所有商品绑定sku及spec信息
        industrySpecDao.batchUpdateMallIndustrySpec(skuIds, goodsAddVO.getId());
        industrySpecDao.batchUpdateMallIndustrySku(skuIds);
        List<GoodsProdSpecVO> list = goodsAddVO.getProductSpec().stream().collect(Collectors.toList());
        industrySkuSpecOperation(new GoodsInfo(goodsAddVO.getId()), list);
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateMallProductSkuSpec(GoodsAddVO goodsAddVO) {
        // 1、普通的sku修改逻辑
        List<GoodsProdSpecVO> specInfo = goodsAddVO.getProductSpec().stream().filter(spec -> !spec.getFlag().equals(1)).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(specInfo)) {
            // 1.1先获取用户输入的商品绑定sku的id集合
            List<GoodsProdSpecVO> updateGoodsSpec = specInfo.stream().filter(d -> d.getId() != null).collect(Collectors.toList());
            List<Integer> inputMallSkuIds = updateGoodsSpec.stream().map(GoodsProdSpecVO::getId).collect(Collectors.toList());
            // 1.2从数据库获取该商品对应绑定的skuId的集合
            List<MallProdInfoDO> mallProSkuInfoList = goodsInfoDao.getMallProSkuInfo(goodsAddVO.getId());
            List<Integer> dbMallSkuIds = mallProSkuInfoList.stream().map(MallProdInfoDO::getId).collect(Collectors.toList());
            // 对比数据库id如果不在就删除
            List<Integer> delIds = dbMallSkuIds.stream().filter(id -> !inputMallSkuIds.contains(id)).collect(Collectors.toList());
            if (delIds.size() != 0) {
                goodsInfoDao.batchUpdateMallProductSku(delIds);
            }
            // 用途：修改商品规格前选项来源保留一份快照
            Map<Integer, MallProdInfoDO> mallProSkuInfoMap = mallProSkuInfoList.stream().collect(Collectors.toMap(MallProdInfoDO::getId, Function.identity()));
            // 修改商品绑定的sku信息
            List<MallProdInfoDO> mallProdSkuInfoList = updateGoodsSpec.stream().map(d -> {
                MallProdInfoDO mallProdSkuInfoById = mallProSkuInfoMap.get(d.getId());
                MallProdInfoDO mallProdInfoDO = new MallProdInfoDO();
                mallProdInfoDO.setId(d.getId()).setProdSkuId(d.getSkuId())
                        .setCategoriesId(d.getCategoryId()).setProdSkuSpecName(d.getGoodsSpecName())
                        .setChooseType(d.getChooseType()).setSkuUnitId(d.getSkuUnitId()).setMust(d.getMust())
                        .setProductSpecIdList(d.getSpecIds().stream().map(String::valueOf).collect(Collectors.joining(",")))
                        .setBeforeUpdateSpec(mallProdSkuInfoById.getProductSpecIdList());
                return mallProdInfoDO;
            }).collect(Collectors.toList());
            // 1.3批量修改sku的信息
            goodsInfoDao.batchUpdateMallProdSkuInfo(mallProdSkuInfoList);
            // 1.4对新的sku进行操作
            List<GoodsProdSpecVO> list = specInfo.stream().filter(d -> d.getId() == null).collect(Collectors.toList());
            productSkuSpecOperation(new GoodsInfo(goodsAddVO.getId()), list);
            // 2、对自定义的sku进行操作
            List<GoodsProdSpecVO> zdySpecInfo = goodsAddVO.getProductSpec().stream().filter(spec -> spec.getFlag().equals(1)).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(zdySpecInfo)) {
                updateCustomize(zdySpecInfo, goodsAddVO.getId());
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateCustomize(List<GoodsProdSpecVO> zdySpecInfo, Integer id) {
        // 2.1先获取自定义的sku及规格（原来有的）
        List<GoodsProdSpecVO> zdySpec = zdySpecInfo.stream().filter(d -> d.getId() != null).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(zdySpec)) {
            // 根据获取到的自定义sku及规格进行操作
            List<MallProdInfoDO> zdySpecList = zdySpec.stream().map(d -> {
                // 构建商品对应的sku信息
                ProductSkuDO productSkuDO = new ProductSkuDO().setCategoriesId(d.getCategoryId())
                        .setProductName(d.getProductName()).setCustomize(1);
                productDao.updateProductSku(productSkuDO);
                if (CollectionUtils.isNotEmpty(d.getDelProductSpecId())) {
                    // 2.2删除这个sku下的spec
                    productDao.batchUpdateMallProdSpec(d.getDelProductSpecId());
                }
                List<Integer> zdySpecIds = d.getCustomizeInfo().stream().map(ProductSpecVO::getId).collect(Collectors.toList());
                return new MallProdInfoDO(d).setId(d.getId()).setProdSkuId(d.getSkuId())
                        .setProductSpecIdList(zdySpecIds.stream().map(String::valueOf).collect(Collectors.joining(",")));
            }).collect(Collectors.toList());
            // 批量修改sku的信息
            goodsInfoDao.batchUpdateMallProdSkuInfo(zdySpecList);
        }
        // 2.2 新增新的自定义规格信息
        List<GoodsProdSpecVO> newZdySpec = zdySpecInfo.stream().filter(d -> d.getId() == null).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(newZdySpec)) {
            productSkuSpecOperation(new GoodsInfo(id), newZdySpec);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateOtherService(GoodsAddVO goodsAddVO) {
        // 删除其他服务
        goodsInfoDao.deleteGoodsServiceByGoodsId(goodsAddVO.getId());
        // 插入其他服务
        if (!CollectionUtils.isEmpty(goodsAddVO.getOtherService())) {
            List<Integer> other = goodsAddVO.getOtherService();
            List<GoodsServiceDO> otherList = other.stream().map(d -> {
                GoodsServiceDO goodsServiceDO = new GoodsServiceDO();
                goodsServiceDO.setGoodsInfoId(goodsAddVO.getId());
                goodsServiceDO.setSaleServiceId(d);
                return goodsServiceDO;
            }).collect(Collectors.toList());
            goodsInfoDao.insertGoodsService(otherList);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateImageInfo(GoodsAddVO goodsAddVO) {
        Set<Integer> imgIds = goodsAddVO.getImages().stream().map(GoodsImgVO::getId).filter(Objects::nonNull).collect(Collectors.toSet());
        List<GoodsImgDO> dbImgList = goodsInfoDao.listGoodsInfoByGoodsId(goodsAddVO.getId());
        List<Integer> deleteIds = dbImgList.stream().map(GoodsImgDO::getId).filter(id -> !imgIds.contains(id)).collect(Collectors.toList());
        if (deleteIds.size() != 0) {
            // 删除多余的图片
            goodsInfoDao.deleteImgByIds(deleteIds);
        }
        // 新增图片
        List<GoodsImgDO> imgDOList = goodsAddVO.getImages().stream().filter(d -> d.getId() == null).map(d -> {
            GoodsImgDO goodsImgDO = new GoodsImgDO(d);
            goodsImgDO.setGoodsInfoId(goodsAddVO.getId());
            return goodsImgDO;
        }).collect(Collectors.toList());
        if (imgDOList.size() != 0) {
            goodsInfoDao.insertGoodsImgInfo(imgDOList);
        }
        // 删除视频
        goodsInfoDao.deleteGoodsVideoById(goodsAddVO.getId());
        // 插入视频
        if (goodsAddVO.getGoodsVideo() != null) {
            GoodsVideoDO goodsVideoDO = new GoodsVideoDO();
            goodsVideoDO.setGoodsInfoId(goodsAddVO.getId());
            goodsVideoDO.setVideoUrl(goodsAddVO.getGoodsVideo());
            goodsInfoDao.insertVideoInfo(goodsVideoDO);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateGoodsDetail(GoodsAddVO goodsAddVO) {
        GoodsDetailDO goodsDetailDO = new GoodsDetailDO(goodsAddVO.getGoodsDetailVO()).setGoodsInfoId(goodsAddVO.getId());
        // 商品详情修改
        goodsInfoDao.updateGoodsDetail(goodsDetailDO);
    }


    @Override
    public ResultBody getGoodsInfoDetail(Integer goodsInfoId) {
        // 判断此商品是否还存在
        int count = goodsInfoDao.countGoodsInfoById(goodsInfoId);
        if (count <= 0) {
            return ResultBody.error(ResultEnum.GOODS_NOT_EXIST_OR_ALREADY_DOWN_SHELF);
        }
        // 初始化商品返回信息的对象
        MallGoodsDetailDTO mallGoodsDetail = new MallGoodsDetailDTO();
        // 获取商品基本信息
        GoodsInfo goodsInfo = goodsInfoDao.getGoodsSimpleInfo(goodsInfoId);
        mallGoodsDetail.setId(goodsInfo.getId()).setGoodsName(goodsInfo.getGoodsName())
                .setCategoryByOne(goodsInfo.getCategoryByOne()).setGoodsVideo(goodsInfo.getVideoUrl())
                .setDirectoryId(goodsInfo.getDirectoryId()).setCategoryByTwo(goodsInfo.getCategoryByTwo())
                .setTag(goodsInfo.getEcoLabel()).setShelfStatus(goodsInfo.getShelfStatus())
                .setGoodsVideoId(goodsInfo.getGoodsVideoId());
        // 获取商品图片信息
        mallGoodsDetail.setImages(getGoodsImageInfo(goodsInfoId));
        // 获取商品详细信息
        mallGoodsDetail.setGoodsDetail(getGoodsDetail(goodsInfoId));
        // 获取其他服务信息
        mallGoodsDetail.setOtherService(getOtherServiceInfo(goodsInfoId));
        // 获取规格信息
        if (!goodsInfo.getDirectoryId().equals(2)) {
            // 获取产品规格信息
            mallGoodsDetail.setGoodsSpec(getProductSpecInfo(goodsInfoId));
        } else {
            // 获取行业规格信息
            mallGoodsDetail.setGoodsSpec(getIndustrySpecInfo(goodsInfoId));
        }
        return ResultBody.success(mallGoodsDetail);
    }

    private List<GoodsSpecDTO> getIndustrySpecInfo(Integer goodsInfoId) {
        // 获取商品对应绑定的行业sku信息
        List<MallIndustrySkuInfoDO> mallIndustrySkuInfoList = goodsInfoDao.getMallIndustrySkuInfo(goodsInfoId);
        List<GoodsSpecDTO> list = mallIndustrySkuInfoList.stream().map(MallIndustrySkuInfoDO::buildGoodsSpecDTO).collect(Collectors.toList());
        // 根据商品id查出该商品下绑定的规格信息
        List<MallIndustrySkuInfoSpecDO> mallIndustrySkuInfoSpec = industrySpecDao.getIndustrySkuInfoSpec(goodsInfoId);
        list = list.stream().peek(d -> {
            List<MallIndustrySpecDTO> industrySpec = new ArrayList<>();
            for (MallIndustrySkuInfoSpecDO e : mallIndustrySkuInfoSpec) {
                if (d.getId().equals(e.getMallIndustrySkuInfoId())) {
                    IndustrySpecDO industrySpecDO = e.getIndustrySpecDO();
                    MallIndustrySpecDTO industrySpecDTO = industrySpecDO.buildMallIndustrySpecDTO();
                    industrySpecDTO.setId(e.getId());
                    industrySpecDTO.setIndustrySpecId(e.getIndustrySpecId());
                    industrySpecDTO.setIndustrySkuId(d.getSkuId());
                    industrySpec.add(industrySpecDTO);
                }
            }
            d.setIndustrySpecList(industrySpec);
        }).collect(Collectors.toList());
        return list;
    }

    private List<GoodsSpecDTO> getProductSpecInfo(Integer goodsInfoId) {
        // 获取商品对应绑定sku的信息
        List<MallProdInfoDO> mallProSkuInfo = goodsInfoDao.getMallProSkuInfo(goodsInfoId);
        // 获取该产品下的规格id
        Map<Integer, String> specIdsMap = mallProSkuInfo.stream().collect(Collectors.toMap(MallProdInfoDO::getId, MallProdInfoDO::getProductSpecIdList));
        List<GoodsSpecDTO> list = mallProSkuInfo.stream().map(MallProdInfoDO::buildGoodsSpecDTO).collect(Collectors.toList());
        for (GoodsSpecDTO goodsSpecDTO : list) {
            String specIds = specIdsMap.get(goodsSpecDTO.getId());
            String[] ids = specIds.split(",");
            List<Integer> idList = new ArrayList<>();
            for (String id : ids) {
                idList.add(Integer.parseInt(id));
            }
            // 根据specIds集合找出spec的信息
            List<ProductSpecDO> productSpecDOS = productDao.listProductSpecInfo(idList);
            List<MallProductSpecDTO> mallProductSpecList = productSpecDOS.stream().map(ProductSpecDO::buildMallProductSpecDTO).collect(Collectors.toList());
            goodsSpecDTO.setProductSpecList(mallProductSpecList);
        }
        return list;
    }

    private List<GoodsOtherServiceDTO> getOtherServiceInfo(Integer goodsInfoId) {
        List<GoodsServiceDO> goodsServiceDO = goodsInfoDao.listGoodsServiceByGoodsId(goodsInfoId);
        return goodsServiceDO.stream().map(GoodsServiceDO::buildGoodsOtherServiceDTO).collect(Collectors.toList());
    }

    private GoodsDetailInfoDTO getGoodsDetail(Integer goodsInfoId) {
        GoodsDetailDO goodsDetail = goodsInfoDao.getGoodsDetailByGoodsId(goodsInfoId);
        return goodsDetail.buildGoodsDetailInfoDTO();
    }

    private List<GoodsImgDTO> getGoodsImageInfo(Integer goodsInfoId) {
        List<GoodsImgDO> goodsImgList = goodsInfoDao.listGoodsInfoByGoodsId(goodsInfoId);
        return goodsImgList.stream().map(GoodsImgDO::buildGoodsImgDTO).collect(Collectors.toList());
    }


    @Override
    public ResultBody getSkuUnit() {
        List<SkuUnitDO> skuUnitList = goodsInfoDao.getSkuUnit();
        List<SkuUnitDTO> list = skuUnitList.stream().map(SkuUnitDO::buildSkuUnitDTO).collect(Collectors.toList());
        return ResultBody.success(list);
    }
}




