package com.mmc.pms.service.impl;

import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.BrandManageDao;
import com.mmc.pms.dao.ProductDao;
import com.mmc.pms.entity.BrandInfoDO;
import com.mmc.pms.model.sale.dto.BrandInfoDTO;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.BrandManageService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @Author LW
 *
 * @date 2023/3/14 13:27 概要：
 */
@Service
public class BrandManageServiceImpl implements BrandManageService {

  @Autowired private BrandManageDao brandManageDao;
  @Resource private ProductDao productDao;

  /**
   * 添加品牌信息
   *
   * @param brandName 品牌名称
   * @return {@link ResultBody}
   */
  @Override
  public ResultBody addBrandInfo(String brandName) {
    // 根据品牌名称判断是否存在
    int count = brandManageDao.countBrandInfoByName(null, brandName);
    if (count > 0) {
      return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
    }
    BrandInfoDO brandInfoDO = new BrandInfoDO(brandName);
    brandManageDao.insertBrandInfo(brandInfoDO);
    return ResultBody.success();
  }

  /**
   * 品牌列表信息
   *
   * @return {@link ResultBody}
   */
  @Override
  public PageResult listBrandInfo(Integer pageNo, Integer pageSize) {
    int count = brandManageDao.countBrandInfo();
    if (count == 0) {
      return PageResult.buildPage(pageNo, pageSize, 0);
    }
    int itemIndex = (pageNo - 1) * pageSize;
    List<BrandInfoDO> brandInfo = brandManageDao.listBrandInfo(itemIndex, pageSize);
    List<BrandInfoDTO> brandInfoList =
        brandInfo.stream().map(BrandInfoDO::buildBrandInfoDTO).collect(Collectors.toList());
    return PageResult.buildPage(pageNo, pageSize, count, brandInfoList);
  }

  /**
   * 删除品牌信息
   *
   * @param id id
   * @return {@link ResultBody}
   */
  @Override
  public ResultBody deleteBrandInfo(Integer id) {
    // 判断该品牌是否绑定产品信息
    int count = productDao.countProductSpecByBrandId(id);
    if (count > 0) {
      return ResultBody.error(ResultEnum.BRAND_DELETE_FAIL);
    }
    brandManageDao.removeBrandInfoById(id);
    return ResultBody.success();
  }

  @Override
  public ResultBody editBrandInfo(Integer id, String brandName) {
    // 根据品牌名称判断是否存在
    int count = brandManageDao.countBrandInfoByName(id, brandName);
    if (count > 0) {
      return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
    }
    BrandInfoDO brandInfoDO = new BrandInfoDO(brandName);
    brandInfoDO.setId(id);
    brandManageDao.updateBrandInfo(brandInfoDO);
    return ResultBody.success();
  }
}
