package com.mmc.pms.service.Impl;

import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.IndustrySpecDao;
import com.mmc.pms.dao.ProductDao;
import com.mmc.pms.entity.*;
import com.mmc.pms.model.lease.vo.PriceAcquisition;
import com.mmc.pms.model.qo.ProductSkuQO;
import com.mmc.pms.model.sale.dto.*;
import com.mmc.pms.model.sale.vo.ProductSpecCPQVO;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.ProductSkuService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author 23214
 * @description 针对表【product_sku(产品sku表)】的数据库操作Service实现
 * @createDate 2023-05-25 14:55:56
 */
@Service
public class ProductServiceImpl implements ProductSkuService {
  @Autowired private ProductDao productDao;
  @Autowired private IndustrySpecDao industrySpecDao;

  @Override
  public ResultBody addProductSku(ProductSkuVO param) {
    // 获取名称判断此前是否已经存在
    int count = productDao.countSkuName(param);
    if (count > 0) {
      return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
    }
    ProductSkuDO productSkuDO = new ProductSkuDO(param);
    // 新增产品sku
    int status = productDao.insertProductSku(productSkuDO);
    if (status <= 0) {
      return ResultBody.error(ResultEnum.FAILED_TO_ADD_DATA);
    }
    return ResultBody.success();
  }

  @Override
  public ResultBody getProductSkuDetail(Integer id) {
    // 校验此sku是否还存在或已删除
    int count = productDao.countSkuIsExist(id);
    if (count <= 0) {
      return ResultBody.error(ResultEnum.SKU_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
    }
    ProductSkuDO productSkuDO = productDao.getProductSkuDetail(id);
    return ResultBody.success(productSkuDO.buildProductSkuDTO());
  }

  @Override
  public ResultBody editProductSku(ProductSkuVO param) {
    // 获取名称判断此前是否已经存在
    int count = productDao.countSkuName(param);
    if (count > 0) {
      return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
    }
    ProductSkuDO productSkuDO = new ProductSkuDO(param);
    int status = productDao.updateProductSku(productSkuDO);
    if (status <= 0) {
      return ResultBody.error(ResultEnum.FAILED_TO_EDIT_DATA);
    }
    return ResultBody.success();
  }

  @Override
  public ResultBody listPageProductSku(ProductSkuQO productSkuQO) {
    int count = productDao.countListPageProductSku(productSkuQO);
    if (count == 0) {
      return ResultBody.success(
          PageResult.buildPage(productSkuQO.getPageNo(), productSkuQO.getPageSize(), count));
    }
    Integer pageNo = productSkuQO.getPageNo();
    productSkuQO.buildCurrentPage();
    List<ProductSkuDO> productSkuList = productDao.listPageProductSku(productSkuQO);
    List<Integer> productSkuIds =
        productSkuList.stream().map(ProductSkuDO::getId).collect(Collectors.toList());
    List<ProductSpecDO> productSpecList = productDao.getProductSpecList(productSkuIds);
    Map<Integer, List<ProductSpecDO>> productSpecMap =
        productSpecList.stream().collect(Collectors.groupingBy(ProductSpecDO::getProductSkuId));
    List<ProductSkuDTO> list =
        productSkuList.stream().map(ProductSkuDO::buildProductSkuDTO).collect(Collectors.toList());
    list.stream()
        .peek(
            d -> {
              List<ProductSpecDO> productSpecDOS = productSpecMap.get(d.getId());
              if (CollectionUtils.isEmpty(productSpecDOS)) {
                d.setProductSpecList(null);
              } else {
                d.setProductSpecList(
                    productSpecDOS.stream()
                        .map(ProductSpecDO::buildProductSpecDTO)
                        .collect(Collectors.toList()));
              }
            })
        .collect(Collectors.toList());
    return ResultBody.success(
        PageResult.buildPage(pageNo, productSkuQO.getPageSize(), count, list));
  }

  @Override
  public ResultBody addOrEditProductSpec(ProductSpecVO param) {
    // 获取名称判断此前是否已经存在
    int count = productDao.countSpecName(param);
    if (count > 0) {
      return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
    }
    if (param.getId() == null) {
      ProductSpecDO productSpecDO = new ProductSpecDO(param);
      // 新增产品sku
      int status = productDao.insertProductSpec(productSpecDO);
      if (status <= 0) {
        return ResultBody.error(ResultEnum.FAILED_TO_ADD_DATA);
      }
    } else {
      ProductSpecDO productSpecDO = new ProductSpecDO(param);
      int status = productDao.updateProductSpec(productSpecDO);
      if (status <= 0) {
        return ResultBody.error(ResultEnum.FAILED_TO_EDIT_DATA);
      }
    }
    return ResultBody.success();
  }

  @Override
  public ResultBody getProductSpecDetail(Integer id) {
    // 校验此sku是否还存在或已删除
    int count = productDao.countSpecIsExist(id);
    if (count <= 0) {
      return ResultBody.error(ResultEnum.SPEC_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
    }
    ProductSpecDO productSpecDO = productDao.getProductSpecDetail(id);
    return ResultBody.success(productSpecDO.buildProductSpecDTO());
  }

  @Override
  public ResultBody listPageProductSpec(
      Integer pageNo, Integer pageSize, Integer productSkuId, String keyword) {
    int count = productDao.countListPageProductSpec(productSkuId, keyword);
    if (count == 0) {
      return ResultBody.success(PageResult.buildPage(pageNo, pageSize, count));
    }
    List<ProductSpecDO> productSpecList =
        productDao.listPageProductSpec((pageNo - 1) * pageSize, pageSize, productSkuId, keyword);
    List<ProductSpecDTO> list =
        productSpecList.stream()
            .map(ProductSpecDO::buildProductSpecDTO)
            .collect(Collectors.toList());
    return ResultBody.success(PageResult.buildPage(pageNo, pageSize, count, list));
  }

  @Override
  public ResultBody productSpecCPQ(ProductSpecCPQVO productSpecCPQVO) {
    // 判断该规格是否存在
    int count = productDao.countSpecIsExist(productSpecCPQVO.getProductSpecId());
    if (count <= 0) {
      return ResultBody.error(ResultEnum.SPEC_DOES_NOT_EXIST_OR_HAS_BEEN_DELETED);
    }
    // 批量插入规格销售或租赁价格
    return insertSpecPrice(productSpecCPQVO);
  }

  @Override
  public List<ProductSpecPriceDO> getProductSpecPriceDOS(ProductSpecCPQVO productSpecCPQVO) {
    // 批量插入规格价格
    return productSpecCPQVO.getSpecPrice().stream()
        .map(
            d -> {
              ProductSpecPriceDO productSpecPriceDO = new ProductSpecPriceDO();
              productSpecPriceDO.setCooperationTag(d.getCooperationTag());
              productSpecPriceDO.setPrice(d.getPrice());
              productSpecPriceDO.setType(productSpecCPQVO.getType());
              productSpecPriceDO.setProductSpecId(productSpecCPQVO.getProductSpecId());
              if (productSpecCPQVO.getType().equals(1)) {
                productSpecPriceDO.setLeaseTerm(productSpecCPQVO.getLeaseTerm());
              }
              return productSpecPriceDO;
            })
        .collect(Collectors.toList());
  }

  @Override
  public ResultBody updateProductSpecCPQ(ProductSpecCPQVO productSpecCPQVO) {
    // 先删除原来该规格下的所有价格配置信息
    productDao.removeProductSpecCPQ(productSpecCPQVO);
    // 批量插入规格销售或租赁价格
    return insertSpecPrice(productSpecCPQVO);
  }

  @NotNull
  public ResultBody insertSpecPrice(ProductSpecCPQVO productSpecCPQVO) {
    List<ProductSpecPriceDO> list = getProductSpecPriceDOS(productSpecCPQVO);
    if (productSpecCPQVO.getType().equals(1) && productSpecCPQVO.getLeaseTerm() == null) {
      return ResultBody.error("租赁期限不能为空！");
    }
    // 批量插入规格销售价格
    if (productSpecCPQVO.getType().equals(0)) {
      productDao.batchInsertSpecPrice(list);
    } else {
      // 批量插入租赁价格
      productDao.batchInsertLeaseSpecPrice(list);
    }
    return ResultBody.success();
  }

  @Override
  public ResultBody getProductSpecCPQ(ProductSpecCPQVO productSpecCPQVO) {
    if (productSpecCPQVO.getType().equals(1) && productSpecCPQVO.getLeaseTerm() == null) {
      return ResultBody.error("租赁期限不能为空！");
    }
    List<ProductSpecPriceDO> productSpecPriceList =
        productDao.getProductSpecPrice(productSpecCPQVO);
    List<ProductSpecPriceDTO> list =
        productSpecPriceList.stream()
            .map(ProductSpecPriceDO::buildProductSpecPriceDTO)
            .collect(Collectors.toList());
    return ResultBody.success(list);
  }

  @Override
  public ResultBody removeProductSku(Integer id) {
    // 判断该sku下是否有规格，没有则删除，有则不能删除
    int specCount = productDao.countSpecByProdSkuId(id);
    if (specCount <= 0) {
      // 删除
      productDao.removeProductSku(id);
      return ResultBody.success();
    } else {
      return ResultBody.error(ResultEnum.SPEC_EXIST_UNDER_THE_SKU);
    }
  }

  @Override
  public ResultBody removeProductSpec(Integer id) {
    // 根据规格id查询绑定的商品或者行业的sku
    RemoveSkuDTO removeSkuDTO = new RemoveSkuDTO();
    List<MallProdSkuInfoSpecDO> mallProdSkuInfoSpecList = productDao.listMallProductSpec(id);
    // 拿到清单的id
    List<InventorySpecDO> inventorySpecList = productDao.listInventorySpec(id);
    if (mallProdSkuInfoSpecList.size() != 0 || inventorySpecList.size() != 0) {
      if (inventorySpecList.size() != 0) {
        // 根据清单id 获取行业规格id
        Set<Integer> inventoryIds =
            inventorySpecList.stream()
                .map(InventorySpecDO::getIndustryProductInventoryId)
                .collect(Collectors.toSet());
        List<IndustryProductInventoryDO> industryProductInventoryList =
            industrySpecDao.listIndustryProdInventory(inventoryIds);
        // 获取行业规格id
        Set<Integer> industrySpecIds =
            industryProductInventoryList.stream()
                .map(IndustryProductInventoryDO::getIndustrySpecId)
                .collect(Collectors.toSet());
        // 分别获取规格对应的sku信息
        List<IndustrySpecDO> industrySpecList = industrySpecDao.listIndustrySpec(industrySpecIds);
        List<SkuAndSpecDTO> industrySkuNameList = new ArrayList<>();
        industrySpecList.stream()
            .peek(
                d -> {
                  SkuAndSpecDTO skuAndSpecDTO = new SkuAndSpecDTO();
                  skuAndSpecDTO.setIndustrySkuName(d.getSolutionName());
                  skuAndSpecDTO.setIndustrySpecNames(d.getSpecName());
                  industrySkuNameList.add(skuAndSpecDTO);
                })
            .collect(Collectors.toList());
        removeSkuDTO.setIndustrySkuName(industrySkuNameList);
      }
      if (mallProdSkuInfoSpecList.size() != 0) {
        // 获取商品名称
        List<String> goodsNameList = new ArrayList<>();
        mallProdSkuInfoSpecList.forEach(d -> goodsNameList.add(d.getGoodsName()));
        removeSkuDTO.setGoodsName(goodsNameList);
      }
      return ResultBody.success(removeSkuDTO);
    } else {
      productDao.removeProductSpec(id);
      return ResultBody.success();
    }
  }

  @Override
  public BigDecimal feignGetUnitPriceByTag(PriceAcquisition priceAcquisition) {
    Map<Integer, Integer> dayMap = new HashMap<>();
    dayMap.put(0, 7);
    dayMap.put(8, 15);
    dayMap.put(16, 30);
    dayMap.put(31, Integer.MAX_VALUE);

    Integer dayRange =
        dayMap.entrySet().stream()
            .filter(entry -> priceAcquisition.getDay() <= entry.getKey())
            .findFirst()
            .map(Map.Entry::getValue)
            .orElseThrow(() -> new RuntimeException("租赁期限错误错误!"));
    priceAcquisition.setDay(dayRange);
    return productDao.feignGetUnitPriceByTag(priceAcquisition);
  }
}
