package com.mmc.pms.service.Impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.mmc.pms.common.ResultBody;
import com.mmc.pms.dao.BackstageTaskServiceDao;
import com.mmc.pms.dao.ServiceOrderProvinceCityDao;
import com.mmc.pms.dao.ServiceOrderTaskDao;
import com.mmc.pms.dao.ServiceOrderTaskImageDao;
import com.mmc.pms.entity.ServiceDO;
import com.mmc.pms.entity.ServiceOrderProvinceCityDO;
import com.mmc.pms.entity.ServiceOrderTaskDO;
import com.mmc.pms.entity.ServiceOrderTaskImageDO;
import com.mmc.pms.model.qo.ServiceOrderTaskQO;
import com.mmc.pms.model.work.dto.ServiceOrderTaskDTO;
import com.mmc.pms.model.work.vo.ServiceOrderTaskVO;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.ServiceOrderTaskService;
import com.mmc.pms.util.BeanCopyUtils;
import com.mmc.pms.util.CodeUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 云享飞服务订单-主表 服务实现类
 *
 * @author Pika
 * @since 2023-06-10
 */
@Service
public class ServiceOrderTaskServiceImpl
    extends ServiceImpl<ServiceOrderTaskDao, ServiceOrderTaskDO>
    implements ServiceOrderTaskService {

  @Autowired private ServiceOrderTaskDao serviceOrderTaskDao;

  @Autowired private ServiceOrderProvinceCityDao serviceOrderProvinceCityDao;

  @Autowired private ServiceOrderTaskImageDao serviceOrderTaskImageDao;

  @Autowired private BackstageTaskServiceDao backstageTaskServiceDao;

  @Override
  @Transactional(rollbackFor = Exception.class)
  public ResultBody insertOrderTask(ServiceOrderTaskVO param, Integer userAccountId) {
    ServiceDO serviceDO = backstageTaskServiceDao.queryById(param.getInspectionId());
    if (serviceDO == null) {
      return ResultBody.error("服务id不存在");
    }
    // 保存作业服务订单
    ServiceOrderTaskDO orderTaskDO =
        ServiceOrderTaskDO.builder()
            .orderNo(CodeUtil.createOrderTaskNO())
            .orderName(serviceDO.getServiceName())
            .inspectionId(param.getInspectionId())
            .inspectionName(serviceDO.getServiceName())
            .taskDescription(param.getTaskDescription())
            .startTime(param.getStartTime())
            .endTime(param.getEndTime())
            .userAccountId(userAccountId)
            .build();
    serviceOrderTaskDao.insert(orderTaskDO);

    // 保存作业服务地址信息
    ServiceOrderProvinceCityDO provinceCityDO =
        ServiceOrderProvinceCityDO.builder()
            .serviceOrderTaskId(orderTaskDO.getId())
            .province(param.getProvince())
            .city(param.getCity())
            .address(param.getAddress())
            .lat(param.getLat())
            .lon(param.getLon())
            .build();
    serviceOrderProvinceCityDao.insert(provinceCityDO);

    // 保存作业服务图片
    List<String> images = param.getImages();
    if (CollectionUtils.isNotEmpty(images)) {
      ServiceOrderTaskImageDO imageDO =
          ServiceOrderTaskImageDO.builder()
              .orderTaskId(orderTaskDO.getId())
              .picture(String.join(",", images))
              .build();
      serviceOrderTaskImageDao.insert(imageDO);
    }
    return ResultBody.success();
  }

  @Override
  public PageResult queryOrderTaskList(ServiceOrderTaskQO param, Integer userAccountId) {
    Integer pageNo = param.getPageNo();
    param.buildCurrentPage();
    Integer count = serviceOrderTaskDao.conditionCount(param, userAccountId);
    if (count == null || count == 0) {
      return PageResult.buildPage(param.getPageNo(), param.getPageSize(), count);
    }
    List<ServiceOrderTaskDO> orderTaskList =
        serviceOrderTaskDao.queryPageByLimit(param, userAccountId);
    List<Integer> orderTaskIds =
        orderTaskList.stream().map(ServiceOrderTaskDO::getId).collect(Collectors.toList());

    Map<Integer, ServiceOrderProvinceCityDO> provinceCityMap = getOrderProvinceCityDOMapById(orderTaskIds);

    Map<Integer, List<ServiceOrderTaskImageDO>> imageMap = getImageListMapById(orderTaskIds);
    List<ServiceOrderTaskDTO> pageList = new ArrayList<>();
    for (ServiceOrderTaskDO item : orderTaskList) {
      ServiceOrderTaskDTO orderTaskDTO = getServiceOrderTaskDTO(provinceCityMap, imageMap, item);
      pageList.add(orderTaskDTO);
    }
    return PageResult.buildPage(pageNo, param.getPageSize(), count, pageList);
  }

  @Override
  public ResultBody<ServiceOrderTaskDTO> queryById(Integer id) {
    ServiceOrderTaskDO order = serviceOrderTaskDao.selectById(id);
    if(order==null){
      return ResultBody.success();
    }
    Map<Integer, ServiceOrderProvinceCityDO> provinceCityMap = getOrderProvinceCityDOMapById(Arrays.asList(id));
    Map<Integer, List<ServiceOrderTaskImageDO>> imageMap = getImageListMapById(Arrays.asList(id));
    ServiceOrderTaskDTO serviceOrderTaskDTO = getServiceOrderTaskDTO(provinceCityMap, imageMap, order);
    return ResultBody.success(serviceOrderTaskDTO);
  }

  private ServiceOrderTaskDTO getServiceOrderTaskDTO(Map<Integer, ServiceOrderProvinceCityDO> provinceCityMap, Map<Integer, List<ServiceOrderTaskImageDO>> imageMap, ServiceOrderTaskDO item) {
    ServiceOrderTaskDTO orderTaskDTO = BeanCopyUtils.properties(item, new ServiceOrderTaskDTO());
    BeanCopyUtils.properties(provinceCityMap.get(item.getId()), orderTaskDTO);
    List<ServiceOrderTaskImageDO> imageDOList = imageMap.get(item.getId());
    List<String> images = null;
    if (CollectionUtils.isNotEmpty(imageDOList)) {
      images =
          imageDOList.stream()
              .map(ServiceOrderTaskImageDO::getPicture)
              .collect(Collectors.toList());
    }
    orderTaskDTO.setImages(images);
    return orderTaskDTO;
  }

  private Map<Integer, List<ServiceOrderTaskImageDO>> getImageListMapById(List<Integer> orderTaskIds) {
    LambdaQueryWrapper<ServiceOrderTaskImageDO> imageWrapper = new LambdaQueryWrapper<>();
    imageWrapper.in(ServiceOrderTaskImageDO::getOrderTaskId, orderTaskIds);
    List<ServiceOrderTaskImageDO> imageList = serviceOrderTaskImageDao.selectList(imageWrapper);
    Map<Integer, List<ServiceOrderTaskImageDO>> imageMap = new HashMap<>();
    if (CollectionUtils.isNotEmpty(imageList)) {
      imageMap =
          imageList.stream()
              .collect(Collectors.groupingBy(ServiceOrderTaskImageDO::getOrderTaskId));
    }
    return imageMap;
  }

  private Map<Integer, ServiceOrderProvinceCityDO> getOrderProvinceCityDOMapById(List<Integer> orderTaskIds) {
    LambdaQueryWrapper<ServiceOrderProvinceCityDO> provinceCityWrapper = new LambdaQueryWrapper<>();
    provinceCityWrapper.in(ServiceOrderProvinceCityDO::getServiceOrderTaskId, orderTaskIds);
    List<ServiceOrderProvinceCityDO> orderProvinceCityDOList =
        serviceOrderProvinceCityDao.selectList(provinceCityWrapper);
    Map<Integer, ServiceOrderProvinceCityDO> provinceCityMap =
        orderProvinceCityDOList.stream()
            .collect(
                Collectors.toMap(
                    ServiceOrderProvinceCityDO::getServiceOrderTaskId, d -> d, (k1, k2) -> k1));
    return provinceCityMap;
  }
}
