package com.mmc.pms.service.Impl;

import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.CategoriesDao;
import com.mmc.pms.dao.GoodsInfoDao;
import com.mmc.pms.entity.Categories;
import com.mmc.pms.entity.Directory;
import com.mmc.pms.entity.GoodsInfo;
import com.mmc.pms.model.categories.dto.ClassifyInfoDTO;
import com.mmc.pms.model.categories.vo.ClassifyInfoVO;
import com.mmc.pms.model.categories.vo.DirectoryInfoVO;
import com.mmc.pms.model.categories.vo.RelevantBusinessVO;
import com.mmc.pms.model.sale.vo.QueryClassifyVO;
import com.mmc.pms.page.PageResult;
import com.mmc.pms.service.CategoriesService;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author lw
 * @description 针对表【categories(通用分类表)】的数据库操作Service实现
 * @createDate 2023-05-24 10:29:28
 */
@Service
public class CategoriesServiceImpl implements CategoriesService {
    @Autowired
    private CategoriesDao categoriesDao;
    @Resource
    private GoodsInfoDao goodsInfoDao;

    @Override
    public ResultBody addOrEditDirectory(DirectoryInfoVO param) {
        int type = categoriesDao.countUpdateDirectoryName(param);
        if (type > 0) {
            return ResultBody.error(ResultEnum.DIRECTORY_NAME_HAS_BEEN_EXIST);
        }
        Directory directory = new Directory(param);
        if (param.getId() == null) {
            categoriesDao.insertDirectory(directory);
        } else {
            int count = categoriesDao.selectDirectoryById(param.getId());
            if (count > 0) {
                return ResultBody.error("默认目录不可修改！");
            }
            categoriesDao.updateDirectory(directory);
        }
        return ResultBody.success();
    }

    @Override
    public PageResult directoryList(Integer pageNo, Integer pageSize, Integer type) {
        int count = categoriesDao.countDirectoryList();
        if (count == 0) {
            return PageResult.buildPage(pageNo, pageSize, count);
        }
        List<Directory> directoryList = categoriesDao.directoryList((pageNo - 1) * pageSize, pageSize, type);
        List<DirectoryInfoVO> list = directoryList.stream().map(Directory::buildDirectoryInfoVO).collect(Collectors.toList());
        return PageResult.buildPage(pageNo, pageSize, count, list);
    }

    @Override
    public ResultBody removeDirectory(Integer id) {
        // 查询该目录下是否有分类有的话就不给删除
        int count = categoriesDao.countDirectory(id);
        if (count > 0) {
            return ResultBody.error(ResultEnum.THERE_ARE_CATEGORIES_IN_THE_DIRECTORY);
        }
        categoriesDao.removeDirectory(id);
        return ResultBody.success();
    }

    @Override
    public ResultBody addClassification(ClassifyInfoVO classifyInfoVO) {
        int count = categoriesDao.countClassificationByName(classifyInfoVO);
        if (count > 0) {
            return ResultBody.error(ResultEnum.WARE_TYPE_NAME_HAS_BEEN_EXIST);
        }
        Categories categories = new Categories(classifyInfoVO);
        if (classifyInfoVO.getPid() == 0) {
            if (classifyInfoVO.getIcon() != null) {
                int typeCount = categoriesDao.getCountCategoriesByPid(0, classifyInfoVO.getType());
                categories.setSort(typeCount + 1);
                categoriesDao.insertClassification(categories);
            } else {
                return ResultBody.error(ResultEnum.WARE_TYPE_ICON_NOT_NULL);
            }
        } else {
            categoriesDao.insertClassification(categories);
        }
        return ResultBody.success();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody updateClassification(ClassifyInfoVO classifyInfoVO) {
        int count = categoriesDao.countClassificationByName(classifyInfoVO);
        if (count > 0) {
            return ResultBody.error(ResultEnum.WARE_TYPE_NAME_HAS_BEEN_EXIST);
        }
        if (classifyInfoVO.getPid() == 0) {
            if (classifyInfoVO.getIcon() != null) {
                categoriesDao.updateClassification(classifyInfoVO);
            } else {
                return ResultBody.error(ResultEnum.WARE_TYPE_ICON_NOT_NULL);
            }
        } else {
            categoriesDao.updateClassification(classifyInfoVO);
        }
        return ResultBody.success();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody exchangeSortType(Integer firstId, Integer secondId) {
        Categories firstCategories = categoriesDao.getGoodsGroupById(firstId);
        Categories secondCategories = categoriesDao.getGoodsGroupById(secondId);
        int updateCount1 = categoriesDao.updateTypeSort(firstId, secondCategories.getSort());
        int updateCount2 = categoriesDao.updateTypeSort(secondId, firstCategories.getSort());
        if (updateCount1 == updateCount2) {
            return ResultBody.success();
        } else {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            return ResultBody.error("排序失败");
        }
    }

    @Override
    public PageResult getClassificationList(QueryClassifyVO queryClassifyVO) {
        int count = categoriesDao.countListClassification(queryClassifyVO);
        if (count == 0) {
            return PageResult.buildPage(queryClassifyVO.getPageNo(), queryClassifyVO.getPageSize(), count);
        }

        int pageNo = queryClassifyVO.getPageNo();
        queryClassifyVO.buildCurrentPage();
        List<Categories> categories = categoriesDao.selectAllClassification(queryClassifyVO);
        List<ClassifyInfoDTO> categoriesList = categories.stream().map(Categories::buildClassifyInfoDTO).collect(Collectors.toList());

        List<ClassifyInfoDTO> topLevelCategories = new ArrayList<>();
        Map<Integer, ClassifyInfoDTO> categoriesMap = new HashMap<>();

        // 将每个数据模型对象添加到Map中，以便在递归过程中查找它们的父母
        for (ClassifyInfoDTO category : categoriesList) {
            category.setChildren(new ArrayList<>());
            categoriesMap.put(category.getId(), category);
        }
        // 构建树结构
        for (ClassifyInfoDTO category : categoriesList) {
            if (category.getPid() == 0) {
                topLevelCategories.add(category);
            } else {
                ClassifyInfoDTO parent = categoriesMap.get(category.getPid());
                parent.getChildren().add(category);
            }
        }
        return PageResult.buildPage(pageNo, queryClassifyVO.getPageSize(), count, topLevelCategories);
    }

    @Override
    public ResultBody getClassifyDetails(Integer id) {
        Categories goodsGroup = categoriesDao.getGoodsGroupById(id);
        return ResultBody.success(goodsGroup == null ? null : goodsGroup.buildClassifyDetailsDTO()
        );
    }

    @Override
    public ResultBody queryRelevantBusiness(Integer id, Integer type) {
        RelevantBusinessVO relevantBusinessVO = new RelevantBusinessVO();
        switch (type) {
            case 0:
                List<GoodsInfo> goodsInfo = goodsInfoDao.ListGoodsInfoByCategoryId(id);
                if (CollectionUtils.isNotEmpty(goodsInfo)) {
                    relevantBusinessVO.setRelevanceGoodsInfoVOs(goodsInfo.stream().map(GoodsInfo::buildRelevanceGoodsInfoVO).collect(Collectors.toList()));
                    return ResultBody.success(relevantBusinessVO);
                }
                break;
            default:
                return ResultBody.error("输入类型有误！");
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody getDirectoryList(Integer type) {
        List<Directory> directoryList = categoriesDao.getDirectoryList(type);
        List<DirectoryInfoVO> list = directoryList.stream().map(Directory::buildDirectoryInfoVO).collect(Collectors.toList());
        return ResultBody.success(list);
    }


    @Override
    public ResultBody deleteRelevantBusiness(Integer id) {
        int count = categoriesDao.deleteById(id);
        return ResultBody.success();
    }


}




