package com.mmc.pms.service.Impl;

import com.mmc.pms.common.ResultBody;
import com.mmc.pms.common.ResultEnum;
import com.mmc.pms.dao.GoodsInfoDao;
import com.mmc.pms.dao.ProductSkuDao;
import com.mmc.pms.entity.*;
import com.mmc.pms.model.vo.*;
import com.mmc.pms.service.GoodsInfoService;
import com.mmc.pms.util.CodeUtil;
import com.mmc.pms.util.TDateUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author 23214
 * @description 针对表【goods_info(商品基本信息)】的数据库操作Service实现
 * @createDate 2023-05-27 14:08:45
 */
@Service
public class GoodsInfoServiceImpl implements GoodsInfoService {

    @Resource
    private GoodsInfoDao goodsInfoDao;

    @Resource
    private ProductSkuDao productSkuDao;

    @Resource
    private ProductSkuServiceImpl productSkuService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody addGoods(GoodsAddVO goodsAddVO) {
        // 判断商品名称是否存在
        if (goodsInfoDao.countGoodsInfoByName(goodsAddVO) > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        // 判断商品详情中描述是否为空
        if (goodsAddVO.getGoodsDetailVO().getGoodsDesc() == null) {
            return ResultBody.error(ResultEnum.GOODS_DESC_IS_NOT_NULL);
        }
        String YYYY_MM_DD_HH_MM_SS = "yyyyMMddHHmmss";
        // 初始化商品对象,并构建对象
        GoodsInfo goodsInfo = new GoodsInfo(goodsAddVO);
        goodsInfo.setGoodsNo("IUAV" + TDateUtil.getDateStr(new Date(), YYYY_MM_DD_HH_MM_SS) + CodeUtil.getRandomNum(4));
        // 暂未接入用户默认传1
        goodsInfo.setAddGoodsUserId(1);
        goodsInfo.setSort(goodsInfoDao.countGoodsInfo() + 1);
        // 插入商品基本信息
        goodsInfoDao.insertGoodsInfo(goodsInfo);
        // 调用图片视频插入信息方法
        addGoodsImageInfo(goodsInfo.getId(), goodsAddVO);
        // 调用商品详情信息方法
        addGoodsDetail(goodsInfo.getId(), goodsAddVO.getGoodsDetailVO());
        // 判断其他服务是否为空,不为空则插入
        if (!CollectionUtils.isEmpty(goodsAddVO.getOtherService())) {
            addOtherService(goodsAddVO.getOtherService(), goodsInfo.getId());
        }
        // 调用产品规格的新增方法
        if (!goodsAddVO.getDirectoryId().equals(2)) {
            // 添加产品规格信息
            productSkuSpecOperation(goodsInfo, goodsAddVO.getProductSpec());
        }
        return ResultBody.success();
    }

    @Transactional(rollbackFor = Exception.class)
    public void productSkuSpecOperation(GoodsInfo goodsInfo, List<GoodsProdSpecVO> productSpec) {
        // 遍历规格信息，获取其中非自定义的规格信息
        List<GoodsProdSpecVO> customGoodsSpecList = productSpec.stream().filter(spec -> spec.getFlag().equals(1)).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(customGoodsSpecList)) {
            addCustomization(goodsInfo, customGoodsSpecList);
        }
        // 遍历规格信息，获取其中非自定义的规格信息
        List<GoodsProdSpecVO> goodsSpecList = productSpec.stream().filter(spec -> spec.getFlag().equals(0)).collect(Collectors.toList());
        for (GoodsProdSpecVO goodsSpec : goodsSpecList) {
            MallProdSkuInfoDO mallProdSkuInfoDO = buildMallProdSkuInfo(goodsInfo, goodsSpec, goodsSpec.getSkuId());
            mallProdSkuInfoDO.setProductSpecIdList(goodsSpec.getSpecIds().stream().map(String::valueOf).collect(Collectors.joining(",")));
            // 插入数据库商品对应的sku信息
            productSkuDao.insertMallProdSkuInfo(mallProdSkuInfoDO);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public MallProdSkuInfoDO buildMallProdSkuInfo(GoodsInfo goodsInfo, GoodsProdSpecVO goodsSpec, Integer id) {
        MallProdSkuInfoDO mallProdSkuInfoDO = new MallProdSkuInfoDO();
        mallProdSkuInfoDO.setGoodsInfoId(goodsInfo.getId());
        mallProdSkuInfoDO.setProdSkuId(id);
        mallProdSkuInfoDO.setProdSkuSpecName(goodsSpec.getGoodsSpecName());
        mallProdSkuInfoDO.setCategoryId(goodsSpec.getCategoryId());
        mallProdSkuInfoDO.setChooseType(goodsSpec.getChooseType());
        mallProdSkuInfoDO.setMust(goodsSpec.getMust());
        mallProdSkuInfoDO.setFlag(goodsSpec.getFlag());
        mallProdSkuInfoDO.setSkuUnitId(goodsSpec.getSkuUnitId());
        return mallProdSkuInfoDO;
    }

    @Transactional(rollbackFor = Exception.class)
    public void addCustomization(GoodsInfo goodsInfo, List<GoodsProdSpecVO> customGoodsSpecList) {
        // 遍历自定义规格信息
        for (GoodsProdSpecVO productSpec : customGoodsSpecList) {
            // 构建商品对应的sku信息
            ProductSkuDO productSkuDO = new ProductSkuDO().setCategoriesId(productSpec.getCategoryId())
                    .setProductName(productSpec.getProductName()).setCustomize(1)
                    .setDirectoryId(goodsInfo.getDirectoryId());
            // 插入产品sku信息
            productSkuDao.insertProductSku(productSkuDO);
            // 先将自定的信息存储到数据库中
            List<ProductSpecDO> customizeSpec = productSpec.getCustomizeInfo().stream().map(param -> {
                ProductSpecDO productSpecDO = new ProductSpecDO().setProductSkuId(productSkuDO.getId())
                        .setVersionDesc("自定义").setSpecImage(param.getSpecImage())
                        .setSpecName(param.getSpecName()).setPartNo(param.getPartNo());
                // 新增产品sku
                productSkuDao.insertProductSpec(productSpecDO);
                // 批量配置价格信息
                ProductSpecCPQVO productSpecCPQVO = param.getProductSpecCPQVO();
                if (productSpecCPQVO != null) {
                    productSpecCPQVO.setProductSpecId(productSpecDO.getId());
                    productSkuService.insertSpecPrice(productSpecCPQVO);
                }
                return productSpecDO;
            }).collect(Collectors.toList());
            List<Integer> ids = customizeSpec.stream().map(ProductSpecDO::getId).collect(Collectors.toList());
            MallProdSkuInfoDO mallProdSkuInfoDO = buildMallProdSkuInfo(goodsInfo, productSpec, productSkuDO.getId());
            mallProdSkuInfoDO.setProductSpecIdList(ids.stream().map(String::valueOf).collect(Collectors.joining(",")));
            // 插入数据库商品对应的sku信息
            productSkuDao.insertMallProdSkuInfo(mallProdSkuInfoDO);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void addOtherService(List<Integer> otherService, Integer goodsId) {
        // 插入其他服务
        List<GoodsServiceDO> otherList = otherService.stream().map(d -> {
            GoodsServiceDO goodsServiceDO = new GoodsServiceDO();
            goodsServiceDO.setGoodsInfoId(goodsId);
            goodsServiceDO.setSaleServiceId(d);
            return goodsServiceDO;
        }).collect(Collectors.toList());
        goodsInfoDao.insertGoodsService(otherList);
    }

    @Transactional(rollbackFor = Exception.class)
    public void addGoodsDetail(Integer goodsId, GoodsDetailVO goodsDetailVO) {
        GoodsDetailDO goodsDetailDO = new GoodsDetailDO(goodsDetailVO);
        goodsDetailDO.setGoodsInfoId(goodsId);
        goodsInfoDao.insertGoodsDetail(goodsDetailDO);
    }

    @Transactional(rollbackFor = Exception.class)
    public void addGoodsImageInfo(Integer goodsId, GoodsAddVO goodsAddVO) {
        List<GoodsImgDO> list = goodsAddVO.getImages().stream().map(d -> {
            GoodsImgDO goodsImgDO = new GoodsImgDO(d);
            goodsImgDO.setGoodsInfoId(goodsId);
            return goodsImgDO;
        }).collect(Collectors.toList());
        // 插入图片信息
        goodsInfoDao.insertGoodsImgInfo(list);
        // 插入商品视频信息
        if (goodsAddVO.getGoodsVideo() != null) {
            GoodsVideoDO goodsVideoDO = new GoodsVideoDO().setGoodsInfoId(goodsId).setVideoUrl(goodsAddVO.getGoodsVideo());
            goodsInfoDao.insertVideoInfo(goodsVideoDO);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody editGoodsInfo(GoodsAddVO goodsAddVO) {
        // 判断商品是否还存在
        int count = goodsInfoDao.countGoodsInfoById(goodsAddVO.getId());
        if (count <= 0) {
            return ResultBody.error(ResultEnum.GOODS_NOT_EXIST_OR_ALREADY_DOWN_SHELF);
        }
        // 判断商品名称是否存在
        if (goodsInfoDao.countGoodsInfoByName(goodsAddVO) > 0) {
            return ResultBody.error(ResultEnum.GOODS_CATEGORY_NAME_EXIST_ERROR);
        }
        // 判断商品详情中描述是否为空
        if (goodsAddVO.getGoodsDetailVO().getGoodsDesc() == null) {
            return ResultBody.error(ResultEnum.GOODS_DESC_IS_NOT_NULL);
        }
        // 初始化商品对象,并构建对象
        GoodsInfo goodsInfo = new GoodsInfo(goodsAddVO);
        // 插入商品基本信息
        goodsInfoDao.updateGoodsInfo(goodsInfo);
        // 修改商品详情信息
        updateGoodsDetail(goodsAddVO);
        // 修改商品图片及视频信息
        updateImageInfo(goodsAddVO);
        // 修改其他服务信息
        updateOtherService(goodsAddVO);

        return null;
    }

    private void updateOtherService(GoodsAddVO goodsAddVO) {
        // 删除其他服务
        goodsInfoDao.deleteGoodsServiceByGoodsId(goodsAddVO.getId());
        // 插入其他服务
        if (!CollectionUtils.isEmpty(goodsAddVO.getOtherService())) {
            List<Integer> other = goodsAddVO.getOtherService();
            List<GoodsServiceDO> otherList = other.stream().map(d -> {
                GoodsServiceDO goodsServiceDO = new GoodsServiceDO();
                goodsServiceDO.setGoodsInfoId(goodsAddVO.getId());
                goodsServiceDO.setSaleServiceId(d);
                return goodsServiceDO;
            }).collect(Collectors.toList());
            goodsInfoDao.insertGoodsService(otherList);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateImageInfo(GoodsAddVO goodsAddVO) {
        Set<Integer> imgIds = goodsAddVO.getImages().stream().map(GoodsImgVO::getId).filter(Objects::nonNull).collect(Collectors.toSet());
        List<GoodsImgDO> dbImgList = goodsInfoDao.listGoodsInfoByGoodsId(goodsAddVO.getId());
        List<Integer> deleteIds = dbImgList.stream().map(GoodsImgDO::getId).filter(id -> !imgIds.contains(id)).collect(Collectors.toList());
        if (deleteIds.size() != 0) {
            // 删除多余的图片
            goodsInfoDao.deleteImgByIds(deleteIds);
        }
        // 新增图片
        List<GoodsImgDO> imgDOList = goodsAddVO.getImages().stream().filter(d -> d.getId() == null).map(d -> {
            GoodsImgDO goodsImgDO = new GoodsImgDO(d);
            goodsImgDO.setGoodsInfoId(goodsAddVO.getId());
            return goodsImgDO;
        }).collect(Collectors.toList());
        if (imgDOList.size() != 0) {
            goodsInfoDao.insertGoodsImgInfo(imgDOList);
        }
        // 删除视频
        goodsInfoDao.deleteGoodsVideoById(goodsAddVO.getId());
        // 插入视频
        if (goodsAddVO.getGoodsVideo() != null) {
            GoodsVideoDO goodsVideoDO = new GoodsVideoDO();
            goodsVideoDO.setGoodsInfoId(goodsAddVO.getId());
            goodsVideoDO.setVideoUrl(goodsAddVO.getGoodsVideo());
            goodsInfoDao.insertVideoInfo(goodsVideoDO);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateGoodsDetail(GoodsAddVO goodsAddVO) {
        GoodsDetailDO goodsDetailDO = new GoodsDetailDO(goodsAddVO.getGoodsDetailVO()).setGoodsInfoId(goodsAddVO.getId());
        // 商品详情修改
        goodsInfoDao.updateGoodsDetail(goodsDetailDO);
    }
}




