package com.mmc.payment.service.Impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.RandomUtil;
import com.alibaba.fastjson2.JSONObject;
import com.mmc.payment.common.result.ResultBody;
import com.mmc.payment.common.util.CodeUtil;
import com.mmc.payment.common.util.TDateUtil;
import com.mmc.payment.constant.UserSystemConstant;
import com.mmc.payment.dao.WechatPayDao;
import com.mmc.payment.entity.order.ApplyRefundLogDO;
import com.mmc.payment.entity.order.WxPayLogDO;
import com.mmc.payment.enums.OrderPayStatus;
import com.mmc.payment.feign.UserAppApi;
import com.mmc.payment.model.dto.user.UserAccountSimpleDTO;
import com.mmc.payment.model.vo.order.ApplyRefundVO;
import com.mmc.payment.model.vo.order.OrderRequestParamsVO;
import com.mmc.payment.model.vo.order.UserPayInfoVO;
import com.mmc.payment.model.vo.order.WxPayLogVO;
import com.mmc.payment.model.vo.wallet.TopUpOrderVO;
import com.mmc.payment.mq.PublishMsg;
import com.mmc.payment.mq.constant.RabbitmqConstant;
import com.mmc.payment.service.WechatPayService;
import com.mmc.payment.util.WxConfigUtils;
import com.wechat.pay.java.core.Config;
import com.wechat.pay.java.core.cipher.SignatureResult;
import com.wechat.pay.java.core.exception.HttpException;
import com.wechat.pay.java.core.exception.MalformedMessageException;
import com.wechat.pay.java.core.exception.ServiceException;
import com.wechat.pay.java.core.notification.NotificationConfig;
import com.wechat.pay.java.core.notification.NotificationParser;
import com.wechat.pay.java.core.notification.RequestParam;
import com.wechat.pay.java.service.payments.jsapi.JsapiService;
import com.wechat.pay.java.service.payments.jsapi.model.Amount;
import com.wechat.pay.java.service.payments.jsapi.model.*;
import com.wechat.pay.java.service.payments.model.Transaction;
import com.wechat.pay.java.service.refund.RefundService;
import com.wechat.pay.java.service.refund.model.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * @Author LW
 * @date 2023/7/19 16:56
 * 概要：
 */
@Service
@Slf4j
public class WechatPayServiceImpl implements WechatPayService {
    @Resource
    UserSystemConstant userSystemConstant;
    @Resource
    WechatPayDao wechatPayDao;
    @Resource
    UserAppApi userAppApi;
    @Resource
    WxConfigUtils wxConfigUtils;
    @Resource
    PublishMsg publishMsg;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map orderPay(OrderRequestParamsVO orderRequestParamsVO, Integer userAccountId, HttpServletRequest request) {
        UserAccountSimpleDTO userSimpleInfo = userAppApi.feignGetUserSimpleInfo(userAccountId, request.getHeader("token"));
        if (userSimpleInfo == null) {
            throw new RuntimeException("服务器内部错误！");
        }
        if(orderRequestParamsVO.getChatType() != null && orderRequestParamsVO.getOrderPort() == 7){
            Integer integer = userAppApi.timeTypeById(orderRequestParamsVO.getChatType(), userAccountId, request.getHeader("token"));
            orderRequestParamsVO.setAmount(integer);
        }
        Config config = wxConfigUtils.createConfig();
        // 查询该订单是否已下单
        WxPayLogDO wxPayLogDO = wechatPayDao.selectWxPayInfoByOrderNo(orderRequestParamsVO.getOrderNo());
        if (wxPayLogDO != null && wxPayLogDO.getTradeState().equals(OrderPayStatus.SUCCESS.getStatus())) {
            throw new RuntimeException("订单已支付!");
        } else if (wxPayLogDO == null) {
            // 构建service
            JsapiService jsapiService = new JsapiService.Builder().config(config).build();
            // request.setXxx(val)设置所需参数，具体参数可见Request定义
            PrepayRequest prepayRequest = new PrepayRequest();
            Amount amount = new Amount();
            amount.setTotal(orderRequestParamsVO.getAmount());
            prepayRequest.setAmount(amount);
            Payer payer = new Payer();
            payer.setOpenid(userSimpleInfo.getOpenid());
            prepayRequest.setPayer(payer);
            prepayRequest.setAttach(orderRequestParamsVO.getAttach());
            prepayRequest.setAppid(userSystemConstant.getWxAppId());
            prepayRequest.setMchid(userSystemConstant.getMchid());
            prepayRequest.setDescription(orderRequestParamsVO.getDescription());
            prepayRequest.setNotifyUrl(userSystemConstant.getNotifyUrl());
            prepayRequest.setOutTradeNo(orderRequestParamsVO.getOrderNo());
            // 调用下单方法，得到应答
            PrepayResponse prepay = jsapiService.prepay(prepayRequest);
            String prepayId = prepay.getPrepayId();
            // 录入数据库记录数据
            WxPayLogDO wxPrepayLogDO = new WxPayLogDO();
            wxPrepayLogDO.setOrderPort(orderRequestParamsVO.getOrderPort());
            wxPrepayLogDO.setOrderNo(orderRequestParamsVO.getOrderNo());
            wxPrepayLogDO.setDescription(orderRequestParamsVO.getDescription());
            wxPrepayLogDO.setUserAccountId(userAccountId);
            wxPrepayLogDO.setAmount(orderRequestParamsVO.getAmount());
            wxPrepayLogDO.setPrepayId(prepayId);
            wxPrepayLogDO.setOpenId(userSimpleInfo.getOpenid());
            wxPrepayLogDO.setTradeState(OrderPayStatus.WAIT.getStatus());
            // 往数据库插入下单的日志信息
            wechatPayDao.insertWxPayLog(wxPrepayLogDO);
            Map<String, Object> map = getSignInfoMap(config, prepayId);
            return map;
        } else {
            Map<String, Object> map = getSignInfoMap(config, wxPayLogDO.getPrepayId());
            return map;
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public Map<String, Object> getSignInfoMap(Config config, String prepayId) {
        // 获取时间戳
        String timeStamp = System.currentTimeMillis() / 1000 + "";
        // 获取随机字符串
        String nonceStr = RandomUtil.randomString(32);
        // 签名方式
        String signType = "RSA";
        // 订单详情扩展字符串
        String prepayPackage = "prepay_id=" + prepayId;
        // 构造签名串
        StringBuilder sb = new StringBuilder();
        sb.append(userSystemConstant.getWxAppId()).append("\n");
        sb.append(timeStamp).append("\n");
        sb.append(nonceStr).append("\n");
        sb.append(prepayPackage).append("\n");
        // 生成签名
        SignatureResult sign = config.createSigner().sign(sb.toString());
        Map<String, Object> map = new HashMap<>(16);
        map.put("timeStamp", timeStamp);
        map.put("nonceStr", nonceStr);
        map.put("package", prepayPackage);
        map.put("signType", signType);
        map.put("paySign", sign);
        return map;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map payCallback(HttpServletRequest request) {
        Map<String, String> result = new HashMap(16);
        result.put("code", "FAIL");
        // 获取应答时间戳
        String timestamp = request.getHeader("Wechatpay-Timestamp");
        // 获取应答随机串
        String nonce = request.getHeader("Wechatpay-Nonce");
        // 获取应答签名
        String signature = request.getHeader("Wechatpay-Signature");
        // 获取应答序列号
        String serialNumber = request.getHeader("Wechatpay-Serial");
        log.info("应答时间戳: {},应答随机串:{},应答签名:{},应答序列号{}", timestamp, nonce, signature, serialNumber);
        try {
            // 获取body请求报文
            BufferedReader br = request.getReader();
            String str = null;
            StringBuilder sb = new StringBuilder();
            while ((str = br.readLine()) != null) {
                sb.append(str);
            }
            log.info("请求体数据：{}", sb);
            NotificationConfig config = wxConfigUtils.createNotificationConfig();
            // 构造 RequestParam
            RequestParam requestParam = new RequestParam.Builder()
                    .serialNumber(serialNumber)
                    .nonce(nonce)
                    .signature(signature)
                    .timestamp(timestamp)
                    .body(sb.toString())
                    .build();
            // 初始化 NotificationParser
            NotificationParser parser = new NotificationParser(config);
            // 以支付通知回调为例，验签、解密并转换成 Transaction
            Transaction transaction = parser.parse(requestParam, Transaction.class);
            log.info("解密resource数据：{}", transaction);
            WxPayLogDO wxPayLogDO = new WxPayLogDO();
            wxPayLogDO.setOutTradeNo(transaction.getOutTradeNo());
            wxPayLogDO.setTransactionId(transaction.getTransactionId());
            wxPayLogDO.setBankType(transaction.getBankType());
            wxPayLogDO.setAttach(transaction.getAttach());
            wxPayLogDO.setSuccessTime(transaction.getSuccessTime());
            wxPayLogDO.setWxNotifyOpenid(transaction.getPayer().getOpenid());
            wxPayLogDO.setWxNotifyTotal(transaction.getAmount().getTotal());
            wxPayLogDO.setWxNotifyPayerTotal(transaction.getAmount().getPayerTotal());
            wxPayLogDO.setTradeType(transaction.getTradeType().toString());
            wxPayLogDO.setTradeState(transaction.getTradeState().toString());
            wxPayLogDO.setTradeStateDesc(transaction.getTradeStateDesc());
            wechatPayDao.updateWxPayLog(wxPayLogDO);
            result.put("code", "SUCCESS");
            if ("TOP_UP".equals(transaction.getAttach())) {
                // 充值订单，发送mq消息进行操作
                sendTopUpMsg(transaction, 0);
            }
            if ("TOP_UP".equals(transaction.getAttach()) && "SUCCESS".equals(transaction.getTradeState().toString())) {
                // 发送充值成功消息
                sendTopUpMsg(transaction, 1);
            }
            if ("PAY_UAV_ORDER".equals(transaction.getAttach()) && "SUCCESS".equals(transaction.getTradeState().toString())) {
                // 交易成功
                sendTopUpMsg(transaction, 2);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return result;
    }

    private void sendTopUpMsg(Transaction transaction, Integer flag) {
        TopUpOrderVO topUpOrderVO = new TopUpOrderVO();
        topUpOrderVO.setOpenid(transaction.getPayer().getOpenid());
        topUpOrderVO.setOrderNo(transaction.getOutTradeNo());
        topUpOrderVO.setAmount(transaction.getAmount().getTotal());
        topUpOrderVO.setTradeState(transaction.getTradeState().toString());
        topUpOrderVO.setTradeStateDesc(transaction.getTradeStateDesc());
        if (flag == 0) {
            publishMsg.sendNewTopUpOrder(topUpOrderVO, RabbitmqConstant.USER_TOP_UP_ROUTING_KEY);
        }
        if (flag == 1) {
            publishMsg.sendNewTopUpOrder(topUpOrderVO, RabbitmqConstant.USER_TOP_UP_SUCCESS_ROUTING_KEY);
        }
        if (flag == 2) {
            publishMsg.sendPayUavOrder(topUpOrderVO, RabbitmqConstant.PAY_UAV_ORDER_SUCCESS_ROUTING_KEY);
        }
    }

    @Override
    public ResultBody closeOrder(String orderNo) {
        Config config = wxConfigUtils.createConfig();
        // 构建service
        JsapiService jsapiService = new JsapiService.Builder().config(config).build();
        CloseOrderRequest closeRequest = new CloseOrderRequest();
        closeRequest.setMchid(userSystemConstant.getMchid());
        closeRequest.setOutTradeNo(orderNo);
        // 方法没有返回值，意味着成功时API返回204 No Content
        jsapiService.closeOrder(closeRequest);
        // 删除数据库记录
        wechatPayDao.deleteWxPayLogByOrderNo(orderNo);
        return ResultBody.success();
    }

    @Override
    public ResultBody queryOrder(String transactionId, String orderNo) {
        try {

            Config config = wxConfigUtils.createConfig();
            // 构建service
            JsapiService service = new JsapiService.Builder().config(config).build();
            if (orderNo != null) {
                QueryOrderByOutTradeNoRequest queryRequest = new QueryOrderByOutTradeNoRequest();
                queryRequest.setOutTradeNo(orderNo);
                queryRequest.setMchid(userSystemConstant.getMchid());
                Transaction result = service.queryOrderByOutTradeNo(queryRequest);
                return getTransactionInfo(result);
            }
            if (transactionId != null) {
                QueryOrderByIdRequest queryRequest = new QueryOrderByIdRequest();
                queryRequest.setTransactionId(transactionId);
                queryRequest.setMchid(userSystemConstant.getMchid());
                Transaction result = service.queryOrderById(queryRequest);
                return getTransactionInfo(result);
            }
        } catch (ServiceException e) {
            String responseBody = e.getResponseBody();
            JSONObject jsonObject = JSONObject.parseObject(responseBody);
            return ResultBody.error(jsonObject.getString("message"));
        }
        return ResultBody.success();
    }

    private ResultBody getTransactionInfo(Transaction result) {
        WxPayLogVO wxPayLogVO = new WxPayLogVO();
        wxPayLogVO.setOutTradeNo(result.getOutTradeNo());
        wxPayLogVO.setBankType(result.getBankType());
        wxPayLogVO.setAttach(result.getAttach());
        wxPayLogVO.setSuccessTime(result.getSuccessTime());
        wxPayLogVO.setWxNotifyOpenid(result.getPayer() == null ? null : result.getPayer().getOpenid());
        wxPayLogVO.setWxNotifyTotal(result.getAmount() == null ? null : result.getAmount().getTotal());
        wxPayLogVO.setWxNotifyPayerTotal(result.getAmount() == null ? null : result.getAmount().getPayerTotal());
        wxPayLogVO.setTradeType(result.getTradeType() == null ? null : result.getTradeType().toString());
        wxPayLogVO.setTradeState(result.getTradeState() == null ? null : result.getTradeState().toString());
        wxPayLogVO.setTradeStateDesc(result.getTradeStateDesc());
        return ResultBody.success(wxPayLogVO);
    }

    @Override
    public Map refundCallback(HttpServletRequest request) {
        Map<String, String> result = new HashMap(16);
        result.put("code", "FAIL");
        // 获取应答时间戳
        String timestamp = request.getHeader("Wechatpay-Timestamp");
        // 获取应答随机串
        String nonce = request.getHeader("Wechatpay-Nonce");
        // 获取应答签名
        String signature = request.getHeader("Wechatpay-Signature");
        // 获取应答序列号
        String serialNumber = request.getHeader("Wechatpay-Serial");
        log.info("应答时间戳: {},应答随机串:{},应答签名:{},应答序列号{}", timestamp, nonce, signature, serialNumber);
        try {
            // 获取body请求报文
            BufferedReader br = request.getReader();
            String str = null;
            StringBuilder sb = new StringBuilder();
            while ((str = br.readLine()) != null) {
                sb.append(str);
            }
            log.info("请求体数据：{}", sb);
            NotificationConfig config = wxConfigUtils.createNotificationConfig();
            // 构造 RequestParam
            RequestParam requestParam = new RequestParam.Builder()
                    .serialNumber(serialNumber)
                    .nonce(nonce)
                    .signature(signature)
                    .timestamp(timestamp)
                    .body(sb.toString())
                    .build();
            // 初始化 NotificationParser
            NotificationParser parser = new NotificationParser(config);
            // 验签、解密并转换成 Refund
            RefundNotification parse = parser.parse(requestParam, RefundNotification.class);
            log.info("解密resource数据：{}", parser);
            ApplyRefundLogDO applyRefundLogDO = new ApplyRefundLogDO();
            applyRefundLogDO.setNotifyRefundStatus(parse.getRefundStatus().toString());
            applyRefundLogDO.setOrderNo(parse.getOutTradeNo());
            applyRefundLogDO.setSuccessTime(parse.getSuccessTime());
            wechatPayDao.updateApplyRefundLog(applyRefundLogDO);
            result.put("code", "SUCCESS");
        } catch (IOException e) {
            e.printStackTrace();
        }
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ResultBody applyRefund(ApplyRefundVO applyRefundVO) {
        log.info("=======申请退款信息入库=======");
        ApplyRefundLogDO refundInfo = this.createRefundInfo(applyRefundVO);
        Config config = wxConfigUtils.createConfig();
        log.info("========调用微信退款接口=========");
        // 初始化服务
        RefundService service = new RefundService.Builder().config(config).build();
        try {
            CreateRequest request = new CreateRequest();
            request.setOutTradeNo(applyRefundVO.getOutTradeNo());
            request.setOutRefundNo(refundInfo.getOutRefundNo());
            request.setReason(refundInfo.getReason());
            // 创建退款金额对象
            AmountReq amountReq = new AmountReq();
            amountReq.setTotal(Long.valueOf(refundInfo.getTotal()));
            amountReq.setRefund(applyRefundVO.getRefund());
            amountReq.setCurrency("CNY");
            request.setNotifyUrl(userSystemConstant.getRefundNotifyUrl());
            request.setAmount(amountReq);
            if (CollectionUtil.isNotEmpty(applyRefundVO.getGoodsDetail())) {
                request.setGoodsDetail(applyRefundVO.getGoodsDetail());
            }
            Refund refund = service.create(request);
            this.updateRefundInfo(refund);
            return ResultBody.success();
        } catch (HttpException e) {
            log.info("申请退款HttpException日志打印：{}", e.getHttpRequest());
            return ResultBody.error("申请退款出错，请联系技术人员上服务器查询日志！");
        } catch (ServiceException e) {
            log.info("申请退款ServiceException日志打印：{}", e.getResponseBody());
            return ResultBody.error("申请退款出错，请联系技术人员上服务器查询日志！");
        } catch (MalformedMessageException e) {
            log.info("申请退款MalformedMessageException日志打印：{}", e.getMessage());
            return ResultBody.error("申请退款出错，请联系技术人员上服务器查询日志！");
        }
    }

    private void updateRefundInfo(Refund refund) {
        ApplyRefundLogDO applyRefundLogDO = new ApplyRefundLogDO();
        applyRefundLogDO.setRefundId(refund.getRefundId());
        applyRefundLogDO.setTransactionId(refund.getTransactionId());
        applyRefundLogDO.setChannel(refund.getChannel().toString());
        applyRefundLogDO.setUserReceivedAccount(refund.getUserReceivedAccount());
        applyRefundLogDO.setSuccessTime(refund.getSuccessTime());
        applyRefundLogDO.setCreateTime(refund.getCreateTime());
        applyRefundLogDO.setRefundStatus(refund.getStatus() == null ? null : refund.getStatus().toString());
        applyRefundLogDO.setFundsAccount(refund.getFundsAccount() == null ? null : refund.getFundsAccount().toString());
        applyRefundLogDO.setOrderNo(refund.getOutTradeNo());
        wechatPayDao.updateApplyRefundLog(applyRefundLogDO);
    }

    @Transactional(rollbackFor = Exception.class)
    public ApplyRefundLogDO createRefundInfo(ApplyRefundVO applyRefundVO) {
        // 生成商户退款订单号
        String outRefundNo = "RF" + TDateUtil.getDateStr(new Date(), "yyyyMMddHHmmss") + CodeUtil.getRandomNum(4);
        // 查询订单信息
        WxPayLogDO wxPayLogDO = wechatPayDao.selectWxPayInfoByOrderNo(applyRefundVO.getOutTradeNo());
        ApplyRefundLogDO applyRefundLogDO = new ApplyRefundLogDO();
        applyRefundLogDO.setReason(applyRefundVO.getReason());
        applyRefundLogDO.setOrderNo(applyRefundVO.getOutTradeNo());
        applyRefundLogDO.setOutRefundNo(outRefundNo);
        applyRefundLogDO.setRefund(applyRefundVO.getRefund());
        applyRefundLogDO.setTotal(wxPayLogDO.getAmount());
        applyRefundLogDO.setCurrency("CNY");
        wechatPayDao.insertApplyRefundLog(applyRefundLogDO);
        return applyRefundLogDO;
    }

    @Override
    public ResultBody queryRefund(String outRefundNo) {
        Config config = wxConfigUtils.createConfig();
        // 初始化服务
        RefundService service = new RefundService.Builder().config(config).build();
        QueryByOutRefundNoRequest request = new QueryByOutRefundNoRequest();
        request.setOutRefundNo(outRefundNo);
        Refund refund = service.queryByOutRefundNo(request);
        ApplyRefundLogDO applyRefundLogDO = new ApplyRefundLogDO();
        applyRefundLogDO.setRefundId(refund.getRefundId());
        applyRefundLogDO.setOutRefundNo(refund.getOutRefundNo());
        applyRefundLogDO.setTransactionId(refund.getTransactionId());
        applyRefundLogDO.setOrderNo(refund.getOutTradeNo());
        applyRefundLogDO.setChannel(refund.getChannel().toString());
        applyRefundLogDO.setUserReceivedAccount(refund.getUserReceivedAccount());
        applyRefundLogDO.setSuccessTime(refund.getSuccessTime() == null ? null : refund.getSuccessTime());
        applyRefundLogDO.setCreateTime(refund.getCreateTime());
        applyRefundLogDO.setRefundStatus(refund.getStatus().toString());
        applyRefundLogDO.setTotal(Math.toIntExact(refund.getAmount().getTotal()));
        applyRefundLogDO.setRefund(refund.getAmount().getRefund());
        applyRefundLogDO.setCurrency(refund.getAmount().getCurrency());
        return ResultBody.success(applyRefundLogDO);
    }

    @Override
    public UserPayInfoVO queryUserPayInfo(String orderNo) {
        WxPayLogDO wxPayLogDO = wechatPayDao.selectWxPayInfoByOrderNo(orderNo);
        UserPayInfoVO userPayInfoVO = new UserPayInfoVO();
        userPayInfoVO.setOutTradeNo(wxPayLogDO.getOutTradeNo());
        userPayInfoVO.setSuccessTime(wxPayLogDO.getSuccessTime());
        userPayInfoVO.setWxNotifyPayerTotal(wxPayLogDO.getWxNotifyPayerTotal());
        userPayInfoVO.setUserAccountId(wxPayLogDO.getUserAccountId());
        userPayInfoVO.setTradeState(wxPayLogDO.getTradeState());
        return userPayInfoVO;
    }
}
