package com.mmc.oms.service.uav.impl;

import com.mmc.oms.common.ResultEnum;
import com.mmc.oms.common.result.PageResult;
import com.mmc.oms.common.result.ResultBody;
import com.mmc.oms.dao.uav.UavOrderExpressDao;
import com.mmc.oms.dao.uav.UavPODao;
import com.mmc.oms.entity.uav.UavOrderExpressDO;
import com.mmc.oms.entity.uav.UavOrderPayDO;
import com.mmc.oms.entity.uav.UavPurchaseOrderDO;
import com.mmc.oms.entity.uav.UavPurchaseOrderPayDO;
import com.mmc.oms.enums.UavOrderStatus;
import com.mmc.oms.feign.UserAppApi;
import com.mmc.oms.model.dto.ContractInfoDTO;
import com.mmc.oms.model.dto.uav.UavOrderPayDTO;
import com.mmc.oms.model.dto.uav.UavPurchaseOrderDTO;
import com.mmc.oms.model.dto.user.BaseAccountDTO;
import com.mmc.oms.model.qo.uav.UavPOrderQO;
import com.mmc.oms.model.vo.uav.UavOrderExpressVO;
import com.mmc.oms.model.vo.uav.UavOrderPayVO;
import com.mmc.oms.model.vo.uav.UavPOConfirmVO;
import com.mmc.oms.service.uav.UavOrderService;
import com.mmc.oms.service.uav.UavPOService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author: zj
 * @Date: 2023/9/9 10:03
 */
@Service
@Slf4j
public class UavPOServiceImpl implements UavPOService {

    @Autowired
    private UavPODao uavPODao;

    @Autowired
    private UserAppApi userAppApi;

    @Autowired
    private UavOrderExpressDao uavOrderExpressDao;

    @Autowired
    private UavOrderService uavOrderService;

    @Override
    public ResultBody confirmPOrder(UavPOConfirmVO uavPOConfirmVO, BaseAccountDTO currentAccount) {
        UavPurchaseOrderDO uavPOrder = uavPODao.getUavPOrder(uavPOConfirmVO.getId());
        if (!uavPOrder.getStatusCode().equals(UavOrderStatus.CONFIRM.getCode())) {
            return ResultBody.error("操作有误，请重试！");
        }
        // 合同签署信息，确认平台要先完成签署
        List<String> orderNos = Arrays.asList(uavPOrder.getOrderNo());
        List<ContractInfoDTO> contractInfoDTOS = userAppApi.listContractInfoByOrderNo(orderNos, currentAccount.getToken());
        if (CollectionUtils.isNotEmpty(contractInfoDTOS) && !contractInfoDTOS.get(0).getSignStatus().equals(2)) {
            // 确认平台要先完成签署
            return ResultBody.error("合同未签署");
        }
        // 修改订单状态
        this.updateUavPOrderStatus(uavPOConfirmVO.getId(), UavOrderStatus.CONFIRM);
        // 修改订单信息
        uavPOrder.setUserRemark(uavPOConfirmVO.getUserRemark());
        uavPOrder.setOrderAmount(uavPOConfirmVO.getOrderAmount());
        uavPODao.updateUavPOrderDO(uavPOrder);
        return ResultBody.success();
    }

    /**
     * 修改采购订单状态
     * @param id
     * @param uavOrderStatus
     */
    @Transactional
    public void updateUavPOrderStatus(Integer id, UavOrderStatus uavOrderStatus){
        Integer nextCode = null;
        if (uavOrderStatus.getNextCode().equals(UavOrderStatus.PAYING.getCode())) {
            nextCode = 420;
        }else {
            nextCode = uavOrderStatus.getNextCode();
        }
        uavPODao.updateUavPOrderStatus(id, nextCode);
    }

    @Override
    public ResultBody listPurchase(UavPOrderQO uavPOrderQO) {
        int count = uavPODao.countListPurchaseOrder(uavPOrderQO);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(uavPOrderQO.getPageNo(), uavPOrderQO.getPageSize(), count));
        }
        Integer pageNo = uavPOrderQO.getPageNo();
        uavPOrderQO.buildCurrentPage();
        List<UavPurchaseOrderDO> uavPurchaseOrderDOS = uavPODao.listPurchaseOrder(uavPOrderQO);
        List<UavPurchaseOrderDTO> purchaseOrderDTOS = uavPurchaseOrderDOS.stream().map(UavPurchaseOrderDO::buildUavPurchaseOrderDTO).collect(Collectors.toList());
        // TODO 买家和买家信息
        return ResultBody.success(PageResult.buildPage(pageNo, uavPOrderQO.getPageSize(), count, purchaseOrderDTOS));
    }

    @Override
    public ResultBody getPurchaseOrder(Integer id, BaseAccountDTO currentAccount) {
        UavPurchaseOrderDO uavPOrder = uavPODao.getUavPOrder(id);
        if(uavPOrder == null) {
            return ResultBody.error("订单不存在！");
        }
        UavPurchaseOrderDTO uavPurchaseOrderDTO = uavPOrder.buildUavPurchaseOrderDTO();
        // 收货信息
        UavOrderExpressDO uavOrderExpressDO = uavOrderExpressDao.getUavOrderExpressDO(uavPOrder.getUavOrderId());
        uavPurchaseOrderDTO.setUavOrderExpressDTO(uavOrderExpressDO == null ? null : uavOrderExpressDO.buildUavOrderExpressDTO());
        // 付款凭证信息
        List<UavPurchaseOrderPayDO> purchaseOrderPayDOS = uavPODao.listUavPOrderPay(id);
        List<UavOrderPayDTO> uavOrderPayDTOS = purchaseOrderPayDOS.stream().map(UavPurchaseOrderPayDO::buildUavOrderPayDTO).collect(Collectors.toList());
        uavPurchaseOrderDTO.setUavOrderPayDTOS(uavOrderPayDTOS);
        // 合同签署信息
        List<String> orderNos = Arrays.asList(uavPOrder.getOrderNo());
        List<ContractInfoDTO> contractInfoDTOS = userAppApi.listContractInfoByOrderNo(orderNos, currentAccount.getToken());
        if (CollectionUtils.isNotEmpty(contractInfoDTOS)) {
            uavPurchaseOrderDTO.setContractInfoDTO(contractInfoDTOS.get(0));
        }
        return ResultBody.success(uavPurchaseOrderDTO);
    }

    @Override
    public ResultBody upLoadPay(UavOrderPayVO uavOrderPayVO) {
        UavPurchaseOrderDO uavPOrder = uavPODao.getUavPOrder(uavOrderPayVO.getUavOrderId());
        if (!uavPOrder.getStatusCode().equals(UavOrderStatus.PAYING.getCode())) {
            return ResultBody.error("操作有误，请重试！");
        }
        UavPurchaseOrderPayDO uavOrderPayDO = new UavPurchaseOrderPayDO(uavOrderPayVO);
        uavOrderPayDO.setUavPOrderId(uavOrderPayVO.getUavOrderId());
        uavPODao.addPurchaseOrderPay(uavOrderPayDO);
        // 修改订单状态
        this.updateUavPOrderStatus(uavPOrder.getId(), UavOrderStatus.PAYING);
        return ResultBody.success();
    }

    @Transactional
    @Override
    public ResultBody checkPay(UavOrderPayVO uavOrderPayVO) {
        UavPurchaseOrderDO uavPOrder = uavPODao.getUavPOrder(uavOrderPayVO.getUavOrderId());
        if (!uavPOrder.getStatusCode().equals(UavOrderStatus.FINISH_PAYING.getCode())) {
            return ResultBody.error("操作有误，请重试！");
        }
        UavPurchaseOrderPayDO uavOrderPayDO = new UavPurchaseOrderPayDO(uavOrderPayVO);
        uavPODao.updateUavPOrderPay(uavOrderPayDO);
        if (uavOrderPayVO.getCheckStatus().equals(1)) {
            // 修改订单状态
            this.updateUavPOrderStatus(uavPOrder.getId(), UavOrderStatus.FINISH_PAYING);
        }else {
            this.updateUavPOrderStatus(uavPOrder.getId(), UavOrderStatus.PAYING);
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody send(UavOrderExpressVO param) {
        UavPurchaseOrderDO uavPOrder = uavPODao.getUavPOrderByUavOId(param.getUavOrderId());
        if (!uavPOrder.getStatusCode().equals(UavOrderStatus.WAITING_DELIVER_GOODS.getCode())) {
            return ResultBody.error("操作有误，请重试！");
        }
        // 修改主订单信息
        uavOrderService.send(param);
        // 修改子订单信息
        this.updateUavPOrderStatus(uavPOrder.getId(), UavOrderStatus.WAITING_DELIVER_GOODS);
        return ResultBody.success();
    }

    @Override
    public ResultBody uavPOFinishSign(String orderNo) {
        UavPurchaseOrderDO uavPOrder = uavPODao.getUavPOrderByNo(orderNo);
        if (uavPOrder == null) {
            return ResultBody.error(ResultEnum.NOT_FOUND);
        }
        if (!uavPOrder.getStatusCode().toString().equals(UavOrderStatus.SIGN.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        this.updateUavPOrderStatus(uavPOrder.getId(), UavOrderStatus.SIGN);
        return ResultBody.success();
    }
}
