package com.mmc.oms.service.lease.impl;

import com.alibaba.fastjson2.JSONObject;
import com.mmc.oms.common.ResultEnum;
import com.mmc.oms.common.constant.UavOrderProportion;
import com.mmc.oms.common.json.JsonUtil;
import com.mmc.oms.common.result.PageResult;
import com.mmc.oms.common.result.ResultBody;
import com.mmc.oms.common.util.CodeUtil;
import com.mmc.oms.common.util.KdnExpressUtil;
import com.mmc.oms.dao.UserAddressDao;
import com.mmc.oms.dao.lease.LeaseOrderDao;
import com.mmc.oms.entity.lease.LeaseOrderDO;
import com.mmc.oms.entity.lease.LeaseOrderExpressDO;
import com.mmc.oms.entity.lease.LeaseRefundOrderDO;
import com.mmc.oms.entity.mall.UserAddressDO;
import com.mmc.oms.entity.transtatus.TranStatusDicDO;
import com.mmc.oms.entity.uav.UavOrderRemarkDO;
import com.mmc.oms.enums.LeaseOrderStatus;
import com.mmc.oms.enums.LeaseTermInfo;
import com.mmc.oms.enums.OrderType;
import com.mmc.oms.feign.PaymentAppApi;
import com.mmc.oms.feign.PmsAppApi;
import com.mmc.oms.feign.UserAppApi;
import com.mmc.oms.model.dto.kdn.KdnExpDTO;
import com.mmc.oms.model.dto.lease.LeaseOrderDTO;
import com.mmc.oms.model.dto.lease.LeaseRefundOrderDTO;
import com.mmc.oms.model.dto.order.ExpStationsDTO;
import com.mmc.oms.model.dto.transtatus.TranStatusDicDTO;
import com.mmc.oms.model.dto.uav.PayWalletDTO;
import com.mmc.oms.model.dto.user.BaseAccountDTO;
import com.mmc.oms.model.dto.user.UserAccountSimpleDTO;
import com.mmc.oms.model.qo.UserAccountQO;
import com.mmc.oms.model.qo.uav.UavOrderQO;
import com.mmc.oms.model.vo.ApplyRefundVO;
import com.mmc.oms.model.vo.demand.CommonPaymentVO;
import com.mmc.oms.model.vo.lease.AddLeaseOrderVO;
import com.mmc.oms.model.vo.lease.LeaseGoodsVO;
import com.mmc.oms.model.vo.lease.LeaseOrderRefundVO;
import com.mmc.oms.model.vo.lease.LeasePriceStockVO;
import com.mmc.oms.model.vo.uav.GoodsResourcesVO;
import com.mmc.oms.model.vo.uav.PayUavWalletVO;
import com.mmc.oms.model.vo.uav.UavOrderExpressVO;
import com.mmc.oms.model.vo.uav.UavOrderRemarkVO;
import com.mmc.oms.service.demand.DemandReleaseOrderService;
import com.mmc.oms.service.lease.LeaseOrderService;
import com.mmc.oms.service.uav.UavOrderService;
import com.mmc.oms.util.TDateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Author LW
 * @date 2023/9/21 16:46
 * 概要：
 */
@Service
@Slf4j
public class LeaseOrderServiceImpl implements LeaseOrderService {
    @Resource
    private PmsAppApi pmsAppApi;
    @Resource
    private UavOrderService uavOrderService;
    @Resource
    private LeaseOrderDao leaseOrderDao;
    @Resource
    private UserAddressDao userAddressDao;
    @Resource
    private UserAppApi userAppApi;
    @Resource
    private PaymentAppApi paymentAppApi;
    @Resource
    DemandReleaseOrderService demandReleaseOrderService;

    @Override
    @Transactional
    public ResultBody addLeaseOrder(AddLeaseOrderVO param, BaseAccountDTO currentAccount) {
        log.info("user:{},addOrder param:{}", currentAccount.getUserAccountId(), param);
        if (StringUtils.isEmpty(param.getSku())) {
            return ResultBody.error("至少选择一个规格！");
        }
        if (param.getWareNum() <= 0) {
            return ResultBody.error("购买数量不得少于1");
        }
        LeaseGoodsVO leaseGoodsVO;
        LeasePriceStockVO priceStock = null;
        try {
            leaseGoodsVO = pmsAppApi.leaseGoodsDetails(param.getLeaseGoodsId()).getResult();
            // 检验商品是否在卖
            if (leaseGoodsVO == null || !leaseGoodsVO.getShelfStatus().equals(1)) {
                return ResultBody.error("商品已下架");
            }
            // 判断租赁时间是否超出当前商品的最长租期或小于最短租期
            int distanceDays = TDateUtil.distanceDays(param.getStartDate(), param.getEndDate());
            param.setLeaseTerm(distanceDays);
            // 获取租赁商品的最低租期
            Integer minLeaseTerm = getLeaseTerm(leaseGoodsVO.getMinLeaseTerm());
            // 获取租赁商品的最大租期
            Integer maxLeaseTerm = getLeaseTerm(leaseGoodsVO.getMaxLeaseTerm());
            if (distanceDays < minLeaseTerm || distanceDays > maxLeaseTerm) {
                return ResultBody.error("商品信息发生变更，刷新后重试!");
            }
            // 判断该租期在那个价格区间
            Integer priceSection = getPriceSection(distanceDays);
            // 找到对应的规格
            boolean flag = false;
            for (LeasePriceStockVO priceStockVO : leaseGoodsVO.getPriceStock()) {
                if (param.getSku().equals(priceStockVO.getProductSpec())) {
                    priceStock = priceStockVO;
                    // 获取最新的规格单价
                    param.setUnitPrice(getUnitPrice(priceSection, priceStockVO));
                    param.setDeposit(priceStockVO.getCashPledge());
                    flag = true;
                    break;
                }
            }
            if (!flag) {
                return ResultBody.error("选择规格不存在");
            }
        } catch (Exception e) {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            return ResultBody.error(e.getMessage());
        }

        String orderNo = CodeUtil.generateOrderId();
        // 计算订单总金额
        BigDecimal orderTotalAmount = param.getUnitPrice().multiply(BigDecimal.valueOf(param.getLeaseTerm()))
                .multiply(BigDecimal.valueOf(param.getWareNum())).add(param.getDeposit().multiply(BigDecimal.valueOf(param.getWareNum())));
        BigDecimal otherAmount = BigDecimal.ZERO;
        // 用来进行操作的总金额
        otherAmount = otherAmount.add(orderTotalAmount);
        // 总押金金额
        BigDecimal totalDeposit = param.getDeposit().multiply(BigDecimal.valueOf(param.getWareNum()));
        // 用来操作的总押金
        BigDecimal otherCashPledge = BigDecimal.ZERO;
        otherCashPledge = otherCashPledge.add(totalDeposit);
        // 云享金押金金额,余额押金金额
        BigDecimal shareCashPledge = BigDecimal.ZERO;
        BigDecimal salaryCashPledge = BigDecimal.ZERO;
        // 校验云享金、钱包余额是否足够抵扣
        BigDecimal shareAmount = BigDecimal.ZERO;
        BigDecimal salaryAmount = BigDecimal.ZERO;
        // 其他抵扣押金
        BigDecimal otherPledge = BigDecimal.ZERO;
        Integer statusCode = null;
        // 获取用户钱包信息
        if (param.getDeductSalaryAmount().equals(1) || param.getDeductShareAmount().equals(1)) {
            ResultBody<PayWalletDTO> payResInfo = userAppApi.getCurrentUserPayWalletInfo(currentAccount.getToken());
            if (!payResInfo.getCode().equals(ResultEnum.SUCCESS.getResultCode()) || payResInfo.getResult() == null) {
                return payResInfo;
            }
            PayWalletDTO payWalletDTO = payResInfo.getResult();
            // 先使用云享金来扣减押金
            if (param.getDeductShareAmount().equals(1)) {
                if (otherCashPledge.compareTo(payWalletDTO.getCashAmt()) >= 0) {
                    // 不够抵扣押金
                    shareCashPledge = payWalletDTO.getCashAmt();
                } else {
                    // 够抵扣押金
                    shareCashPledge = otherCashPledge;
                }
            }
            otherCashPledge = otherCashPledge.subtract(shareCashPledge);
            if (otherCashPledge.compareTo(BigDecimal.ZERO) == 0) {
                otherAmount = otherAmount.subtract(shareCashPledge);
            } else {
                if (param.getDeductSalaryAmount().equals(1)) {
                    if (otherCashPledge.compareTo(payWalletDTO.getSalaryAmt()) >= 0) {
                        salaryCashPledge = payWalletDTO.getSalaryAmt();
                    } else {
                        salaryCashPledge = otherCashPledge;
                    }
                    otherCashPledge = otherCashPledge.subtract(salaryCashPledge);
                    if (otherCashPledge.compareTo(BigDecimal.ZERO) == 0) {
                        otherAmount = otherAmount.subtract(salaryCashPledge.add(shareCashPledge));
                    } else {
                        otherPledge = otherCashPledge;
                    }
                }
            }
            // 抵扣完后当押金为0的时候 继续抵扣其余金额
            if (otherCashPledge.compareTo(BigDecimal.ZERO) == 0) {
                if (param.getDeductShareAmount().equals(1)) {
                    if (otherAmount.compareTo(payWalletDTO.getCashAmt().subtract(shareCashPledge)) >= 0) {
                        // 抵扣云享金小于等于订单总额时，直接使用云享金
                        shareAmount = payWalletDTO.getCashAmt().subtract(shareCashPledge);
                    } else {
                        // 余额超过订单金额时，订单总额使用余额支付
                        shareAmount = otherAmount;
                    }
                }
                otherAmount = otherAmount.subtract(shareAmount);
                // 支付完成
                if (otherAmount.compareTo(BigDecimal.ZERO) == 0) {
                    statusCode = LeaseOrderStatus.PAID.getCode();
                } else {
                    if (param.getDeductSalaryAmount().equals(1)) {
                        if (otherAmount.compareTo(payWalletDTO.getSalaryAmt().subtract(salaryCashPledge)) >= 0) {
                            // 抵扣余额小于等于订单总额时，直接使用余额
                            salaryAmount = payWalletDTO.getSalaryAmt().subtract(salaryCashPledge);
                        } else {
                            // 余额超过订单金额时，订单总额使用余额支付
                            salaryAmount = otherAmount;
                        }
                    }
                    otherAmount = otherAmount.subtract(salaryAmount);
                    if (otherAmount.compareTo(BigDecimal.ZERO) == 0) {
                        statusCode = LeaseOrderStatus.PAID.getCode();
                    }
                }
            }
            // 抵扣余额
            uavOrderService.deductWallet(shareAmount.add(shareCashPledge), salaryAmount.add(salaryCashPledge), currentAccount);
        }
        // 获取商品主图
        String url = null;
        for (GoodsResourcesVO goodsResourcesVO : leaseGoodsVO.getResourcesList()) {
            if (goodsResourcesVO.getType().equals(0)) {
                url = goodsResourcesVO.getUrl();
                break;
            }
        }
        // 添加总订单表
        LeaseOrderDO leaseOrderDO = new LeaseOrderDO();
        leaseOrderDO.setOrderNo(orderNo);
        leaseOrderDO.setOrderTotalAmount(orderTotalAmount);
        leaseOrderDO.setStatusCode(otherAmount.compareTo(BigDecimal.ZERO) > 0 ? LeaseOrderStatus.PLACED_AN_ORDER.getCode() : statusCode);
        leaseOrderDO.setUserAccountId(currentAccount.getUserAccountId());
        leaseOrderDO.setLeaseGoodsId(param.getLeaseGoodsId());
        leaseOrderDO.setCompanyName(param.getCompanyName());
        leaseOrderDO.setThirdBackUserAccountId(param.getThirdBackUserAccountId());
        leaseOrderDO.setOrderName(leaseGoodsVO.getTradeName());
        leaseOrderDO.setOtherPledge(otherPledge);
        leaseOrderDO.setModeOfDeliveryInfo(leaseGoodsVO.getModeOfDeliveryInfo());
        if (priceStock.getSkuImage() != null) {
            leaseOrderDO.setSkuImg(priceStock.getSkuImage());
        } else {
            leaseOrderDO.setSkuImg(url);
        }
        leaseOrderDO.setSalaryAmount(salaryAmount);
        leaseOrderDO.setSku(param.getSku());
        leaseOrderDO.setSalaryCashPledge(salaryCashPledge);
        leaseOrderDO.setShareCashPledge(shareCashPledge);
        leaseOrderDO.setShareAmount(shareAmount);
        leaseOrderDO.setOtherAmount(otherAmount);
        leaseOrderDO.setUserAddressId(param.getUserAddressId());
        leaseOrderDO.setUserRemark(param.getUserRemark());
        leaseOrderDO.setUnitPrice(param.getUnitPrice());
        leaseOrderDO.setWareNum(param.getWareNum());
        leaseOrderDO.setDeposit(param.getDeposit());
        leaseOrderDO.setStartDate(param.getStartDate());
        if (null != statusCode && statusCode.equals(LeaseOrderStatus.PAID.getCode())) {
            leaseOrderDO.setPayTime(new Date());
        }
        leaseOrderDO.setEndDate(param.getEndDate());
        leaseOrderDao.addOrder(leaseOrderDO);
        // 添加收货地址
        UserAddressDO userAddressDO = userAddressDao.selectById(param.getUserAddressId());
        LeaseOrderExpressDO leaseOrderExpressDO = new LeaseOrderExpressDO(userAddressDO);
        leaseOrderExpressDO.setLeaseOrderId(leaseOrderDO.getId());
        leaseOrderDao.addAddress(leaseOrderExpressDO);
        LeaseOrderDTO leaseOrderDTO = leaseOrderDO.buildLeaseOrderDTO();
        leaseOrderDTO.setModeOfDeliveryInfo(leaseGoodsVO.getModeOfDeliveryInfo());
        return ResultBody.success(leaseOrderDTO);
    }

    private Integer getLeaseTerm(Integer leaseTermId) {
        if (leaseTermId.equals(LeaseTermInfo.THREE_DAY.getId())) {
            return LeaseTermInfo.THREE_DAY.getDays();
        } else if (leaseTermId.equals(LeaseTermInfo.SEVEN_DAY.getId())) {
            return LeaseTermInfo.SEVEN_DAY.getDays();
        } else if (leaseTermId.equals(LeaseTermInfo.THIRTY_DAY.getId())) {
            return LeaseTermInfo.THIRTY_DAY.getDays();
        } else if (leaseTermId.equals(LeaseTermInfo.NINETY_DAY.getId())) {
            return LeaseTermInfo.NINETY_DAY.getDays();
        } else {
            return LeaseTermInfo.ONE_HUNDRED_AND_EIGHTY_DAY.getDays();
        }
    }


    private BigDecimal getUnitPrice(Integer priceSection, LeasePriceStockVO priceStockVO) {
        if (priceSection.equals(LeaseTermInfo.THREE_DAY.getDays())) {
            return priceStockVO.getThreeDaysRental();
        } else if (priceSection.equals(LeaseTermInfo.SEVEN_DAY.getDays())) {
            return priceStockVO.getSevenDaysRental();
        } else if (priceSection.equals(LeaseTermInfo.THIRTY_DAY.getDays())) {
            return priceStockVO.getThirtyDaysRental();
        } else if (priceSection.equals(LeaseTermInfo.NINETY_DAY.getDays())) {
            return priceStockVO.getNinetyDaysRental();
        } else {
            return priceStockVO.getMaxDaysRental();
        }
    }

    private Integer getPriceSection(int distanceDays) {
        if (distanceDays >= LeaseTermInfo.THREE_DAY.getDays() && distanceDays < LeaseTermInfo.SEVEN_DAY.getDays()) {
            return LeaseTermInfo.THREE_DAY.getDays();
        } else if (distanceDays >= LeaseTermInfo.SEVEN_DAY.getDays() && distanceDays < LeaseTermInfo.THIRTY_DAY.getDays()) {
            return LeaseTermInfo.SEVEN_DAY.getDays();
        } else if (distanceDays >= LeaseTermInfo.THIRTY_DAY.getDays() && distanceDays < LeaseTermInfo.NINETY_DAY.getDays()) {
            return LeaseTermInfo.THIRTY_DAY.getDays();
        } else if (distanceDays >= LeaseTermInfo.NINETY_DAY.getDays() && distanceDays < LeaseTermInfo.ONE_HUNDRED_AND_EIGHTY_DAY.getDays()) {
            return LeaseTermInfo.NINETY_DAY.getDays();
        } else {
            return LeaseTermInfo.ONE_HUNDRED_AND_EIGHTY_DAY.getDays();
        }
    }

    @Override
    public ResultBody detail(Integer id, BaseAccountDTO currentAccount) throws Exception {
        LeaseOrderDO leaseOrderDO = leaseOrderDao.detail(id);
        if (leaseOrderDO == null) {
            return ResultBody.error("订单信息不存在！");
        }
        LeaseOrderDTO leaseOrderDTO = leaseOrderDO.buildLeaseOrderDTO();
        Integer statusCode = leaseOrderDO.getStatusCode();
        if (statusCode.equals(LeaseOrderStatus.PLACED_AN_ORDER.getCode())) {
            leaseOrderDTO.setNeedPayAmount(leaseOrderDO.getOtherAmount());
        }
        // 收货信息、付款信息、规格信息、快递信息、签署信息
        LeaseOrderExpressDO leaseOrderExpressDO = leaseOrderDao.getLeaseOrderExpress(id);
        leaseOrderDTO.setUavOrderExpressDTO(leaseOrderExpressDO.buildUavOrderExpressDTO());
        // 快递信息
        KdnExpDTO kdnExpDTO = this.getOrderExpInfo(leaseOrderExpressDO);
        leaseOrderDTO.setKdnExpDTO(kdnExpDTO);
        // 判断订单状态是否是已归还状态
        if (statusCode.compareTo(LeaseOrderStatus.WAITING_MERCHANT_CONFIRM.getCode()) >= 0) {
            // 快递信息
            KdnExpDTO returnKdnExpDTO = this.getReturnOrderExpInfo(leaseOrderExpressDO);
            leaseOrderDTO.setReturnKdnExpDTO(returnKdnExpDTO);
        }
        // 设置确认收货信息
        setConfirmReceiptInfo(id, OrderType.REC.getCode(), leaseOrderDTO);
        // 设置退款信息
        setConfirmReceiptInfo(id, OrderType.RETREAT.getCode(), leaseOrderDTO);
        // 设置破损信息
        setConfirmReceiptInfo(id, OrderType.PFREC.getCode(), leaseOrderDTO);
        // 设置个人信息
        UserAccountSimpleDTO userAccountSimpleDTO = userAppApi.feignGetUserSimpleInfo(leaseOrderDO.getUserAccountId(), currentAccount.getToken());
        if (userAccountSimpleDTO != null) {
            leaseOrderDTO.setNickname(userAccountSimpleDTO.getNickName());
            leaseOrderDTO.setPhoneNum(userAccountSimpleDTO.getPhoneNum());
            leaseOrderDTO.setUserName(userAccountSimpleDTO.getUserName());
        }
        return ResultBody.success(leaseOrderDTO);
    }

    private void setConfirmReceiptInfo(Integer id, Integer orderType, LeaseOrderDTO leaseOrderDTO) {
        LeaseRefundOrderDO leaseRefundOrderDO = leaseOrderDao.selectGiveBackByOrderId(id, orderType);
        switch (orderType) {
            case 1:
                leaseOrderDTO.setConfirmReceipt(leaseRefundOrderDO == null ? null : leaseRefundOrderDO.buildLeaseRefundOrderDTO());
                break;
            case 4:
                leaseOrderDTO.setLeaseRefundOrder(leaseRefundOrderDO == null ? null : leaseRefundOrderDO.buildLeaseRefundOrderDTO());
                break;
            case 3:
                leaseOrderDTO.setStoreDamageOrder(leaseRefundOrderDO == null ? null : leaseRefundOrderDO.buildLeaseRefundOrderDTO());
                break;
            default:
                break;
        }
    }


    private KdnExpDTO getOrderExpInfo(LeaseOrderExpressDO exp) throws Exception {
        String json = "";
        if ("SF".equals(exp.getSendExpCode())) {
            json = KdnExpressUtil.SF(exp.getTakePhone(), exp.getSendExpNo());
        } else if ("JD".equals(exp.getSendExpCode())) {
            json = KdnExpressUtil.JD("", exp.getSendExpNo());
        } else {
            json = KdnExpressUtil.others(exp.getSendExpCode(), exp.getSendExpNo());
        }
        KdnExpDTO kdn = JsonUtil.parseJsonToObj(json, KdnExpDTO.class);
        kdn.setLogisticCode(exp.getSendExpNo());
        if (!org.springframework.util.CollectionUtils.isEmpty(kdn.getTraces())) {
            Collections.sort(kdn.getTraces(), Comparator.comparing(ExpStationsDTO::getAcceptTime).reversed());// 升序
        }
        return kdn;
    }

    private KdnExpDTO getReturnOrderExpInfo(LeaseOrderExpressDO exp) throws Exception {
        String json = "";
        if ("SF".equals(exp.getReturnExpCode())) {
            json = KdnExpressUtil.SF("", exp.getReturnExpNo());
        } else if ("JD".equals(exp.getReturnExpCode())) {
            json = KdnExpressUtil.JD("", exp.getReturnExpNo());
        } else {
            json = KdnExpressUtil.others(exp.getReturnExpCode(), exp.getReturnExpNo());
        }
        KdnExpDTO kdn = JsonUtil.parseJsonToObj(json, KdnExpDTO.class);
        kdn.setLogisticCode(exp.getReturnExpNo());
        if (!org.springframework.util.CollectionUtils.isEmpty(kdn.getTraces())) {
            Collections.sort(kdn.getTraces(), Comparator.comparing(ExpStationsDTO::getAcceptTime).reversed());// 升序
        }
        return kdn;
    }

    @Override
    @Transactional
    public ResultBody close(LeaseOrderRefundVO leaseOrderRefundVO, String token) {
        LeaseOrderDO leaseOrderDO = leaseOrderDao.detail(leaseOrderRefundVO.getOrderId());
        if (leaseOrderDO == null) {
            return ResultBody.error("订单信息不存在！");
        }
        // 判断订单状态
        if (leaseOrderDO.getStatusCode().compareTo(LeaseOrderStatus.PAID.getCode()) > 0) {
            if (leaseOrderDO.getStatusCode().compareTo(LeaseOrderStatus.TO_BE_RETURNED.getCode()) == 0) {
                return ResultBody.error("当前状态不可申请退款");
            }
            // 判断商品是否需要归还
            if (leaseOrderRefundVO.getStatus().equals(0)) {
                if (leaseOrderRefundVO.getExpressCode() == null || leaseOrderRefundVO.getExpressNum() == null) {
                    return ResultBody.error("请完善物流信息");
                }
                // 插入退货地址信息
                LeaseOrderExpressDO leaseOrderExpressDO = new LeaseOrderExpressDO();
                leaseOrderExpressDO.setId(leaseOrderRefundVO.getId());
                leaseOrderExpressDO.setLeaseOrderId(leaseOrderRefundVO.getOrderId());
                leaseOrderExpressDO.setReturnExpCode(leaseOrderRefundVO.getExpressCode());
                leaseOrderExpressDO.setReturnExpNo(leaseOrderRefundVO.getExpressNum());
                leaseOrderExpressDO.setReturnTime(new Date());
                leaseOrderDao.updateLeaseOrderExpress(leaseOrderExpressDO);
                // 插入退款订单信息
                LeaseRefundOrderDO leaseRefundOrderDO = new LeaseRefundOrderDO(leaseOrderRefundVO);
                leaseRefundOrderDO.setType(OrderType.RETREAT.getCode());
                leaseOrderDao.addLeaseRefundOrder(leaseRefundOrderDO);
                // 修改订单状态
                leaseOrderDao.updateLeaseOrderStatus(leaseOrderRefundVO.getOrderId(), LeaseOrderStatus.APPLY_REFUND.getCode());
            }
            return ResultBody.success();
        } else {
            // 退云享金、佣金
            if (leaseOrderDO.getShareAmount().compareTo(BigDecimal.ZERO) > 0 || leaseOrderDO.getSalaryAmount().compareTo(BigDecimal.ZERO) > 0 ||
                    leaseOrderDO.getSalaryCashPledge().compareTo(BigDecimal.ZERO) > 0 || leaseOrderDO.getShareCashPledge().compareTo(BigDecimal.ZERO) > 0) {
                PayUavWalletVO userPayUavWalletVO = amountUnfreezing(leaseOrderDO, leaseOrderDO.getShareAmount(), leaseOrderDO.getSalaryAmount());
                userPayUavWalletVO.setOrderStatus(1400);
                ResultBody resultBody = userAppApi.feignPayUavWallet(userPayUavWalletVO, token);
                if (!resultBody.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                    return resultBody;
                }
            }
            // 微信退款
            if (leaseOrderDO.getOtherPledge() != null && leaseOrderDO.getOtherPledge().compareTo(BigDecimal.ZERO) > 0) {
                ApplyRefundVO applyRefundVO = new ApplyRefundVO();
                applyRefundVO.setOutTradeNo(leaseOrderDO.getOrderNo());
                applyRefundVO.setReason("关闭租赁订单");
                applyRefundVO.setRefund(leaseOrderDO.getOtherAmount().multiply(BigDecimal.valueOf(100)).longValue());
                ResultBody applyRefund = paymentAppApi.applyRefund(applyRefundVO, token);
                log.info("applyRefund res：{}", applyRefund);
                if (!applyRefund.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                    TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
                    return ResultBody.error("关闭订单失败！");
                }
            }
            // 修改订单状态
            this.updateLeaseOrderStatus(leaseOrderRefundVO.getOrderId(), LeaseOrderStatus.CLOSE);
            return ResultBody.success();
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateLeaseOrderStatus(Integer id, LeaseOrderStatus leaseOrderStatus) {
        leaseOrderDao.updateLeaseOrderStatus(id, leaseOrderStatus.getCode());
    }

    @Override
    public ResultBody<LeaseOrderDTO> list(UavOrderQO uavOrderQO, BaseAccountDTO currentAccount) {
        if (!currentAccount.getRoleInfo().getSuperAdmin().equals(1)) {
            uavOrderQO.setThirdBackUserAccountId(currentAccount.getUserAccountId());
        }
        int count = leaseOrderDao.countList(uavOrderQO);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(uavOrderQO.getPageNo(), uavOrderQO.getPageSize(), count));
        }
        Integer pageNo = uavOrderQO.getPageNo();
        uavOrderQO.buildCurrentPage();
        List<LeaseOrderDO> leaseOrderDOS = leaseOrderDao.list(uavOrderQO);

        Map<Integer, LeaseOrderDTO> leaseOrderDTOMap = new LinkedHashMap<>();
        List<Integer> ids = new ArrayList<>();
        List<Integer> userIds = new ArrayList<>();

        for (LeaseOrderDO leaseOrderDO : leaseOrderDOS) {
            LeaseOrderDTO leaseOrderDTO = leaseOrderDO.buildLeaseOrderDTO();
            leaseOrderDTOMap.put(leaseOrderDO.getId(), leaseOrderDTO);
            if (leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.APPLY_REFUND.getCode()) || leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.WAITING_MERCHANT_CONFIRM.getCode())) {
                ids.add(leaseOrderDO.getId());
            }

            userIds.add(leaseOrderDTO.getUserAccountId());
        }

        buildRefundOrder(leaseOrderDTOMap, ids);

        UserAccountQO userAccountQO = new UserAccountQO();
        userAccountQO.setUserIds(userIds);
        List<UserAccountSimpleDTO> userDTOS = userAppApi.feignListAppUserAccount(userAccountQO, currentAccount.getToken());

        if (CollectionUtils.isNotEmpty(userDTOS)) {
            Map<Integer, UserAccountSimpleDTO> userDTOMap = userDTOS.stream().collect(Collectors.toMap(UserAccountSimpleDTO::getId, user -> user, (k1, k2) -> k2));
            for (LeaseOrderDTO leaseOrderDTO : leaseOrderDTOMap.values()) {
                UserAccountSimpleDTO user = userDTOMap.get(leaseOrderDTO.getUserAccountId());
                if (user != null) {
                    leaseOrderDTO.setPhoneNum(user.getPhoneNum());
                    leaseOrderDTO.setNickname(user.getNickName());
                    leaseOrderDTO.setUserName(user.getUserName());
                }
            }
        }

        return ResultBody.success(PageResult.buildPage(pageNo, uavOrderQO.getPageSize(), count, new ArrayList<>(leaseOrderDTOMap.values())));
    }


    @Override
    public ResultBody<LeaseOrderDTO> listByUid(UavOrderQO uavOrderQO, BaseAccountDTO currentAccount) {
        uavOrderQO.setUserAccountId(currentAccount.getUserAccountId());
        uavOrderQO.setShowUserPort(1);
        int count = leaseOrderDao.countList(uavOrderQO);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(uavOrderQO.getPageNo(), uavOrderQO.getPageSize(), count));
        }
        Integer pageNo = uavOrderQO.getPageNo();
        uavOrderQO.buildCurrentPage();
        List<LeaseOrderDO> leaseOrderDOS = leaseOrderDao.list(uavOrderQO);
        Map<Integer, LeaseOrderDTO> leaseOrderDTOMap = new LinkedHashMap<>();
        List<Integer> ids = new ArrayList<>();
        for (LeaseOrderDO leaseOrderDO : leaseOrderDOS) {
            LeaseOrderDTO leaseOrderDTO = leaseOrderDO.buildLeaseOrderDTO();
            leaseOrderDTOMap.put(leaseOrderDO.getId(), leaseOrderDTO);
            if (leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.APPLY_REFUND.getCode())) {
                ids.add(leaseOrderDO.getId());
            }
        }
        buildRefundOrder(leaseOrderDTOMap, ids);
        return ResultBody.success(PageResult.buildPage(pageNo, uavOrderQO.getPageSize(), count, new ArrayList<>(leaseOrderDTOMap.values())));
    }

    private void buildRefundOrder(Map<Integer, LeaseOrderDTO> leaseOrderDTOMap, List<Integer> ids) {
        if (ids.size() > 0) {
            List<LeaseRefundOrderDO> leaseRefundOrderDO = leaseOrderDao.getLeaseRefundOrder(ids);
            List<Integer> expressIds = leaseRefundOrderDO.stream().map(LeaseRefundOrderDO::getLeaseOrderExpressId).collect(Collectors.toList());
            List<LeaseOrderExpressDO> expressDOS = leaseOrderDao.listLeaseOrderExpress(expressIds);

            Map<Integer, LeaseOrderExpressDO> expressDOMap = expressDOS.stream().collect(Collectors.toMap(LeaseOrderExpressDO::getId, Function.identity()));

            for (LeaseRefundOrderDO refundOrderDO : leaseRefundOrderDO) {
                LeaseRefundOrderDTO refundOrderDTO = refundOrderDO.buildLeaseRefundOrderDTO();
                LeaseOrderExpressDO expressDO = expressDOMap.get(refundOrderDO.getLeaseOrderExpressId());
                if (expressDO != null) {
                    refundOrderDTO.setExpressCode(expressDO.getReturnExpCode());
                    refundOrderDTO.setExpressNum(expressDO.getReturnExpNo());
                }
                LeaseOrderDTO correspondingLeaseOrderDTO = leaseOrderDTOMap.get(refundOrderDTO.getLeaseOrderId());
                if (correspondingLeaseOrderDTO != null) {
                    if (refundOrderDO.getType().equals(OrderType.RETREAT.getCode())) {
                        correspondingLeaseOrderDTO.setLeaseRefundOrder(refundOrderDTO);
                    } else {
                        correspondingLeaseOrderDTO.setStoreReceiveOrder(refundOrderDTO);
                    }
                }
            }
        }
    }


    @Override
    public ResultBody pay(Integer id, BaseAccountDTO currentAccount, HttpServletRequest request) {
        // 发起微信支付
        LeaseOrderDO leaseOrderDO = leaseOrderDao.detail(id);
        if (!leaseOrderDO.getStatusCode().toString().equals(LeaseOrderStatus.PLACED_AN_ORDER.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        CommonPaymentVO commonPaymentVO = new CommonPaymentVO();
        int shouldPay = leaseOrderDO.getOtherAmount().multiply(BigDecimal.valueOf(100)).intValue();
        if (shouldPay <= 0) {
            return ResultBody.success("已支付完成");
        }
        commonPaymentVO.setAmount(shouldPay);
        commonPaymentVO.setDescription(leaseOrderDO.getOrderName());
        commonPaymentVO.setOrderNumber(leaseOrderDO.getOrderNo());
        commonPaymentVO.setOrderPort(1);
        ResultBody resultBody = demandReleaseOrderService.orderPayment(commonPaymentVO, request, currentAccount.getUserAccountId());
        return resultBody;
    }


    @Override
    public ResultBody send(UavOrderExpressVO param) {
        // 修改订单状态
        LeaseOrderDO leaseOrderDO = leaseOrderDao.detail(param.getUavOrderId());
        if (!leaseOrderDO.getStatusCode().toString().equals(LeaseOrderStatus.PAID.getCode().toString())) {
            return ResultBody.error("状态流程有误！");
        }
        this.updateLeaseOrderStatus(param.getUavOrderId(), LeaseOrderStatus.SHIPPED);
        // 修改收货信息
        LeaseOrderExpressDO leaseOrderExpressDO = new LeaseOrderExpressDO(param);
        leaseOrderExpressDO.setSendTime(new Date());
        leaseOrderDao.updateLeaseOrderExpress(leaseOrderExpressDO);
        return ResultBody.success();
    }

    @Override
    public ResultBody receive(LeaseOrderRefundVO leaseOrderRefundVO, BaseAccountDTO currentAccount) {
        LeaseOrderDO leaseOrderDO = leaseOrderDao.detail(leaseOrderRefundVO.getOrderId());
        if (!leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.SHIPPED.getCode())) {
            return ResultBody.error("状态流程有误！");
        }
        if (StringUtils.isNotEmpty(leaseOrderRefundVO.getFile()) || StringUtils.isNotEmpty(leaseOrderRefundVO.getReason())) {
            // 插入确认信息
            LeaseRefundOrderDO leaseRefundOrderDO = new LeaseRefundOrderDO(leaseOrderRefundVO);
            leaseRefundOrderDO.setStatus(1);
            leaseRefundOrderDO.setType(OrderType.REC.getCode());
            leaseOrderDao.addLeaseRefundOrder(leaseRefundOrderDO);
        }
        // 修改收货信息
        LeaseOrderExpressDO leaseOrderExpress = leaseOrderDao.getLeaseOrderExpress(leaseOrderRefundVO.getOrderId());
        leaseOrderExpress.setReceive(1);
        leaseOrderExpress.setReceiveTime(new Date());
        leaseOrderDao.updateLeaseOrderExpress(leaseOrderExpress);
        this.updateLeaseOrderStatus(leaseOrderRefundVO.getOrderId(), LeaseOrderStatus.TO_BE_RETURNED);
        return ResultBody.success();
    }

    @Override
    public ResultBody statusList() {
        List<TranStatusDicDO> statusDOS = leaseOrderDao.statusList();
        List<TranStatusDicDTO> uavOrderStatusDTOS = statusDOS.stream().map(TranStatusDicDO::buildTranStatusDicDTO).collect(Collectors.toList());
        return ResultBody.success(uavOrderStatusDTOS);
    }


    @Override
    public ResultBody removeLeaseOrder(Integer id) {
        LeaseOrderDO leaseOrderDO = leaseOrderDao.detail(id);
        log.info("leaseOrderDO:{}", JSONObject.toJSONString(leaseOrderDO));
        if (leaseOrderDO.getStatusCode().compareTo(LeaseOrderStatus.COMPLETE_TRANSACTION.getCode()) < 0) {
            return ResultBody.error("该订单状态不可删除！");
        }
        leaseOrderDao.closeShowLeaseOrder(id);
        return ResultBody.success();
    }

    @Override
    @Transactional
    public ResultBody giveBack(LeaseOrderRefundVO leaseOrderRefundVO, BaseAccountDTO currentAccount) {
        LeaseOrderDO leaseOrderDO = leaseOrderDao.detail(leaseOrderRefundVO.getOrderId());
        log.info("leaseOrderDO:{}", JSONObject.toJSONString(leaseOrderDO));
        if (!leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.TO_BE_RETURNED.getCode())) {
            return ResultBody.error("状态流程有误！");
        }
        LeaseOrderExpressDO leaseOrderExpressDO = new LeaseOrderExpressDO(leaseOrderRefundVO);
        leaseOrderDao.updateLeaseOrderExpress(leaseOrderExpressDO);
        this.updateLeaseOrderStatus(leaseOrderRefundVO.getOrderId(), LeaseOrderStatus.WAITING_MERCHANT_CONFIRM);
        return ResultBody.success();
    }

    @Override
    @Transactional
    public ResultBody pfConfirmOrderWare(BaseAccountDTO currentAccount, LeaseOrderRefundVO param) {
        LeaseOrderDO leaseOrderDO = leaseOrderDao.detail(param.getOrderId());
        log.info("平台确认归还操作，leaseOrderDO:{}", JSONObject.toJSONString(leaseOrderDO));
        if (leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.REMAIN_TO_BE_EVALUATED.getCode())) {
            return ResultBody.error("请勿重复操作");
        }

        boolean isWaitingMerchantConfirm = leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.WAITING_MERCHANT_CONFIRM.getCode());
        boolean isApplyRefund = leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.APPLY_REFUND.getCode());

        if (isWaitingMerchantConfirm || isApplyRefund) {
            if (param.getDamage().equals(0)) {
                // 解冻用户钱包金额
                BigDecimal shareAmount = leaseOrderDO.getShareAmount();
                BigDecimal salaryAmount = leaseOrderDO.getSalaryAmount();

                ApplyRefundVO applyRefundVO = null;
                if (isApplyRefund && leaseOrderDO.getOtherAmount().compareTo(BigDecimal.ZERO) > 0) {
                    applyRefundVO = wxRefund(leaseOrderDO);
                    applyRefundVO.setRefund((leaseOrderDO.getOtherPledge().add(leaseOrderDO.getOtherAmount())).multiply(BigDecimal.valueOf(100)).longValue());
                } else if (leaseOrderDO.getOtherPledge() != null && leaseOrderDO.getOtherPledge().compareTo(BigDecimal.ZERO) > 0) {
                    applyRefundVO = wxRefund(leaseOrderDO);
                }

                if (applyRefundVO != null) {
                    ResultBody refundOtherPledge = paymentAppApi.applyRefund(applyRefundVO, currentAccount.getToken());
                    if (!refundOtherPledge.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                        return refundOtherPledge;
                    }
                }

                if (shareAmount.compareTo(BigDecimal.ZERO) > 0 || salaryAmount.compareTo(BigDecimal.ZERO) > 0) {
                    // 金额解冻
                    PayUavWalletVO userPayUavWalletVO = amountUnfreezing(leaseOrderDO, shareAmount, salaryAmount);
                    userPayUavWalletVO.setOrderStatus(isWaitingMerchantConfirm ? 999 : 1400);
                    ResultBody resultBody = userAppApi.feignPayUavWallet(userPayUavWalletVO, currentAccount.getToken());
                    if (!resultBody.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                        return resultBody;
                    }
                    // 如果是交易完成则退回押金
                    if (isWaitingMerchantConfirm) {
                        BigDecimal shareCashPledge = leaseOrderDO.getShareCashPledge();
                        BigDecimal salaryCashPledge = leaseOrderDO.getSalaryCashPledge();
                        if ((shareCashPledge.compareTo(BigDecimal.ZERO) > 0 || salaryCashPledge.compareTo(BigDecimal.ZERO) > 0)) {
                            moneyRefund(shareCashPledge, salaryCashPledge, currentAccount, leaseOrderDO);
                        }
                    }
                }

                // 修改退款操作
                if (isApplyRefund) {
                    this.updateLeaseOrderStatus(param.getOrderId(), LeaseOrderStatus.CLOSE);
                }

                if (isWaitingMerchantConfirm) {
                    return businessIncome(currentAccount, param.getOrderId(), leaseOrderDO, 0);
                }
            } else {
                // 有破损等待平台
                if (StringUtils.isEmpty(param.getFile()) || StringUtils.isEmpty(param.getReason())) {
                    return ResultBody.error("请填充破损照片和原因！");
                }
                LeaseRefundOrderDO leaseRefundOrderDO = new LeaseRefundOrderDO(param);
                leaseRefundOrderDO.setStatus(1);
                leaseRefundOrderDO.setType(OrderType.PFREC.getCode());
                leaseOrderDao.addLeaseRefundOrder(leaseRefundOrderDO);
                // 修改订单状态
                if (isWaitingMerchantConfirm) {
                    this.updateLeaseOrderStatus(param.getOrderId(), LeaseOrderStatus.MODIFIED_DEPOSIT_TO_BE_CONFIRMED);
                } else {
                    this.updateLeaseOrderStatus(param.getOrderId(), LeaseOrderStatus.MODIFIED_DEPOSIT_REFUND);
                }
            }
        } else {
            return ResultBody.error("状态流程有误");
        }
        return ResultBody.success();
    }

    private ResultBody businessIncome(BaseAccountDTO currentAccount, Integer orderId, LeaseOrderDO leaseOrderDO, Integer flag) {
        // 拿到商家管理员的用户id
        ResultBody managerRes = userAppApi.getManagerIdByBackUserId(leaseOrderDO.getThirdBackUserAccountId(), null);
        if (!managerRes.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
            return managerRes;
        }
        PayUavWalletVO payUavWalletVO = new PayUavWalletVO();
        payUavWalletVO.setCashAmount(BigDecimal.ZERO);
        // 获取损坏的金额
        LeaseRefundOrderDO leaseRefundOrderDO = leaseOrderDao.selectGiveBackByOrderId(leaseOrderDO.getId(), OrderType.PFREC.getCode());
        payUavWalletVO.setSalaryAmount(leaseRefundOrderDO.getModifiedDeposit());
        payUavWalletVO.setRemark("租赁订单获赔，订单编号" + leaseOrderDO.getOrderNo());
        payUavWalletVO.setOrderStatus(1800);

        // 为0时标识订单收益，否则就是修改状态
        if (flag == 0) {
            handleOrderIncome(currentAccount, leaseOrderDO, managerRes);
            leaseOrderDao.updateLeaseOrderProportion(leaseOrderDO.getId(), UavOrderProportion.PROPORTION.intValue());
            this.updateLeaseOrderStatus(orderId, LeaseOrderStatus.REMAIN_TO_BE_EVALUATED);
        } else {
            // 修改退款操状态
            this.updateLeaseOrderStatus(leaseOrderDO.getId(), LeaseOrderStatus.CLOSE);
        }

        return handlePayUavWallet(currentAccount, payUavWalletVO, managerRes);
    }

    private void handleOrderIncome(BaseAccountDTO currentAccount, LeaseOrderDO leaseOrderDO, ResultBody managerRes) {
        BigDecimal orderTotalAmount = leaseOrderDO.getOrderTotalAmount().subtract(leaseOrderDO.getDeposit());
        BigDecimal thirdAmount = orderTotalAmount.multiply(
                BigDecimal.valueOf(100).subtract(UavOrderProportion.PROPORTION).divide(BigDecimal.valueOf(100)));
        PayUavWalletVO payUavWallet = new PayUavWalletVO();
        payUavWallet.setCashAmount(BigDecimal.ZERO);
        payUavWallet.setSalaryAmount(thirdAmount);
        payUavWallet.setRemark("租赁订单收入，订单编号" + leaseOrderDO.getOrderNo());
        payUavWallet.setOrderStatus(1500);

        handlePayUavWallet(currentAccount, payUavWallet, managerRes);
    }

    private ResultBody handlePayUavWallet(BaseAccountDTO currentAccount, PayUavWalletVO payUavWalletVO, ResultBody managerRes) {
        payUavWalletVO.setUserAccountId((Integer) managerRes.getResult());

        ResultBody resultIncomeBody = userAppApi.feignPayUavWallet(payUavWalletVO, currentAccount.getToken());

        if (!resultIncomeBody.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
            return resultIncomeBody;
        }

        return ResultBody.success();
    }

    private ResultBody moneyRefund(BigDecimal shareCashPledge, BigDecimal salaryCashPledge, BaseAccountDTO currentAccount, LeaseOrderDO leaseOrderDO) {
        PayUavWalletVO refundUavWalletVO = new PayUavWalletVO();
        refundUavWalletVO.setRemark("租赁押金退回");
        refundUavWalletVO.setCashAmount(shareCashPledge);
        refundUavWalletVO.setSalaryAmount(salaryCashPledge);
        refundUavWalletVO.setUserAccountId(leaseOrderDO.getUserAccountId());
        refundUavWalletVO.setOrderStatus(1600);
        ResultBody giveBackPledge = userAppApi.feignPayUavWallet(refundUavWalletVO, currentAccount.getToken());
        if (!giveBackPledge.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
            return giveBackPledge;
        }
        return ResultBody.success();
    }

    private ApplyRefundVO wxRefund(LeaseOrderDO leaseOrderDO) {
        ApplyRefundVO applyRefundVO = new ApplyRefundVO();
        applyRefundVO.setOutTradeNo(leaseOrderDO.getOrderNo());
        applyRefundVO.setReason("租赁押金退款");
        applyRefundVO.setRefund(leaseOrderDO.getOtherPledge().multiply(BigDecimal.valueOf(100)).longValue());
        return applyRefundVO;
    }

    private PayUavWalletVO amountUnfreezing(LeaseOrderDO leaseOrderDO, BigDecimal shareAmount, BigDecimal salaryAmount) {
        PayUavWalletVO userPayUavWalletVO = new PayUavWalletVO();
        userPayUavWalletVO.setRemark("租赁订单金额解冻");
        userPayUavWalletVO.setCashAmount(shareAmount.add(leaseOrderDO.getShareCashPledge()));
        userPayUavWalletVO.setSalaryAmount(salaryAmount.add(leaseOrderDO.getSalaryCashPledge()));
        userPayUavWalletVO.setUserAccountId(leaseOrderDO.getUserAccountId());
        return userPayUavWalletVO;
    }

    @Override
    public ResultBody confirmSettlement(Integer id, BaseAccountDTO currentAccount) {
        LeaseOrderDO leaseOrderDO = leaseOrderDao.detail(id);
        log.info("平台确认归还操作，leaseOrderDO:{}", JSONObject.toJSONString(leaseOrderDO));
        if (leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.REMAIN_TO_BE_EVALUATED.getCode())) {
            return ResultBody.error("请勿重复操作");
        }
        boolean isModifiedConfirmed = leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.MODIFIED_DEPOSIT_TO_BE_CONFIRMED.getCode());
        boolean isRefund = leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.MODIFIED_DEPOSIT_REFUND.getCode());
        if (isModifiedConfirmed || isRefund) {
            // 根据订单id查询出修改后的押金
            LeaseRefundOrderDO leaseRefundOrderDO = leaseOrderDao.selectGiveBackByOrderId(id, OrderType.PFREC.getCode());
            BigDecimal operatingDeposit = BigDecimal.ZERO;
            operatingDeposit = operatingDeposit.add(leaseRefundOrderDO.getModifiedDeposit());
            // 判断用户之前的押金，然后进行操作：
            // 1、分别获取用户之前抵扣掉的押金
            BigDecimal shareCashPledge = leaseOrderDO.getShareCashPledge();
            BigDecimal salaryCashPledge = leaseOrderDO.getSalaryCashPledge();
            BigDecimal otherPledge = leaseOrderDO.getOtherPledge();
            // 一一判断押金是否够扣减的
            if (shareCashPledge.compareTo(BigDecimal.ZERO) > 0 || salaryCashPledge.compareTo(BigDecimal.ZERO) > 0) {
                if (shareCashPledge.compareTo(operatingDeposit) >= 0) {
                    // 云享金押金足够抵扣
                    shareCashPledge = shareCashPledge.subtract(operatingDeposit);
                } else {
                    operatingDeposit = operatingDeposit.subtract(shareCashPledge);
                    if (salaryCashPledge.compareTo(operatingDeposit) >= 0) {
                        // 云享金押金+余额押金足够抵扣
                        salaryCashPledge = salaryCashPledge.subtract(operatingDeposit);
                        // 进行后续操作
                    } else {
                        operatingDeposit = operatingDeposit.subtract(salaryCashPledge);
                        if (otherPledge.compareTo(operatingDeposit) >= 0) {
                            // 云享金押金+余额押金+其他押金足够抵扣
                            otherPledge = otherPledge.subtract(operatingDeposit);
                            // 进行后续操作

                        }
                    }
                }
                // 进行金额解冻
                PayUavWalletVO userPayUavWalletVO = amountUnfreezing(leaseOrderDO, leaseOrderDO.getShareAmount(), leaseOrderDO.getSalaryAmount());
                userPayUavWalletVO.setOrderStatus(999);
                ResultBody resultBody = userAppApi.feignPayUavWallet(userPayUavWalletVO, currentAccount.getToken());
                if (!resultBody.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                    return resultBody;
                }
                // 进行多余的押金退回
                if ((shareCashPledge.compareTo(BigDecimal.ZERO) > 0 || salaryCashPledge.compareTo(BigDecimal.ZERO) > 0)) {
                    moneyRefund(shareCashPledge, salaryCashPledge, currentAccount, leaseOrderDO);
                }
            } else {
                if (otherPledge.compareTo(operatingDeposit) >= 0) {
                    // 云享金押金+余额押金+其他押金足够抵扣
                    otherPledge = otherPledge.subtract(operatingDeposit);
                }
            }
            // 退回微信支付的押金部分
            if (otherPledge != null && otherPledge.compareTo(BigDecimal.ZERO) > 0) {
                ApplyRefundVO applyRefundVO = wxRefund(leaseOrderDO);
                applyRefundVO.setRefund(otherPledge.multiply(BigDecimal.valueOf(100)).longValue());
                ResultBody refundOtherPledge = paymentAppApi.applyRefund(applyRefundVO, currentAccount.getToken());
                if (!refundOtherPledge.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                    return refundOtherPledge;
                }
            }
            if (isModifiedConfirmed) {
                // 进行平台打款
                return businessIncome(currentAccount, id, leaseOrderDO, 0);
            }
            if (isRefund) {
                // 退回微信的钱
                if (leaseOrderDO.getOtherAmount().compareTo(BigDecimal.ZERO) > 0) {
                    ApplyRefundVO applyRefundVO = wxRefund(leaseOrderDO);
                    applyRefundVO.setRefund((leaseOrderDO.getOtherPledge().add(leaseOrderDO.getOtherAmount())).multiply(BigDecimal.valueOf(100)).longValue());
                    ResultBody refundOtherPledge = paymentAppApi.applyRefund(applyRefundVO, currentAccount.getToken());
                    if (!refundOtherPledge.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                        return refundOtherPledge;
                    }
                }
                if (leaseOrderDO.getShareAmount().compareTo(BigDecimal.ZERO) > 0 || leaseOrderDO.getSalaryAmount().compareTo(BigDecimal.ZERO) > 0) {
                    // 退回云享金及余额
                    PayUavWalletVO userPayUavWalletVO = new PayUavWalletVO();
                    userPayUavWalletVO.setUserAccountId(leaseOrderDO.getUserAccountId());
                    userPayUavWalletVO.setCashAmount(leaseOrderDO.getShareAmount());
                    userPayUavWalletVO.setSalaryAmount(leaseOrderDO.getSalaryAmount());
                    userPayUavWalletVO.setRemark("租赁订单退款");
                    userPayUavWalletVO.setOrderStatus(1700);
                    ResultBody resultBody = userAppApi.feignPayUavWallet(userPayUavWalletVO, currentAccount.getToken());
                    if (!resultBody.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                        return resultBody;
                    }
                }
                // 进行平台打款
                return businessIncome(currentAccount, id, leaseOrderDO, 1);
            }

            return ResultBody.success();

        } else {
            return ResultBody.error("状态流程有误");
        }
    }

    @Override
    public ResultBody remarkOrder(UavOrderRemarkVO uavOrderRemarkVO) {
        LeaseOrderDO leaseOrderDO = leaseOrderDao.detail(uavOrderRemarkVO.getUavOrderId());
        if (!leaseOrderDO.getStatusCode().equals(LeaseOrderStatus.REMAIN_TO_BE_EVALUATED.getCode())) {
            return ResultBody.error("状态流程有误");
        }
        // 修改订单状态
        this.updateLeaseOrderStatus(uavOrderRemarkVO.getUavOrderId(), LeaseOrderStatus.COMPLETE_TRANSACTION);
        leaseOrderDao.updateLeaseOrderRemarkStatus(leaseOrderDO.getId());
        // 填入评价内容
        UavOrderRemarkDO uavOrderRemarkDO = new UavOrderRemarkDO(uavOrderRemarkVO);
        uavOrderRemarkDO.setRemarkType(1);
        uavOrderRemarkDO.setMallGoodsId(leaseOrderDO.getId());
        leaseOrderDao.addRemarkOrder(uavOrderRemarkDO);
        return ResultBody.success();
    }
}
