package com.mmc.oms.service.uav.impl;

import com.mmc.oms.client.PmsClient;
import com.mmc.oms.common.ResultEnum;
import com.mmc.oms.common.constant.UavOrderProportion;
import com.mmc.oms.common.json.JsonUtil;
import com.mmc.oms.common.result.PageResult;
import com.mmc.oms.common.result.ResultBody;
import com.mmc.oms.common.util.CodeUtil;
import com.mmc.oms.common.util.KdnExpressUtil;
import com.mmc.oms.dao.UserAddressDao;
import com.mmc.oms.dao.uav.UavOrderDao;
import com.mmc.oms.dao.uav.UavOrderExpressDao;
import com.mmc.oms.dao.uav.UavPODao;
import com.mmc.oms.entity.mall.UserAddressDO;
import com.mmc.oms.entity.uav.*;
import com.mmc.oms.enums.UavOrderStatus;
import com.mmc.oms.enums.UavOrderType;
import com.mmc.oms.feign.PaymentAppApi;
import com.mmc.oms.feign.UserAppApi;
import com.mmc.oms.model.dto.ContractInfoDTO;
import com.mmc.oms.model.dto.kdn.KdnExpDTO;
import com.mmc.oms.model.dto.order.ExpStationsDTO;
import com.mmc.oms.model.dto.uav.*;
import com.mmc.oms.model.dto.user.BaseAccountDTO;
import com.mmc.oms.model.dto.user.UserAccountSimpleDTO;
import com.mmc.oms.model.qo.UserAccountQO;
import com.mmc.oms.model.qo.uav.UavOrderQO;
import com.mmc.oms.model.vo.ApplyRefundVO;
import com.mmc.oms.model.vo.demand.CommonPaymentVO;
import com.mmc.oms.model.vo.uav.*;
import com.mmc.oms.model.vo.wallet.TopUpOrderVO;
import com.mmc.oms.service.demand.DemandReleaseOrderService;
import com.mmc.oms.service.uav.UavOrderService;
import com.mmc.oms.service.uav.UavPOService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author: zj
 * @Date: 2023/9/4 15:31
 */
@Service
@Slf4j
public class UavOrderServiceImpl implements UavOrderService {
    @Autowired
    private UavOrderDao uavOrderDao;

    @Autowired
    private UavOrderExpressDao uavOrderExpressDao;

    @Autowired
    private UserAddressDao userAddressDao;

    @Autowired
    private UavPODao uavPODao;

    @Autowired
    private DemandReleaseOrderService demandReleaseOrderService;

    @Autowired
    private UavPOService uavPOService;

    @Autowired
    private UserAppApi userAppApi;

    @Autowired
    private PmsClient pmsClient;

    @Autowired
    private PaymentAppApi paymentAppApi;

    @Override
    @Transactional
    public ResultBody addOrder(AddUavOrderVO param, BaseAccountDTO currentAccount) {
        log.info("user:{},addOrder param:{}", currentAccount.getUserAccountId(), param);
        if (CollectionUtils.isEmpty(param.getSkuVOS())){
            return ResultBody.error("至少选择一个规格！");
        }
        MallGoodsVO mallGoodsVO = null;
        try {
            mallGoodsVO = pmsClient.mallGoodsDetails(param.getMallGoodsId());
            // 检验商品是否在卖
            if (mallGoodsVO == null || !mallGoodsVO.getShelfStatus().equals(1)){
                return ResultBody.error("商品已下架");
            }
            // 找到对应的规格
            for (UavOrderSkuVO skuVO : param.getSkuVOS()) {
                Boolean flag = false;
                for (PriceStockVO priceStockVO : mallGoodsVO.getPriceStock()) {
                    if (skuVO.getProductSpec().equals(priceStockVO.getProductSpec())) {
                        // 获取最新的规格单价
                        skuVO.setUnitPrice(priceStockVO.getSalePrice());
                        flag = true;
                        break;
                    }
                }
                if (!flag){
                    return ResultBody.error("选择规格不存在");
                }
            }
        }catch (Exception e){
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            e.printStackTrace();
        }
        String orderNo = CodeUtil.uavOrderCode();
        BigDecimal orderTotalAmount = BigDecimal.ZERO;
        BigDecimal otherAmount = BigDecimal.ZERO;
        for (UavOrderSkuVO skuVO : param.getSkuVOS()) {
            // 计算规格金额
            BigDecimal skuAmount = BigDecimal.valueOf(skuVO.getOrderNum()).multiply(skuVO.getUnitPrice());
            orderTotalAmount = orderTotalAmount.add(skuAmount);
        }
        otherAmount = otherAmount.add(orderTotalAmount);
        // 校验云享金、钱包余额是否足够抵扣
        BigDecimal shareAmount = BigDecimal.ZERO;
        BigDecimal salaryAmount = BigDecimal.ZERO;
        ResultBody<PayWalletDTO> payResInfo = userAppApi.getCurrentUserPayWalletInfo(currentAccount.getToken());

        if (!payResInfo.getCode().equals(ResultEnum.SUCCESS.getResultCode()) || payResInfo.getResult() == null){
            return payResInfo;
        }
        PayWalletDTO payWalletDTO = payResInfo.getResult();
        if (param.getDeductShareAmount().equals(1)){
            if (otherAmount.compareTo(payWalletDTO.getCashAmt()) >= 0) {
                // 抵扣云享金小于等于订单总额时，直接使用云享金
                shareAmount = payWalletDTO.getCashAmt();
            }else {
                // 余额超过订单金额时，订单总额使用余额支付
                shareAmount = otherAmount;
            }
        }
        otherAmount = otherAmount.subtract(shareAmount);
        Integer statusCode = null;
        // 支付完成
        if (otherAmount.compareTo(BigDecimal.ZERO) == 0 ){
            statusCode = UavOrderStatus.WAITING_DELIVER_GOODS.getCode();
        }else {
            if (param.getDeductSalaryAmount().equals(1)){
                if (otherAmount.compareTo(payWalletDTO.getSalaryAmt()) >= 0) {
                    // 抵扣余额小于等于订单总额时，直接使用余额
                    salaryAmount = payWalletDTO.getSalaryAmt();
                }else {
                    // 余额超过订单金额时，订单总额使用余额支付
                    salaryAmount = otherAmount;
                }
            }
            otherAmount = otherAmount.subtract(salaryAmount);
            if (otherAmount.compareTo(BigDecimal.ZERO) == 0) {
                statusCode = UavOrderStatus.WAITING_DELIVER_GOODS.getCode();
            }
        }
        PayUavWalletVO payUavWalletVO = new PayUavWalletVO();
        payUavWalletVO.setCashAmount(shareAmount);
        payUavWalletVO.setSalaryAmount(salaryAmount);
        payUavWalletVO.setOrderStatus(100);
        payUavWalletVO.setUserAccountId(currentAccount.getUserAccountId());
        ResultBody resultBody = userAppApi.feignPayUavWallet(payUavWalletVO, currentAccount.getToken());
        if (!resultBody.getCode().equals(ResultEnum.SUCCESS.getResultCode())){
            return resultBody;
        }
        // 添加总订单表
        UavOrderDO uavOrderDO = new UavOrderDO();
        uavOrderDO.setOrderNo(orderNo);
        uavOrderDO.setOrderTotalAmount(orderTotalAmount);
        uavOrderDO.setOrderType(param.getOrderType());
        if (param.getOrderType().equals(UavOrderType.PAY.getCode())){
            uavOrderDO.setStatusCode(otherAmount.compareTo(BigDecimal.ZERO) > 0 ? UavOrderStatus.PAYING.getCode() : statusCode);
        }else {
            uavOrderDO.setStatusCode(UavOrderStatus.CONFIRM.getCode());
        }
        log.info("钱包使用情况：salaryAmount-{}，shareAmount-{}，otherAmount-{}", salaryAmount, shareAmount, otherAmount);
        uavOrderDO.setUserAccountId(currentAccount.getUserAccountId());
        uavOrderDO.setThirdBackUserAccountId(param.getThirdBackUserAccountId());
        uavOrderDO.setCompanyName(param.getCompanyName());
        uavOrderDO.setSalaryAmount(salaryAmount);
        uavOrderDO.setShareAmount(shareAmount);
        uavOrderDO.setOtherAmount(otherAmount);
        uavOrderDO.setUserAddressId(param.getUserAddressId());
        uavOrderDO.setUserRemark(param.getUserRemark());
        uavOrderDao.addOrder(uavOrderDO);

        List<UavOrderSkuDO> skuDOS = new ArrayList<>();
        for (UavOrderSkuVO skuVO : param.getSkuVOS()) {
            UavOrderSkuDO uavOrderSkuDO = new UavOrderSkuDO();
            // 添加订单规格信息
            uavOrderSkuDO.setUavOrderId(uavOrderDO.getId());
            uavOrderSkuDO.setMallGoodsId(param.getMallGoodsId());
            uavOrderSkuDO.setTradeName(mallGoodsVO.getTradeName());
            uavOrderSkuDO.setPriceStockId(skuVO.getPriceStockId());
            uavOrderSkuDO.setProductSpec(skuVO.getProductSpec());
            uavOrderSkuDO.setOrderNum(skuVO.getOrderNum());
            uavOrderSkuDO.setUnitPrice(skuVO.getSalePrice());
            uavOrderSkuDO.setSkuNo(skuVO.getSkuNo());
            uavOrderSkuDO.setSkuImage(skuVO.getSkuImage());
            skuDOS.add(uavOrderSkuDO);
        }
        uavOrderDao.addOrderSku(skuDOS);
        // 添加收货地址
        UserAddressDO userAddressDO = userAddressDao.selectById(param.getUserAddressId());
        UavOrderExpressDO uavOrderExpressDO = new UavOrderExpressDO(userAddressDO);
        uavOrderExpressDO.setUavOrderId(uavOrderDO.getId());
        uavOrderExpressDao.addAddress(uavOrderExpressDO);
        return ResultBody.success(uavOrderDO.buildUavOrderDTO());
    }

    @Override
    public ResultBody detail(Integer id, BaseAccountDTO baseAccountDTO) throws Exception {
        UavOrderDO uavOrderDO = uavOrderDao.detail(id);
        UavOrderDTO uavOrderDTO = uavOrderDO.buildUavOrderDTO();
        // 收货信息、付款信息、规格信息、快递信息、签署信息
        UavOrderExpressDO uavOrderExpressDO = uavOrderExpressDao.getUavOrderExpressDO(id);
        if (uavOrderExpressDO != null) {
            uavOrderDTO.setUavOrderExpressDTO(uavOrderExpressDO.buildUavOrderExpressDTO());
        }
        List<UavOrderPayDO> payDOS = uavOrderDao.listUavOrderPay(id);
        if (CollectionUtils.isNotEmpty(payDOS)) {
            List<UavOrderPayDTO> collect = payDOS.stream().map(UavOrderPayDO::buildUavOrderPayDTO).collect(Collectors.toList());
            uavOrderDTO.setPayDTOList(collect);
        }
        List<UavOrderSkuDO> skuDOS =uavOrderDao.listUavOrderSkuDO(id);
        if (CollectionUtils.isNotEmpty(skuDOS)) {
            List<UavOrderSkuDTO> uavOrderSkuDTOS = skuDOS.stream().map(UavOrderSkuDO::buildUavOrderSkuDTO).collect(Collectors.toList());
            uavOrderDTO.setSkuDTOList(uavOrderSkuDTOS);
        }
        // 快递信息
        KdnExpDTO kdnExpDTO = this.getOrderExpInfo(uavOrderExpressDO);
        if (kdnExpDTO != null) {
            uavOrderDTO.setKdnExpDTO(kdnExpDTO);
        }
        // 合同签署信息
        List<String> orderNos = Arrays.asList(uavOrderDO.getOrderNo());
        List<ContractInfoDTO> contractInfoDTOS = userAppApi.listContractInfoByOrderNo(orderNos, baseAccountDTO.getToken());
        if (CollectionUtils.isNotEmpty(contractInfoDTOS)) {
            uavOrderDTO.setContractInfoDTO(contractInfoDTOS.get(0));
        }
        return ResultBody.success(uavOrderDTO);
    }

    private KdnExpDTO getOrderExpInfo(UavOrderExpressDO exp) throws Exception {
        String json = "";
        if ("SF".equals(exp.getSendExpCode())) {
            json = KdnExpressUtil.SF(exp.getTakePhone(), exp.getSendExpNo());
        } else if ("JD".equals(exp.getSendExpCode())) {
            json = KdnExpressUtil.JD("", exp.getSendExpNo());
        } else {
            json = KdnExpressUtil.others(exp.getSendExpCode(), exp.getSendExpNo());
        }
        KdnExpDTO kdn = JsonUtil.parseJsonToObj(json, KdnExpDTO.class);
        kdn.setLogisticCode(exp.getSendExpNo());
        if (!org.springframework.util.CollectionUtils.isEmpty(kdn.getTraces())) {
            Collections.sort(kdn.getTraces(), Comparator.comparing(ExpStationsDTO::getAcceptTime).reversed());// 升序
        }
        return kdn;
    }

    @Transactional
    @Override
    public ResultBody close(Integer id, String token) {
        UavOrderDO uavOrderDO = uavOrderDao.detail(id);
        // 修改订单状态
        this.updateUavOrderStatus(id, UavOrderStatus.CLOSE);
        // 退云享金、佣金
        PayUavWalletVO payUavWalletVO = new PayUavWalletVO();
        payUavWalletVO.setCashAmount(uavOrderDO.getShareAmount());
        payUavWalletVO.setSalaryAmount(uavOrderDO.getSalaryAmount());
        payUavWalletVO.setOrderStatus(1400);
        payUavWalletVO.setUserAccountId(uavOrderDO.getUserAccountId());
        ResultBody resultBody = userAppApi.feignPayUavWallet(payUavWalletVO, token);
        if (!resultBody.getCode().equals(ResultEnum.SUCCESS.getResultCode())){
            return resultBody;
        }
        // 微信退款
        ApplyRefundVO applyRefundVO = new ApplyRefundVO();
        applyRefundVO.setOutTradeNo(uavOrderDO.getOrderNo());
        applyRefundVO.setReason("关闭订单");
        applyRefundVO.setRefund(uavOrderDO.getOtherAmount().multiply(BigDecimal.valueOf(100)).longValue());
        ResultBody applyRefund = paymentAppApi.applyRefund(applyRefundVO);
        if (!applyRefund.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody list(UavOrderQO uavOrderQO, BaseAccountDTO currentAccount) {
        if (!currentAccount.getRoleInfo().getSuperAdmin().equals(1)){
            uavOrderQO.setThirdBackUserAccountId(currentAccount.getUserAccountId());
        }
        int count = uavOrderDao.countList(uavOrderQO);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(uavOrderQO.getPageNo(), uavOrderQO.getPageSize(), count));
        }
        Integer pageNo = uavOrderQO.getPageNo();
        uavOrderQO.buildCurrentPage();
        List<UavOrderDO> uavOrderDOS = uavOrderDao.list(uavOrderQO);
        List<UavOrderDTO> uavOrderDTOS = uavOrderDOS.stream().map(UavOrderDO::buildUavOrderDTO).collect(Collectors.toList());
        List<Integer> userIds = uavOrderDTOS.stream().map(UavOrderDTO::getUserAccountId).collect(Collectors.toList());
        // TODO 买家信息
        UserAccountQO userAccountQO = new UserAccountQO();
        userAccountQO.setUserIds(userIds);
        List<UserAccountSimpleDTO> userDTOS = userAppApi.feignListAppUserAccount(userAccountQO, currentAccount.getToken());
        if (CollectionUtils.isNotEmpty(userDTOS)) {
            Map<Integer, UserAccountSimpleDTO> collect = userDTOS.stream().collect(Collectors.toMap(UserAccountSimpleDTO::getId, user -> user, (k1, k2) -> k2));
            for (UavOrderDTO uavOrderDTO : uavOrderDTOS) {
                if (collect.containsKey(uavOrderDTO.getUserAccountId())) {
                    uavOrderDTO.setPhoneNum(collect.get(uavOrderDTO.getUserAccountId()).getPhoneNum());
                    uavOrderDTO.setNickname(collect.get(uavOrderDTO.getUserAccountId()).getNickName());
                    uavOrderDTO.setUserName(collect.get(uavOrderDTO.getUserAccountId()).getUserName());
                }
            }
        }
        return ResultBody.success(PageResult.buildPage(pageNo, uavOrderQO.getPageSize(), count, uavOrderDTOS));
    }

    @Override
    public ResultBody listByUid(UavOrderQO uavOrderQO, BaseAccountDTO currentAccount) {
        uavOrderQO.setUserAccountId(currentAccount.getUserAccountId());
        int count = uavOrderDao.countList(uavOrderQO);
        if (count == 0) {
            return ResultBody.success(PageResult.buildPage(uavOrderQO.getPageNo(), uavOrderQO.getPageSize(), count));
        }
        Integer pageNo = uavOrderQO.getPageNo();
        uavOrderQO.buildCurrentPage();
        List<UavOrderDO> uavOrderDOS = uavOrderDao.list(uavOrderQO);
        List<UavOrderDTO> uavOrderDTOS = uavOrderDOS.stream().map(UavOrderDO::buildUavOrderDTO).collect(Collectors.toList());
        return ResultBody.success(PageResult.buildPage(pageNo, uavOrderQO.getPageSize(), count, uavOrderDTOS));
    }

    @Override
    public ResultBody pay(Integer id, BaseAccountDTO currentAccount, HttpServletRequest request) {
        // 发起微信支付
        UavOrderDO uavOrderDO = uavOrderDao.detail(id);
        if (!uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.PAYING.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        CommonPaymentVO commonPaymentVO = new CommonPaymentVO();
        int shouldPay = uavOrderDO.getOtherAmount().multiply(BigDecimal.valueOf(100)).intValue();
        if (shouldPay <= 0) {
            return ResultBody.success("已支付完成");
        }
        List<UavOrderSkuDO> orderSkuDOS = uavOrderDao.listUavOrderSkuDO(id);
        StringBuffer sb = new StringBuffer();
        orderSkuDOS.stream().forEach(uavOrderSkuDO -> sb.append(uavOrderSkuDO.getProductSpec()));
        commonPaymentVO.setAmount(shouldPay);
        commonPaymentVO.setDescription(sb.toString());
        commonPaymentVO.setOrderNumber(uavOrderDO.getOrderNo());
        commonPaymentVO.setOrderPort(0);
        ResultBody resultBody = demandReleaseOrderService.orderPayment(commonPaymentVO, request, currentAccount.getUserAccountId());
        return resultBody;
    }

    @Transactional
    @Override
    public ResultBody send(UavOrderExpressVO param) {
        // 修改订单状态
        UavOrderDO uavOrderDO = uavOrderDao.detail(param.getUavOrderId());
        if (!uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.WAITING_DELIVER_GOODS.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        this.updateUavOrderStatus(param.getUavOrderId(), UavOrderStatus.WAITING_DELIVER_GOODS);
        // 修改收货信息
        UavOrderExpressDO uavOrderExpressDO = new UavOrderExpressDO(param);
        uavOrderExpressDao.updateUavOrderExpressDO(uavOrderExpressDO);
        return ResultBody.success();
    }

    @Transactional
    @Override
    public ResultBody receive(Integer id, BaseAccountDTO currentAccount) {
        // 修改订单状态
        UavOrderDO uavOrderDO = uavOrderDao.detail(id);
        if (!uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.WAITING_RECEIVE_GOODS.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        this.updateUavOrderStatus(id, UavOrderStatus.WAITING_RECEIVE_GOODS);
        // 修改收货信息
        UavOrderExpressDO uavOrderExpressDO = uavOrderExpressDao.getUavOrderExpressDO(id);
        uavOrderExpressDO.setReceive(1);
        uavOrderExpressDO.setReceiveTime(new Date());
        uavOrderExpressDao.updateUavOrderExpressDO(uavOrderExpressDO);
        if (uavOrderDO.getOrderType().equals(0)) {
            // 订单抽成设置
            BigDecimal thirdAmount = uavOrderDO.getOrderTotalAmount().multiply(
                    BigDecimal.valueOf(100).subtract(UavOrderProportion.PROPORTION).divide(BigDecimal.valueOf(100)));
            PayUavWalletVO payUavWalletVO = new PayUavWalletVO();
            payUavWalletVO.setCashAmount(BigDecimal.ZERO);
            payUavWalletVO.setSalaryAmount(thirdAmount);
            payUavWalletVO.setOrderStatus(1500);
            payUavWalletVO.setRemark("商城订单收入，订单编号" + uavOrderDO.getOrderNo());
            // 拿到商家管理员的用户id
            ResultBody managerRes = userAppApi.getManagerIdByBackUserId(uavOrderDO.getThirdBackUserAccountId(), null);
            if (!managerRes.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
                return managerRes;
            }
            payUavWalletVO.setUserAccountId((Integer) managerRes.getResult());
            ResultBody resultBody = userAppApi.feignPayUavWallet(payUavWalletVO, null);
            if (!resultBody.getCode().equals(ResultEnum.SUCCESS.getResultCode())) {
                TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
                return resultBody;
            }
        }
        return ResultBody.success();
    }

    @Transactional
    @Override
    public ResultBody remarkOrder(UavOrderRemarkVO uavOrderRemarkVO) {
        UavOrderDO uavOrderDO = uavOrderDao.detail(uavOrderRemarkVO.getUavOrderId());
        if (!uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.WAITING_REMARK.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        // 修改订单状态
        this.updateUavOrderStatus(uavOrderRemarkVO.getUavOrderId(), UavOrderStatus.WAITING_REMARK);
        // 填入评价内容
        UavOrderRemarkDO uavOrderRemarkDO = new UavOrderRemarkDO(uavOrderRemarkVO);
        uavOrderDao.addRemarkOrder(uavOrderRemarkDO);
        return ResultBody.success();
    }

    @Transactional
    public void updateUavOrderStatus(Integer id, UavOrderStatus uavOrderStatus){
        uavOrderDao.updateUavOrderStatus(id, uavOrderStatus.getNextCode());
    }

    @Override
    public ResultBody sellerRemark(Integer id, String content) {
        uavOrderDao.sellerRemark(id, content);
        return ResultBody.success();
    }

    @Override
    public ResultBody statusList() {
        List<UavOrderStatusDO> statusDOS = uavOrderDao.statusList();
        List<UavOrderStatusDTO> uavOrderStatusDTOS = statusDOS.stream().map(UavOrderStatusDO::buildUavOrderStatusDTO).collect(Collectors.toList());
        return ResultBody.success(uavOrderStatusDTOS);
    }

    @Transactional
    @Override
    public ResultBody upLoadPay(UavOrderPayVO uavOrderPayVO) {
        UavOrderDO uavOrderDO = uavOrderDao.detail(uavOrderPayVO.getUavOrderId());
        // 修改订单状态
        if (uavOrderDO.getStatusCode().equals(UavOrderStatus.PAYING.getCode())) {
            uavOrderDao.updateUavOrderStatus(uavOrderDO.getId(), UavOrderStatus.FINISH_PAYING.getCode());
        }
        UavOrderPayDO uavOrderPayDO = new UavOrderPayDO(uavOrderPayVO);
        uavOrderPayDO.setUavOrderId(uavOrderPayVO.getUavOrderId());
        // 审核状态，0待审批，1已通过，2未通过
        uavOrderPayDO.setCheckStatus(0);
        // 0微信支付，1支付宝，2线下支付凭证
        uavOrderPayDO.setPayType(2);
        uavOrderDao.addOrderPay(uavOrderPayDO);
        return ResultBody.success();
    }

    @Transactional
    @Override
    public ResultBody checkPay(UavOrderPayVO uavOrderPayVO) {
        // 一个订单可能会多次上传付款凭证，因此单独审核
        UavOrderDO uavOrderDO = uavOrderDao.detail(uavOrderPayVO.getUavOrderId());
        if (uavOrderPayVO.getCheckStatus().equals(1)){
            // 第一次审核通过，则要把订单状态改为待发货；意向订单还要生成采购订单
            if (uavOrderDO.getStatusCode().equals(UavOrderStatus.FINISH_PAYING.getCode())) {
                this.updateUavOrderStatus(uavOrderPayVO.getUavOrderId(), UavOrderStatus.FINISH_PAYING);
                // TODO 生成采购订单
                UavPurchaseOrderDO uavPurchaseOrderDO = new UavPurchaseOrderDO();
                uavPurchaseOrderDO.setBackUserAccountId(1);
                uavPurchaseOrderDO.setThirdUserAccountId(uavOrderDO.getThirdBackUserAccountId());
                uavPurchaseOrderDO.setOrderAmount(uavOrderDO.getOrderTotalAmount());
                uavPurchaseOrderDO.setOrderNo(CodeUtil.uavPOrderCode());
                uavPurchaseOrderDO.setUavOrderId(uavOrderDO.getId());
                uavPurchaseOrderDO.setStatusCode(UavOrderStatus.CONFIRM.getCode());
                uavPODao.addPurchaseOrder(uavPurchaseOrderDO);
            }
        }
        UavOrderPayDO uavOrderPayDO = new UavOrderPayDO(uavOrderPayVO);
        uavOrderDao.checkPay(uavOrderPayDO);
        return ResultBody.success();
    }

    @Transactional
    @Override
    public void payUavOrder(TopUpOrderVO topUpOrderVO) {
        // 添加支付凭证、记录支付方式
        UavOrderDO uavOrderDO = uavOrderDao.detailByNo(topUpOrderVO.getOrderNo());
        if (uavOrderDO == null) {
            throw new RuntimeException();
        }
        // TODO 有问题
        this.updateUavOrderStatus(uavOrderDO.getId(), UavOrderStatus.PAYING);
        // 修改订单信息
        uavOrderDO.setPayType("wechatpay");
        uavOrderDO.setPayTime(new Date());
        uavOrderDao.updateUavOrderPayInfo(uavOrderDO);
        UavOrderPayDO uavOrderPayDO = new UavOrderPayDO();
        uavOrderPayDO.setUavOrderId(uavOrderDO.getId());
        uavOrderPayDO.setPayRemark(topUpOrderVO.getTradeStateDesc());
        uavOrderPayDO.setPayType(0);
        uavOrderPayDO.setPayAmount(BigDecimal.valueOf(topUpOrderVO.getAmount()).divide(BigDecimal.valueOf(100)));
        uavOrderPayDO.setCheckStatus(1);
        uavOrderPayDO.setCreateTime(new Date());
        uavOrderDao.addOrderPay(uavOrderPayDO);
    }

    @Transactional
    @Override
    public ResultBody confirmOrder(UavOrderVO uavOrderVO, BaseAccountDTO currentAccount) {
        // 修改订单状态
        UavOrderDO uavOrderDO = uavOrderDao.detail(uavOrderVO.getId());
        if (!uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.CONFIRM.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        //核验合同签署状态，确认平台要先完成签署
        List<String> orderNos = Arrays.asList(uavOrderDO.getOrderNo());
        List<ContractInfoDTO> contractInfoDTOS = userAppApi.listContractInfoByOrderNo(orderNos, currentAccount.getToken());
        if (CollectionUtils.isEmpty(contractInfoDTOS) || !contractInfoDTOS.get(0).getSignStatus().equals(2)) {
            return ResultBody.error("平台未签署合同");
        }
        // 修改订单信息
        uavOrderDO.setOrderTotalAmount(uavOrderVO.getOrderTotalAmount());
        uavOrderDO.setDeliveryTime(uavOrderVO.getDeliveryTime());
        uavOrderDO.setSellerRemark(uavOrderVO.getSellerRemark());
        uavOrderDO.setStatusCode(UavOrderStatus.SIGN.getCode());
        uavOrderDao.updateUavOrder(uavOrderDO);
        return ResultBody.success();
    }

    @Override
    public ResultBody checkNoReceive() {
        // 超时未签收的订单列表
        // 自动签收
        List<UavOrderExpressDO> uavOrderExpressDOS = uavOrderExpressDao.listNoReceive();
        for (UavOrderExpressDO expressDO : uavOrderExpressDOS) {
            this.receive(expressDO.getUavOrderId(), null);
        }
        return ResultBody.success();
    }

    @Override
    public ResultBody uavOrderFinishSign(String orderNo) {
        UavOrderDO uavOrderDO = uavOrderDao.detailByNo(orderNo);
        if (uavOrderDO == null) {
            return ResultBody.error(ResultEnum.NOT_FOUND);
        }
        if (!uavOrderDO.getStatusCode().toString().equals(UavOrderStatus.SIGN.getCode().toString())) {
            return ResultBody.error("操作有误，请重试！");
        }
        this.updateUavOrderStatus(uavOrderDO.getId(), UavOrderStatus.SIGN);
        return ResultBody.success();
    }
}
