package com.mmc.oms.service.Impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.mmc.oms.common.*;
import com.mmc.oms.dao.RentalOrdersDao;
import com.mmc.oms.entity.*;
import com.mmc.oms.model.dto.*;
import com.mmc.oms.model.qo.OrderInfoQO;
import com.mmc.oms.model.vo.*;
import com.mmc.oms.service.RentalOrdersService;
import com.mmc.oms.wx.WxMsgTemplete;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @Author small @Date 2023/5/26 10:53 @Version 1.0
 */
@Service
public class RentalOrdersServiceImpl implements RentalOrdersService {

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private RentalOrdersDao rentalOrdersDao;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;


    @Value("${userapp.url}")
    private String userAppUrl;

    @Value("${pms.url}")
    private String pmsUrl;

    @Value("${payment.url}")
    private String paymentUrl;


    @Override
    public OrderInfoDTO feignAddLease(BaseAccountDTO user, LeaseOrderVO param) {
        //用户信息的远程调用
        ResponseEntity<String> responseEntity = UserId(user.getUserAccountId());
        UserAccountSimpleDTO account = JSON.parseObject(responseEntity.getBody(), UserAccountSimpleDTO.class);
        //优惠券id获取优惠券面值

        OrderInfoDO orderInfo = new OrderInfoDO(param);
        PriceAcquisition priceAcquisition = getPriceAcquisition(param, account);
        BigDecimal unitPrice = getUnitPrice(priceAcquisition);
        //每天单价
        orderInfo.setUnitPrice(unitPrice);
        // 天数
        orderInfo.setActualPay(extracted(param, unitPrice));
        orderInfo.setPayDay(day(param));
        orderInfo.setRepoAccountId(user.getUserAccountId());
        orderInfo.setUid(user.getUid());
        orderInfo.setBuyerName(user.getUserName());
        orderInfo.setBuyerPhone(user.getPhoneNum());
        rentalOrdersDao.insertOrderInfo(orderInfo);
        // 添加收货地址
        Date cdate = new Date();
        OrderReceiptVO rv = param.getOrderReceipt();
        OrderReceiptDO receipt = new OrderReceiptDO(rv);
        receipt.setOrderInfoId(orderInfo.getId());
        receipt.setCreateTime(cdate);
        rentalOrdersDao.insertOrderReceipt(receipt);
        //获取剩余的余额
        orderInfo.setBalance(getRemainingBalance(user.getToken(), user.getUserAccountId()));
        // 将当前的用户id及订单编号存入redis
        OrderInfoDTO dto = orderInfo.buildOrderInfoDTO();
        stringRedisTemplate.opsForValue().set(orderInfo.getOrderNo(), JsonUtil.parseObjToJson(dto));
        return dto;
    }

    /**
     * 实际需要支付价格计算
     *
     * @param param
     */
    private BigDecimal extracted(LeaseOrderVO param, BigDecimal unitPrice) {
        CouponDO couponDO = rentalOrdersDao.couponMoney(param.getCouponId());
        BigDecimal couponMoney = BigDecimal.valueOf(0);
        if (couponDO != null) {
            couponMoney = couponDO.getCouponMoney();
        }
        //实际需要支付的价格
        BigDecimal dayDecimal = BigDecimal.valueOf(day(param));
        BigDecimal multiply = dayDecimal.multiply(unitPrice);
        BigDecimal actualPay = multiply.subtract(couponMoney);
        return actualPay;
    }

    //远程设置
    private PriceAcquisition getPriceAcquisition(LeaseOrderVO param, UserAccountSimpleDTO account) {
        PriceAcquisition priceAcquisition = new PriceAcquisition();
        priceAcquisition.setUserId(account.getId());
        priceAcquisition.setDay(day(param));
        priceAcquisition.setSpecsId(param.getSpecsId());
        priceAcquisition.setChannelLevelId(account.getCooperationTagId());
        return priceAcquisition;
    }


    private ResponseEntity<String> UserId(Integer repoAccountId) {
        // 用户信息
        ResponseEntity<String> response =
                restTemplate.getForEntity(
                        userAppUrl
                                + "user-account/feignGetUserSimpleInfo?userAccountId="
                                + repoAccountId,
                        String.class);
        return response;
    }


    /**
     * 通过用户id 获取用户剩余的金额
     *
     * @param repoAccountId
     * @return
     */
    public BigDecimal getRemainingBalance(String token, Integer repoAccountId) {
        HttpHeaders headers = new HttpHeaders();
        //封装请求头
        headers.add("token", token);
        HttpEntity<MultiValueMap<String, Object>> formEntity = new HttpEntity<MultiValueMap<String, Object>>(headers);
        ResponseEntity<String> result4 = restTemplate.exchange(paymentUrl + "repocash/RemainingBalance?repoAccountId=" + repoAccountId,
                HttpMethod.GET, formEntity, String.class);
        String body = result4.getBody();
        BigDecimal bigDecimal = new BigDecimal(body);
        return bigDecimal;
    }

    /**
     * 获取渠道等级
     *
     * @param userId
     * @return
     */
    public Integer getChannelLevel(Integer userId) {
        // 远程接口调用 传一个userId获取到渠道等级的id 接口找洁
        // 需要调整
    /* ResponseEntity<String> response =
            restTemplate.getForEntity("http://localhost:35150/userapp/user-account/feignGetUserSimpleInfo?userAccountId=1",String.class);
    String body1 = response.getBody();
    UserAccountSimpleDTO account = JSON.parseObject(body1, UserAccountSimpleDTO.class);*/
        return 1;
    }


    /**
     * 获取每一天的价格
     *
     * @param
     * @param param
     * @return
     */
    private BigDecimal getUnitPrice(PriceAcquisition param) {
        // 需要支付的价格
       /* ResponseEntity<String> responseEntity = restTemplate.postForEntity(pmsUrl + "product/spec/getSpecLeaseUnitPrice"
                , param, String.class);
        String body = responseEntity.getBody();
        BigDecimal bigDecimal = new BigDecimal(body);*/
        // 模拟
        int i = 12;
        BigDecimal bigDecimal = BigDecimal.valueOf(i);
        return bigDecimal;
    }

    public Integer day(LeaseOrderVO param) {
        Date startDate = param.getStartDate();
        Date endDate = param.getEndDate();
        long start = startDate.getTime();
        long end = endDate.getTime();
        long day = end - start;
        Integer i = (int) day / 24 / 60 / 60 / 1000;
        return i;
    }

    private LeaseOrderVO buildLease(
            RepoAccountDTO account, RentalOrderVO param, SkuOrderDTO sku, OrderReceiptVO rv) {
        Date cdate = new Date();
        return LeaseOrderVO.builder()
                .orderNo(CodeUtil.createRepoOrderNo())
                .wareInfoId(sku.getWareInfoId())
                .wareTitle(sku.getWareTitle())
                .wareNo(sku.getWareNo())
                .skuInfoId(sku.getSkuInfoId())
                .skuTitle(sku.getSkuTitle())
                .unitPrice(sku.getUnitPrice())
                .wareNum(sku.getWareNum())
                .shouldPay(sku.getShouldPay())
                .actualPay(BigDecimal.ZERO)
                .orderType(0)
                .deposit(sku.getDeposit())
                .rentPrice(sku.getRentPrice())
                .startDate(param.getStartDate())
                .endDate(param.getEndDate())
                .payDay(sku.getRentDay())
                .tranStatus(TranStatusDic.INIT.getStatus())
                .remark(param.getRemark())
                .rcdCompanyId(account.getRcdCompanyId())
                .exWare(sku.getPayStatus())
                .wareImg(sku.getMainImg())
                .createTime(cdate)
                .orderReceipt(rv)
                .build();
    }

    @Transactional
    public ResultBody computeStockNum(
            Integer wareInfoId, Integer skuInfoId, Integer addStock, Integer addSale) {
        if (addStock != null && addStock < 0) {
            int stock = (addStock * -1);
            WareInfoDO ware = null;
            // wareInfoDao.getWareSimpleInfo(wareInfoId);
            if (ware.getTotalStock() < stock) {
                return ResultBody.error(ResultEnum.WARE_NOT_ENOUGH_ERROR);
            }
            int newStock = ware.getTotalStock() - stock;
            if (newStock == 0) {
                // 下架
                // wareInfoDao.updateDownWare(wareInfoId);
            }
        }
        if (addStock == null) {
            addStock = 0;
        }
        int count = 0;
        // skuInfoDao.updateSkuNumInfo(skuInfoId,addStock,addSale);
        if (count != 1) {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            return ResultBody.error(ResultEnum.OPERATION_SYS_ERROR);
        }
        int updateCount = 0;
        // wareInfoDao.updateWareNumInfo(wareInfoId,addStock,addSale);

        if (updateCount != 1) {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            return ResultBody.error(ResultEnum.OPERATION_SYS_ERROR);
        }
        return ResultBody.success();
    }

    @Override
    public OrderInfoDTO getOrderDetail(String orderNo) {
        OrderInfoDO order = rentalOrdersDao.getOrderDetail(orderNo);
        OrderInfoDTO orderInfo = order.buildOrderInfoDTO();
        OrderReceiptDO rd = rentalOrdersDao.getOrderReceiptInfo(order.getId());
        OrderReceiptDTO receipt = rd.buildOrderReceiptDTO();
        orderInfo.setReceipt(receipt);
        // 查询-最新的退款详情单
        OrderRefundDO orderRefund = rentalOrdersDao.getNewOrderRefund(order.getId());
        if (orderRefund != null) {
            OrderRefundDTO refund = orderRefund.buildOrderRefundDTO();
            orderInfo.setOrderRefund(refund);
        }
        return orderInfo;
    }

    @Override
    public PageResult listPcOrderInfo(OrderInfoQO param, BaseAccountDTO cuser) {

        Integer count = rentalOrdersDao.countPcOrderInfo(param);
        if (count == 0) {
            return PageResult.buildPage(param.getPageNo(), param.getPageSize(), count);
        }
        Integer pageNo = param.getPageNo();
        param.buildCurrentPage();
        List<OrderInfoDTO> data =
                rentalOrdersDao.listPcOrderInfo(param).stream()
                        .map(
                                d -> {
                                    return d.buildOrderInfoDTO();
                                })
                        .collect(Collectors.toList());
        return PageResult.buildPage(pageNo, param.getPageSize(), count, data);
    }

    @Override
    public PageResult listPageWechatOrder(BaseAccountDTO account, OrderInfoQO param) {
        param.setRepoAccountId(account.getUserAccountId());
        int count = rentalOrdersDao.countPageOrderInfo(param);
        if (count == 0) {
            return PageResult.buildPage(param.getPageNo(), param.getPageSize(), count);
        }
        Integer pageNo = param.getPageNo();
        param.buildCurrentPage();
        List<OrderInfoDTO> data =
                rentalOrdersDao.listPageOrderInfo(param).stream()
                        .map(
                                d -> {
                                    return d.buildOrderInfoDTO();
                                })
                        .collect(Collectors.toList());
        return PageResult.buildPage(pageNo, param.getPageSize(), count, data);
    }

    @Override
    public Object orderStatusChanges(String orderNo, Integer tranStatus) {
        int i = rentalOrdersDao.orderStatusChanges(orderNo, tranStatus);
        return i;
    }


    @Override
    @Transactional
    public ResultBody sendOrderWare(BaseAccountDTO cuser, OrderVcuVO param) {
        // 密码认证
        // ResultBody body = userServletClient.feignAuthUserPwd(cuser.getId(), param.getAuthPwd());
        /*if (!ResultEnum.SUCCESS.getResultCode().equals(body.getCode())) {
            return body;
        }*/
        // 密码认证
        ResultBody body = passwordAuthentication(cuser, param.getAuthPwd());
        if (body != null) {
            return body;
        }
        OrderInfoDO order = rentalOrdersDao.getOrderInfo(param.getOrderInfoId());
        if (!TranStatusDic.YFK.getStatus().equals(order.getTranStatus())
                && !TranStatusDic.TKSQ.getStatus().equals(order.getTranStatus())) {
            return ResultBody.error(ResultEnum.ORDER_STATUS_ERROR);
        }
        Date cdate = new Date();
        // 修改收发货地址信息
        OrderReceiptDO receipt = param.buildReceipt();
        rentalOrdersDao.updateOrderReceipt(receipt);

        // 订单-视频、图片
        OrderVcuDO vcu = param.buildOrdeVcu();
        vcu.setVcuType(VcuType.SEND.getCode());
        vcu.setCreateTime(cdate);
        rentalOrdersDao.insertOrderVcu(vcu);

        // 修改订单状态
        OrderInfoDO upOrder = new OrderInfoDO();
        upOrder.setId(param.getOrderInfoId());
        upOrder.setTranStatus(TranStatusDic.YFH.getStatus());
        upOrder.setSendWareTime(cdate);
        rentalOrdersDao.updateOrderInfo(upOrder);
        this.sendShipMsg(param.getOrderInfoId());
        return ResultBody.success();
    }

    @Override
    public PageResult listOfRentalOrders(OrderStatusVO orderStatusVO, BaseAccountDTO currentAccount) {
        orderStatusVO.setUserAccountId(currentAccount.getUserAccountId());
        int count = rentalOrdersDao.rentalOrdersCount(orderStatusVO);
        if (count == 0) {
            return PageResult.buildPage(orderStatusVO.getPageNo(), orderStatusVO.getPageSize(), count);
        }
        Integer pageNo = orderStatusVO.getPageNo();
        orderStatusVO.buildCurrentPage();

        List<OrderStatusDO> orderInfoDOS = rentalOrdersDao.listOfRentalOrders(orderStatusVO);
        List<OrderStatusDTO> collect = orderInfoDOS.stream().map(d -> {
            return d.buildOrderStatusDTO();
        }).collect(Collectors.toList());
        return PageResult.buildPage(pageNo, orderStatusVO.getPageSize(), count, collect);
    }

    @Override
    public ResultBody renterTakeOrder(OrderVcuVO param) {
        OrderInfoDO order = rentalOrdersDao.getOrderInfo(param.getOrderInfoId());
        if (!TranStatusDic.YFH.getStatus().equals(order.getTranStatus())) {
            return ResultBody.error(ResultEnum.ORDER_STATUS_ERROR);
        }
        // 订单-视频、图片
        OrderVcuDO vcu = param.buildOrdeVcu();
        vcu.setVcuType(VcuType.REC.getCode());
        vcu.setCreateTime(new Date());
        rentalOrdersDao.insertOrderVcu(vcu);

        // 修改订单状态
        OrderInfoDO upOrder = new OrderInfoDO();
        upOrder.setId(param.getOrderInfoId());
        upOrder.setTranStatus(TranStatusDic.ZLZ.getStatus());
        rentalOrdersDao.updateOrderInfo(upOrder);
        return ResultBody.success();
    }


    @Override
    public ResultBody returnOrderWare(ReturnWareVO param) {
        OrderInfoDO order = rentalOrdersDao.getOrderInfo(param.getOrderInfoId());
        if (!TranStatusDic.ZLZ.getStatus().equals(order.getTranStatus())) {
            return ResultBody.error(ResultEnum.ORDER_STATUS_ERROR);
        }

        if (param.getRenMethod() == RenMethod.EXPRESS.getCode()) {
            if (StringUtils.isBlank(param.getRenExCode()) || StringUtils.isBlank(param.getRenExNo())) {
                return ResultBody.error(ResultEnum.EXPRESS_RETURN_ERROR);
            }
            OrderReceiptDO receipt = new OrderReceiptDO();
            receipt.setOrderInfoId(param.getOrderInfoId());
            receipt.setRenMethod(RenMethod.EXPRESS.getCode());
            receipt.setRenExCode(param.getRenExCode());
            receipt.setRenExNo(param.getRenExNo());
            rentalOrdersDao.updateOrderReceipt(receipt);
        } else {
           /* if (param.getRepoInfoId() == null) {
                return ResultBody.error(ResultEnum.SHOP_RETURN_ERROR);
            }*/
            //门店信息
            // RepoInfoDTO repoInfo = repoDeviceServletClient.feignGetRepoShop(param.getRepoInfoId());
            /*if (repoInfo == null) {
                return ResultBody.error(ResultEnum.FEIGN_DEVICE_SERVLET_ERROR);
            }*/
            OrderReceiptDO receipt = new OrderReceiptDO();
            receipt.setOrderInfoId(param.getOrderInfoId());
            receipt.setRenMethod(RenMethod.SHOP.getCode());
            // receipt.setRenRepoName(repoInfo.getRepoName());
            // receipt.setRenRepoAddr(repoInfo.getRepoAddress());
            //UserAccountDTO repoUser = userServletClient.feignGetUserSimpleInfo(repoInfo.getUserAccountId());
           /* if (repoUser == null) {
                return ResultBody.error(ResultEnum.FEIGN_USER_SERVLET_ERROR);
            }
            receipt.setRenRepoPhone(repoUser.getPhoneNum());*/
            rentalOrdersDao.updateOrderReceipt(receipt);
        }

        // 订单-视频、图片
        OrderVcuDO vcu = param.buildOrdeVcu();
        vcu.setVcuType(VcuType.RETURN.getCode());
        vcu.setCreateTime(new Date());
        rentalOrdersDao.insertOrderVcu(vcu);

        // 修改订单状态
        OrderInfoDO upOrder = new OrderInfoDO();
        upOrder.setId(param.getOrderInfoId());
        upOrder.setTranStatus(TranStatusDic.GHZ.getStatus());
        rentalOrdersDao.updateOrderInfo(upOrder);
        return ResultBody.success();
    }


    private void sendShipMsg(Integer orderInfoId) {
        OrderInfoDO orderInfo = rentalOrdersDao.getOrderInfo(orderInfoId);
        //RepoAccountDTO wechat = repoServletClient.feignRepoLoginInfo(orderInfo.getRepoAccountId());
        OrderReceiptDO receipt = rentalOrdersDao.getOrderReceiptByOrder(orderInfoId);
        // if (wechat == null || StringUtils.isBlank(wechat.getOpenId())) {
        //    return;
        // }
        AppletMsgVO aMsg = new AppletMsgVO();
        // aMsg.setTouser(wechat.getOpenId());
        aMsg.setTemplate_id(WxMsgTemplete.AT_REPO_SHIP);
        aMsg.setPage(WxMsgDataConfig.REPO_SHIP_MESSAGE);
        // 订单号
        JSONObject orderNo = new JSONObject();
        orderNo.put("value", orderInfo.getOrderNo());

        // 商品名称
        JSONObject wareName = new JSONObject();
        wareName.put("value", orderInfo.getWareTitle());

        // 快递公司
        JSONObject expressName = new JSONObject();
        expressName.put("value", receipt.getExName());

        // 快递单号
        JSONObject expressNo = new JSONObject();
        expressNo.put("value", receipt.getSendExNo());

        // 收货人
        JSONObject receiptName = new JSONObject();
        receiptName.put("value", receipt.getTakeName());
        JSONObject data = new JSONObject();
        data.put("character_string10", orderNo);
        data.put("thing1", wareName);
        data.put("phrase3", expressName);
        data.put("character_string4", expressNo);
        data.put("thing11", receiptName);
        aMsg.setData(data);
        //消息发送
        // repoServletClient.asynSendUserAppletMsg(aMsg);
    }


    /**
     * 密码认证
     *
     * @param cuser
     * @param authPwd
     * @return
     */
    private ResultBody passwordAuthentication(BaseAccountDTO cuser, String authPwd) {
        ResponseEntity<String> response =
                restTemplate.getForEntity(
                        userAppUrl
                                + "back-user/feignAuthUserPwd?authPwd="
                                + authPwd
                                + "&id="
                                + +cuser.getUserAccountId(),
                        String.class);
        ResultBody body = JSON.parseObject(response.getBody(), ResultBody.class);
        if (!ResultEnum.SUCCESS.getResultCode().equals(body.getCode())) {
            return body;
        }
        return null;
    }


    @Override
    public ResultBody pfConfirmOrderWare(BaseAccountDTO cuser, OrderVcuVO param) {
        // 密码认证
        ResultBody body = passwordAuthentication(cuser, param.getAuthPwd());
        if (body != null) {
            return body;
        }
        OrderInfoDO order = rentalOrdersDao.getOrderInfo(param.getOrderInfoId());
        if (!TranStatusDic.GHZ.getStatus().equals(order.getTranStatus())) {
            return ResultBody.error(ResultEnum.ORDER_STATUS_ERROR);
        }
        // 订单-视频、图片
        OrderVcuDO vcu = param.buildOrdeVcu();
        vcu.setVcuType(VcuType.PFREC.getCode());
        vcu.setCreateTime(new Date());
        rentalOrdersDao.insertOrderVcu(vcu);
        // 修改订单状态
        OrderInfoDO upOrder = new OrderInfoDO();
        upOrder.setId(param.getOrderInfoId());
        upOrder.setTranStatus(TranStatusDic.JYWC.getStatus());
        rentalOrdersDao.updateOrderInfo(upOrder);
        // 以异步方式同步数据到分成明细表 云仓
        // this.SynchronousDataToPushMoney(order.getOrderNo());
        return ResultBody.success();
    }
}
